//******************************************************************************
//* File       : SrcProf_Dialog.cpp                                            *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 1998-2015 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in SrcProf.hpp           *
//* Date       : 29-Oct-2015                                                   *
//* Version    : (see AppVersion string in SrcProf.hpp)                        *
//*                                                                            *
//* Description: This module contains methods for creation, manipulation and   *
//* destruction of NcDialog-class objects. These objects constitute the        *
//* application's 'Interactive Mode'.                                          *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "SrcProf.hpp"

#define DEBUG_CODE (0)
#if DEBUG_CODE != 0
const short debugROWS = 37,   // size of debugging window
            debugCOLS = 52 ;
#endif   // DEBUG_CODE

//****************
//* Local Data   *
//****************

//* Colors for dialog windows and file-selection display data *
static attr_t  dColor,     // dialog background color
               pbnColor,   // pushbutton non-focus color
               pbfColor,   // pushbutton focus color
               rbnColor,   // radiobutton non-focus color
               rbfColor,   // radiobutton focus color
               dirAttr,    // directory name
               srcAttr,    // source code filename (un-selected)
               selAttr ;   // source code filename (selected)

static const char* const parentDir = "..  [parent directory]" ;
static const char fileNone[minTERMCOLS - 1] = // displayed if no files available
{ "************************  NO SOURCE FILES SPECIFIED  *************************" } ;

//*************************
//*   InteractiveMode     *
//*************************
//******************************************************************************
//* Perform the start-up sequence for the NCurses Engine, then open the main   *
//* application dialog.                                                        *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::InteractiveMode ( void )
{
   //** Initialize NCurses Engine **
   if( (nc.StartNCursesEngine ()) == OK )
   {
      //* Verify that the current locale supports UTF-8 character encoding. *
      short localeSetOk = nc.VerifyLocale () ;

      nc.SetCursorState ( nccvINVISIBLE ) ;  // hide the cursor
      nc.SetKeyProcState ( nckpRAW ) ;       // allow CTRL keys through unprocessed
      nc.ClearScreen () ;                    // clear the terminal window 

      //* Get terminal-window size - we need room to play *
      nc.ScreenDimensions ( this->dRows, this->dCols ) ;

      //* Set data color attributes *
      this->pd.goodAttr = nc.gr ;            // statistical data colors
      this->pd.fairAttr = nc.ma ;
      this->pd.poorAttr = nc.reB ;
      this->pd.dfltAttr = nc.bw ;
      dColor   = nc.gybl | ncbATTR ;         // dialog background color
      pbnColor = nc.gyR ;                    // pushbutton control colors
      pbfColor = nc.gyre | ncbATTR ;
      rbnColor = dColor ;                    // radiobutton control colors
      rbfColor = nc.br ;
      dirAttr  = nc.cy ;                     // file-selection colors
      srcAttr  = nc.br ;
      selAttr  = srcAttr | ncuATTR ;

      //* Initialize log-filename (if not previously initialized *
      if ( this->logName.gschars() == 1 )
         this->logName = spSaveTextStats ;

      //* If no start-up errors, open the main application window *
      if ( ((this->dRows >= minTERMLINES) && (this->dCols >= minTERMCOLS)) && 
           (localeSetOk == OK) )
      {
         this->dCols = minTERMCOLS ;
         this->imOpenDialog () ;
      }

      //** Shut down ncurses engine **
      nc.RestoreCursorState () ;             // make cursor visible
      nc.RefreshScreen () ;                  // refresh the main window
      nc.StopNCursesEngine () ;              // Deactivate the NCurses engine

      //* Report any errors that occurred during startup *
      if ( this->dRows < minTERMLINES || this->dCols < minTERMCOLS || localeSetOk != OK )
      {
         gString gsOut ;
         if ( (this->dRows < minTERMLINES) || (this->dCols < minTERMCOLS) )
         {
            gsOut.compose( badFileTemplate[bftTermsize], 
                           &minTERMLINES, &minTERMCOLS ) ;
            gsOut.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
         }
         if ( localeSetOk != OK )
         {
            gsOut.compose( badFileTemplate[bftLocaleA], nc.GetLocale() ) ;
            gsOut.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
            gsOut = badFileTemplate[bftLocaleB] ;
            gsOut.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
            gsOut = badFileTemplate[bftLocaleC] ;
            gsOut.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
         }
         wcout << endl ;
      }
   }
   else
   {
      gString gsOut( badFileTemplate[bftEngineA] ) ;
      gsOut.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
      gsOut = badFileTemplate[bftEngineB] ;
      gsOut.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
   }
}  //* End InteractiveMode() *

//*************************
//*     imOpenDialog      *
//*************************
//******************************************************************************
//* Open the main application dialog window.                                   *
//* 1) If user has specified one or more valid source files, then display the  *
//*    analysis for those files.                                               *
//* 2) Else, if no valiid source files specified, display the source files in  *
//*    the current directory and allow user to choose among them.              *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::imOpenDialog ( void )
{
const short ulY = ZERO,             // upper left corner in Y
            ulX = ZERO ;            // upper left corner in X
const attr_t tColor = nc.gybr | ncbATTR ;// dialog title color
enum ctrlIndices : short
{ donePB = ZERO, savePB, sortPB, helpPB, selPB, fileSE, controlsDEFINED } ;

   
InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'DONE' pushbutton - - - - - - - - - - - - - - - - - - - -     donePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(this->dRows - 2),       // ulY:       upper left corner in Y
      short(this->dCols / 2 - 16),  // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "  DONE  ",                   // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[savePB],                  // nextCtrl:  link in next structure
   },
   {  //* 'SAVE' pushbutton - - - - - - - - - - - - - - - - - - - -     savePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[donePB].ulY,               // ulY:       upper left corner in Y
      short(ic[donePB].ulX + ic[donePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "  SAVE  ",                   // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[sortPB],                  // nextCtrl:  link in next structure
   },
   {  //* 'SORT' pushbutton - - - - - - - - - - - - - - - - - - - -     sortPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[donePB].ulY,               // ulY:       upper left corner in Y
      short(ic[savePB].ulX + ic[savePB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "  SORT  ",                   // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[helpPB],                  // nextCtrl:  link in next structure
   },
   {  //* 'HELP' pushbutton - - - - - - - - - - - - - - - - - - - -     helpPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[donePB].ulY,               // ulY:       upper left corner in Y
      short(ic[sortPB].ulX + ic[sortPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "  HELP  ",                   // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[selPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'SELECT' pushbutton - - - - - - - - - - - - - - - - - - -      selPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[donePB].ulY,               // ulY:       upper left corner in Y
      short(ic[helpPB].ulX + ic[helpPB].cols + 2), // ulX: upper left corner in X
      1,                            // lines:     (n/a)
      14,                           // cols:      control columns
      " SELECT FILES ",             // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[fileSE]                   // nextCtrl:  link in next structure
   },
   {  //* 'FILE' Scroll Ext control - - - - - - - - - - - - - - - -     fileSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      short(this->dRows - 7),       // lines:     control lines
      this->dCols,                  // cols:      control columns
      NULL,                         // dispText:  n/a
      dColor,                       // nColor:    non-focus border color
      nc.cyG,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  number of elements in text/color arrays
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;


   //* Initial parameters for dialog window *
   InitNcDialog dInit( this->dRows,    // number of display lines
                       this->dCols,    // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Give dialog a title (with different color than border color) *
      gString gsOut ;
      gsOut.compose( L"  %S%S%S  ", AppTitle1, AppVersion, AppTitle2 ) ;
      dp->SetDialogTitle ( gsOut.gstr(), tColor ) ;

      //* Connect Scroll Ext control to borders *
      cdConnect cdConn ;
      cdConn.connection = true ;
      cdConn.ll2Left = cdConn.ul2Left = cdConn.ur2Right = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( fileSE, cdConn ) ;

      //* Column headings *
      dp->WriteParagraph ( (ic[fileSE].ulY -2), (ic[fileSE].ulX + 1), 
                           statHead, dColor ) ;

      //* Color code legend *
      winPos wp( (this->dRows - 4), 2 ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, L"Legend:  ", dColor ) ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, L" Good ", this->pd.goodAttr ) ;
      wp.xpos += 2 ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, L" Fair ", this->pd.fairAttr ) ;
      wp.xpos += 2 ;
      wp = dp->WriteString ( wp.ypos, wp.xpos, L" Poor ", this->pd.poorAttr ) ;

      short  icIndex = ZERO ;       // index of control with input focus
      ssetData ssd ;                // display-data specification

      //* If caller has sent us statistical data, format and display it.*
      if ( this->pd.fileCount > ZERO )
      {
         this->imFormatStatData ( ssd ) ;
         dp->SetScrollextText ( fileSE, ssd ) ;
      }
      //* Else, interactively collect names of files to be analyzed, *
      //* do our own analysis and format the results for display.    *
      else
      {
         dp->RefreshWin () ;
         while ( (icIndex = dp->PrevControl ()) != fileSE ) ;
         this->imSelect ( dp, ssd, fileSE ) ;   // user selection
         icIndex = dp->NextControl () ;      // move to next control
      }

      dp->RefreshWin () ;           // make everything visible

      uiInfo Info ;                 // user interface data returned here
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = dp->EditPushbutton ( Info ) ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == donePB )
               {  //* Prepare to return *
                  done = true ;
               }
               else if ( Info.ctrlIndex == helpPB )
               {  //* Give user a clue *
                  dp->SetDialogObscured () ; // save parent dialog's display
                  HelpDialog ( 1, 1, 
                               (dp->Get_NcDialog_Version ()),
                               (dp->Get_nclibrary_Version ()) ) ;
                  dp->RefreshWin () ;        // restore parent dialog's display
               }
               else if ( Info.ctrlIndex == savePB )
               {
                  if ( ssd.dispItems > 1 )
                  {
                     dp->SetDialogObscured () ;
                     winPos ctr( this->dRows / 2 - 1, this->dCols / 2 ) ;
                     this->imSave2File ( ctr ) ;
                     dp->RefreshWin () ;
                  }
               }
               else if ( Info.ctrlIndex == sortPB )
               {
                  dp->SetDialogObscured () ;
                  winPos ctr( this->dRows / 2 - 1, this->dCols / 2 ) ;
                  if ( (this->imSortOption ( ssd, ctr )) != false )
                     dp->SetScrollextText ( fileSE, ssd ) ;
                  dp->RefreshWin () ;     // show the new data (not actually needed)
               }
               else if ( Info.ctrlIndex == selPB )
               {  //* User wants to select a different group of files *
                  //* Set input focus on selection control, then ask  *
                  //* user to select files from a list.               *
                  while ( (icIndex = dp->PrevControl ()) != fileSE ) ;
                  this->imSelect ( dp, ssd, fileSE ) ;
               }
            }
         }
         //* If focus is currently on a Scrollext control *
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {  //* If analytical data available, scroll through it, *
            //* else, move on.                                   *
            if ( ssd.dispItems > 1 )
               icIndex = dp->ViewScrollext ( Info ) ;
         }
         if ( ! done )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while(!done)

      //* Release any dynamically-allocated memory *
      this->imRelease ( ssd ) ;
   }        // if((dp->OpenWindow())==OK)

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;
   
}  //* End imOpenDialog() *

//*************************
//*      imSelect         *
//*************************
//******************************************************************************
//* Allow user to interactively cruise the directory tree in search of source  *
//* code files to be analyzed. Analyze the data for selected files and return  *
//* with the results formatted for display.                                    *
//*                                                                            *
//* Input  : dp  : pointer to caller's dialog window                           *
//*          ssd : (by reference) for writing display data to dctSCROLLEXT     *
//*                control                                                     *
//*          cIndex: index of the dctSCROLLEXT control in which to display data*
//*                                                                            *
//* Returns: number of files selected for analysis                             *
//*          ssd contains the results of the selection and analysis sequence   *
//******************************************************************************
//* Programmer's Note: We play a couple of tricks here to initialize data that *
//* will eventually be referenced through ssetData-object const pointers.      *
//*                                                                            *
//* For display of user instructions and file counters, certain assumptions    *
//* about the layout of the dialog are made. If the layout of the dialog       *
//* changes, be sure to update these assumptions.                              *
//*                                                                            *
//******************************************************************************

int SrcProf::imSelect ( NcDialog* dp, ssetData& ssd, short cIndex )
{
   int   selectCount = ZERO,        // files selected in current view
         selectTotal = ZERO ;       // return value: total number of files selected

   //* Delete the old source-file list (if any) *
   this->spfDeleteTempFiles ( false ) ;

   //* Create, format and display a list of *
   //* interesting files in CWD.            *
   this->imSelectRead ( dp, ssd, cIndex ) ;

   //* Display selection instructions ( see note above ) *
   const char* Instructions =
   "Up/Down arrow keys scroll through list. Space key selects/deselects files.    \n"
   "Ctrl+A to select all. Ctrl+C clears all selections. ENTER changes directory.  " ;
   const attr_t dColor = nc.blG,    // assumed dialog background color
                scColor = nc.bl ;   // select-count display color
   attr_t* colorPtr = (attr_t*)ssd.dispColor ; // (see note above)
   gString gsCount ;                // for formatting display of selection counter 
   short sccY = this->dRows - 4,   // select-count display position
         sccX = this->dCols / 2 ;

   dp->ClearLine ( 1, false ) ;     // display selection instructions
   dp->ClearLine ( 2, false ) ;
   winPos wpi( 1, 1 ) ;             // instruction message position
   dp->WriteParagraph ( wpi.ypos, wpi.xpos, Instructions, dColor ) ;

   //*********************************************
   //* Allow user to select files to be analyzed *
   //*********************************************
   uiInfo Info ;                 // user interface data returned here
   bool   done = false ;         // loop control
   while ( ! done )
   {
      //* Update selection counter *
      winPos wpc = dp->WriteString ( sccY, sccX, "Files Selected: ", dColor ) ;
      gsCount.compose( L" % 4d ", &selectCount ) ;
      wpc = dp->WriteString ( wpc.ypos, wpc.xpos, gsCount, scColor ) ;
      wpc = dp->WriteString ( wpc.ypos, wpc.xpos, " Total: ", dColor ) ;
      int tcnt = selectTotal + selectCount ;
      gsCount.compose( L" % 4d ", &tcnt ) ;
      dp->WriteString ( wpc.ypos, wpc.xpos, gsCount, scColor, true ) ;

      //* Get user input *
      dp->EditScrollext ( Info ) ;

      //* If user has selected a directory name, go there  *
      //* We also accept the standard Linux keycodes used  *
      //* to go to parent directory: Backspace and Alt+Up. *
      if ( ((this->tnfPtr[Info.selMember].fType == fmDIR_TYPE) &&
            (Info.keyIn == nckENTER)) || 
            ((Info.keyIn == nckBKSP) || (Info.keyIn == nckAUP)) )
      {
         //* Remember where we're going because        *
         //* birds are about to eat our bread crumbs.  *
         //* (What? You don't know Hansel and Gretel?) *
         bool childDir = true ;  // assume child directory
         gString gstmp( this->tnfPtr[Info.selMember].fName ) ;
         //* If user has selected parent directory ('..') *
         if ( ((this->tnfPtr[Info.selMember].fName[0] == PERIOD) &&
              (this->tnfPtr[Info.selMember].fName[1] == PERIOD)) || 
              (Info.keyIn == nckBKSP) || (Info.keyIn == nckAUP) )
            childDir = false ;   // parent directory

         //* If source code file selections have been made in the *
         //* current directory, append them to our list file.     *
         short status = OK ;
         if ( selectCount > ZERO )
         {
            status = this->imSelectSave ( ssd, selectCount ) ;
            // Programmer's Note: ssd data is now invalid
            selectTotal += selectCount ;  // update the selection total
            selectCount = ZERO ;          // reset the selection counter
         }

         if ( status == OK )
         {
            //* Go to child directory *
            if ( childDir )
               status = this->spfCdChild ( gstmp.ustr() ) ;
            //* Go to parent directory *
            else
               status = this->spfCdParent () ;
            if ( status == OK )
            {
               //* Read contents of new CWD, reinitialize the color-attribute  *
               //* access pointer, and continue selection process.             *
               this->imSelectRead ( dp, ssd, cIndex ) ;
               colorPtr = (attr_t*)ssd.dispColor ; // (see note above)
            }
            else
            {  //* Unable to enter specified directory (this is unlikely) *
               const char* msg[] = 
               {
                  "  WARNING - WARNING  ",
                  " ",
                  "File system error! Unable to access specified directory.",
                  NULL
               } ;
               genDialog gd( msg, dColor, 6, 60 ) ;
               dp->InfoDialog ( gd ) ;
            }
         }
      }
      //* User selecting/deselecting currently-highlighted file *
      else if ( (tnfPtr[Info.selMember].fType == fmREG_TYPE) &&
                (Info.keyIn == nckSPACE || Info.keyIn == nckENTER) )
      {
         //* Select or de-select the current display item *
         if ( colorPtr[Info.selMember] == selAttr )
         {
            colorPtr[Info.selMember] = srcAttr ;
            --selectCount ;
         }
         else
         {
            colorPtr[Info.selMember] = selAttr ;
            ++selectCount ;
         }
         dp->RefreshScrollextText ( cIndex ) ;
         //* and advance the highlight *
         if ( Info.selMember < (ssd.dispItems - 1) )
            dp->MoveScrollextHighlight ( cIndex, nckDOWN ) ;
      }
      //* User selecting/deselecting All regular files in the current directory*
      else if ( Info.keyIn == nckC_A )
      {  //* If no source code files currently selected, select all of them *
         if ( selectCount == ZERO )
         {
            for ( int i = ZERO ; i < ssd.dispItems ; i++ )
            {
               if ( tnfPtr[i].fType == fmREG_TYPE )
               {
                  colorPtr[i] |= ncuATTR ;
                  ++selectCount ;
               }
            }
         }
         //* De-select all files *
         else
         {
            for ( int i = ZERO ; i < ssd.dispItems ; i++ )
            {
               if ( tnfPtr[i].fType == fmREG_TYPE )
                  colorPtr[i] &= ~ncuATTR ;
            }
            selectCount = ZERO ;
         }
         dp->RefreshScrollextText ( cIndex ) ;
      }
      //* Clear all selections in the current session *
      else if ( Info.keyIn == nckC_C )
      {
         //* Delete the source-file list (if any). *
         //* Reset the 'selection' accumulators.   *
         //* Re-read the current working directory.*
         this->spfDeleteTempFiles ( false ) ;
         selectTotal = selectCount = ZERO ;
         this->imSelectRead ( dp, ssd, cIndex ) ;
      }
      //* Panic button - return with no selections *
      else if ( Info.keyIn == nckESC )
      {
         selectTotal = ZERO ;
         done = true ;
      }
      //* If selections complete *
      else if ( Info.keyIn == nckTAB || Info.keyIn == nckSTAB )
      {
         //* If source code file selections have been made in the *
         //* current directory, append them to our list file.     *
         if ( selectCount > ZERO )
         {
            if ( (this->imSelectSave ( ssd, selectCount )) == OK )
            {  // Programmer's Note: ssd data is now invalid
               selectTotal += selectCount ;  // update the selection total
               selectCount = ZERO ;          // reset selection counter (unnecessary)
            }
         }
         done = true ;
      }
   }     // while()

   //**************************************************************
   //* Selection complete. If user has selected at least one file,*
   //* perform the analysis and format the results for display.   *
   //**************************************************************
   if ( selectTotal > ZERO )
   {
      //* Analyze the new data and format it for display *
      if ( (this->imSelectRefresh ( ssd )) != OK )
         selectTotal = selectCount = ZERO ; // internal error (unlikely)
   }

   //* No files selected OR error creating temp file OR   *
   //* internal processing errors. Create default message.*
   if ( selectTotal == ZERO )
   {
      this->imAllocate ( ssd, 1 ) ;
      ssd.hlIndex = ZERO ;
      ssd.hlShow  = false ;
      ssd.dispText[ZERO] = fileNone ;
      colorPtr = (attr_t*)ssd.dispColor ;
      colorPtr[ZERO] = this->pd.dfltAttr ;
   }

   //* Release the temporary data *
   this->tnfRelease () ;

   //* Restore caller's column headings and erase selection counter (see note *
   //* in method header), then write results fo analysis to the display.      *
   dp->ClearLine ( 1, false ) ;
   dp->ClearLine ( 2, false ) ;
   dp->WriteParagraph ( wpi.ypos, wpi.xpos, statHead, dColor ) ;
   dp->ClearLine ( sccY, false, sccX ) ;
   dp->SetScrollextText ( cIndex, ssd ) ;

   #if DEBUG_CODE != 0
   this->imDebug ( ssd ) ;
   #endif   // DEBUG_CODE

   return selectTotal ;

}  //* End imSelect() *

//*************************
//*     imSelectRead      *
//*************************
//******************************************************************************
//* Create, format and display a list of interesting files in CWD.             *
//*                                                                            *
//*                                                                            *
//* Input  : dp  : pointer to caller's dialog window                           *
//*          ssd : (by reference)                                              *
//*          cIndex: index of the dctSCROLLEXT control in which to display data*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void SrcProf::imSelectRead ( NcDialog* dp, ssetData& ssd, short cIndex )
{

   this->tnfRelease () ;            // release old allocation, if any
   this->imReadCWD ( ssd ) ;        // read contents of current working directory
   dp->SetScrollextText ( cIndex, ssd ) ; // display contents of directory

}  //* End imSelectRead() *

//*************************
//*     imSelectSave      *
//*************************
//******************************************************************************
//* Save 'selected' source code filenames to a temporary file.                 *
//*                                                                            *
//* Input  : ssd        : (by reference) contains display data with            *
//*                       'selected' items marked                              *
//*          selectCount: number of files in this->tnfPtr array which have     *
//*                       been marked as selected                              *
//*                                                                            *
//* Returns: OK   if list successfully created                                 *
//*          ERR  if error creating list or if no files in list                *
//******************************************************************************

short SrcProf::imSelectSave ( ssetData& ssd, UINT selectCount )
{
   short status = ERR ;

   if ( selectCount > ZERO )
   {
      //* Move all 'selected' files to top of tnFName array *
      this->imSortDirEntriesBySelected ( ssd ) ;

      //* Append the list of selected files to the list file *
      gString dPath ;   // path of directory in which source code files were found
      this->spfGetCWD ( dPath ) ;
      status = this->CreateFileList ( dPath, selectCount, true ) ;
      this->tnfRelease () ;         // release the directory-entries
   }
   return status ;

}  //* End imSelectSave() *

//*************************
//*   imSelectRefresh     *
//*************************
//******************************************************************************
//* Analyze the source code files in the list file.                            *
//* a) Release any old analysis/display data.                                  *
//* b) Capture the path/filename data from the list file.                      *
//* c) Analyze the new data.                                                   *
//* d) Format the data for display.                                            *
//*                                                                            *
//* Input  : ssd  : (by reference) may contain stale display data which will   *
//*                 be released                                                *
//*                                                                            *
//* Returns: OK  if successful,                                                *
//*              ssd contains new analysis/displa data                         *
//*          ERR if internal processing errors                                 *
//*              ssd contain no data                                           *
//******************************************************************************

short SrcProf::imSelectRefresh ( ssetData& ssd )
{
   short status = OK ;

   this->pdRelease () ;                   // release old statistical data
   this->imRelease ( ssd ) ;              // release old display data

   //* Validate the source code filenames in the list file, and capture *
   //* path/filename data from all source files in this->pd.fileData[]. *
   if ( (this->ValidateFileList ()) > ZERO )
   {
      //* Analyze the source-code-file data. Then format results  *
      //* for display and copy it into caller's ssetData object.  *
      if ( this->pd.fileCount > ZERO )
      {
         this->AnalyzeFileList () ;
         this->imFormatStatData ( ssd ) ;
      }
      else
         status = ERR ;    // internal errors
   }
   else
      status = ERR ;    // internal errors

   return status ;

}  //* End imSelectRefresh() *

//*************************
//*   imFormatStatData    *
//*************************
//******************************************************************************
//* Format the statistical data for display in the dialog window.              *
//* Format the data for the dialog control (UTF-8 pointers), 1 line for each   *
//* file + 2 summary lines. Initialize the color attribute for each line.      *
//*              We assume that this->pd.fileCount > ZERO.                     *
//*                                                                            *
//* IMPORTANT NOTE: Dynamic memory allocation occurs in this method.           *
//*                 It is caller's responsibility to release it.               *
//*                                                                            *
//* Input  : ssd: (by reference, initial values ignored)                       *
//*               receives pointers to the formatted display data              *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: We play a couple of tricks here to initialize data that *
//* will eventually be referenced through ssetData-object const pointers.      *
//******************************************************************************

void SrcProf::imFormatStatData ( ssetData& ssd )
{
   this->imAllocate ( ssd, (this->pd.fileCount + 2) ) ; // allocate data space
   ssd.hlIndex   = ZERO ;     // set highlight on first item
   ssd.hlShow    = false ;    // set highlight as invisible

   attr_t* colorPtr = (attr_t*)ssd.dispColor ;  // (see note above)
   UINT convIndex ;
   for ( convIndex = ZERO ; convIndex < this->pd.fileCount ; convIndex++ )
   {  //* Copy the text pointer *
      ssd.dispText[convIndex] = this->pd.dispData[convIndex].ustr() ;

      //* Initialize the color attribute for the text *
      if ( (this->pd.fileData[convIndex].sfType == sffC) || 
           (this->pd.fileData[convIndex].sfType == sffVB) )
      {  //* High-level-language group *
         if ( this->pd.fileData[convIndex].maintIndex >= this->pd.goodpctC )
            colorPtr[convIndex] = this->pd.goodAttr ;
         else if ( this->pd.fileData[convIndex].maintIndex >= this->pd.fairpctC )
            colorPtr[convIndex] = this->pd.fairAttr ;
         else
            colorPtr[convIndex] = this->pd.poorAttr ;
      }
      else if ( this->pd.fileData[convIndex].sfType == sffA )
      {  //* Assembly-language group *
         if ( this->pd.fileData[convIndex].maintIndex >= this->pd.goodpctA )
            colorPtr[convIndex] = this->pd.goodAttr ;
         else if ( pd.fileData[convIndex].maintIndex >= this->pd.fairpctA )
            colorPtr[convIndex] = this->pd.fairAttr ;
         else
            colorPtr[convIndex] = this->pd.poorAttr ;
      }
      else if ( (this->pd.fileData[convIndex].sfType == sffPY) || 
                (this->pd.fileData[convIndex].sfType == sffPE) || 
                (this->pd.fileData[convIndex].sfType == sffPH) || 
                (this->pd.fileData[convIndex].sfType == sffRU) || 
                (this->pd.fileData[convIndex].sfType == sffSQ) ||
                (this->pd.fileData[convIndex].sfType == sffTX) ||
                (this->pd.fileData[convIndex].sfType == sffHT) )
      {  //* Scripting-language group *
         if ( this->pd.fileData[convIndex].maintIndex >= this->pd.goodpctS )
            colorPtr[convIndex] = this->pd.goodAttr ;
         else if ( pd.fileData[convIndex].maintIndex >= this->pd.fairpctS )
            colorPtr[convIndex] = this->pd.fairAttr ;
         else
            colorPtr[convIndex] = this->pd.poorAttr ;
      }
      else     // if we get here, it is an application error
         colorPtr[convIndex] = this->pd.dfltAttr ;
   }
   ssd.dispText[convIndex] = statSummary1 ;
   colorPtr[convIndex++] = this->pd.dfltAttr ;
   ssd.dispText[convIndex] = this->pd.summaryData.ustr() ;
   colorPtr[convIndex] = this->pd.dfltAttr ;

}  //* End imFormatStatData() *

//******************************
//* imSortDirEntriesBySelected *
//******************************
//******************************************************************************
//* Sort this->tnfPtr directory entries according to whether user has selected *
//* the corresponding files for analysis.                                      *
//* An entry is 'selected' if the color attribute for its corresponding        *
//* display line includes the ncuATTR (underline bit).                         *
//*                                                                            *
//* Input  : ssd    : (by reference) dispColor array will be sorted            *
//*                   corresponding to the tnfPtr array sort                   *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: This is a brain-dead bubble sort. Both the 'tnfPtr'     *
//* array and the 'ssd.dispColor' array will be sorted together; however, this *
//* will invalidate the ssd display data itself, so caller must not redisplay  *
//* the ssd data.                                                              *
//******************************************************************************

void SrcProf::imSortDirEntriesBySelected ( ssetData& ssd )
{
   if ( ssd.dispItems > 1 )         // if there is something to do
   {
      tnFName tmp ;                 // swap buffer
      attr_t  tmpcolor ;            // swap color attribute
      attr_t* cPtr = (attr_t*)ssd.dispColor ; // for modifying 'const' data
      int indexA = ZERO,            // upper index
          indexB = 1,               // lower index
          lastIndex = ssd.dispItems - 1, // index of last item in list
          moves ;                   // sort operations performed each loop

      do
      {
         moves = ZERO ;
         while ( (indexA < lastIndex) && (cPtr[indexA] & ncuATTR) )
            ++indexA ;
         if ( indexA < lastIndex )
         {
            for ( indexB = indexA + 1 ; indexB <= lastIndex ; indexB++ )
            {
               if ( (cPtr[indexB] & ncuATTR) && 
                   !(cPtr[indexA] & ncuATTR) )
               {
                  tmp = this->tnfPtr[indexA] ;
                  this->tnfPtr[indexA] = this->tnfPtr[indexB] ;
                  this->tnfPtr[indexB] = tmp ;
                  tmpcolor = cPtr[indexA] ;
                  cPtr[indexA] = cPtr[indexB] ;
                  cPtr[indexB] = tmpcolor ;
                  ++indexA ;
                  ++moves ;
               }
            }
         }
      }
      while ( (moves > ZERO) && (indexA < lastIndex) ) ;
   }

}  //* End imSortDirEntriesBySelected() *

//*************************
//*      imReadCWD        *
//*************************
//******************************************************************************
//* Create a list of source file names in CWD for display in dialog window,    *
//* Include '..' (parent) directory and names of any subdirectories.           *
//*                                                                            *
//* IMPORTANT NOTE: Dynamic memory allocation occurs in this method.           *
//*                 It is caller's responsibility to release it.               *
//*                 - ssd.dispText                                             *
//*                 - ssd.dispColor                                            *
//*                 - this->tnfPtr and this->tnfCount                          *
//*                                                                            *
//* Input  : ssd: (by reference, initial values ignored)                       *
//*               receives pointers to the display data                        *
//*                                                                            *
//* Returns: number of display items                                           *
//******************************************************************************
//* Programmer's Note: We play a couple of tricks here to initialize data that *
//* will eventually be referenced through ssetData-object const pointers.      *
//*                                                                            *
//* Note on postulating a 'parent directory': If user invokes the application  *
//* from the root directory, we can safely assume that there are sub-directory *
//* names to report and dirEntries will be > ZERO. If user invokes the         *
//* application anywhere but root, then we can safely assume that we have at   *
//* least a parent directory to which user can navigate.                       *
//******************************************************************************

UINT SrcProf::imReadCWD ( ssetData& ssd )
{
   //* Scan the current working directory for sub-directory names *
   //* and 'regular' files which are recognized source code files.*
   //* Data are referenced by this->tnfPtr and this->tnfCount.    *
   gString dPath ;
   this->spfGetCWD ( dPath ) ;   // get current-working directory path
   UINT itemCount = this->spfReadDirectory ( dPath, true ) ;
   // NOTE: dispItems should always be >= 1, but if by some chance it isn't,
   //       we would die a horrible death here. (see note above)

   //* Sort by filename (parent ('..') and sub-directories at top of list).*
   //* (tnfPtr references a dynamic memory allocation.)                    *
   this->spfSortDirEntriesByName ( this->tnfPtr, this->tnfCount ) ;

   //* Allocate space for display-string pointers and color attributes *
   this->imAllocate ( ssd, itemCount ) ;
   ssd.hlIndex   = ZERO ;           // set highlight on first item
   ssd.hlShow    = true ;           // set highlight as visible
   char**  tmpt = (char**)ssd.dispText ;  // (see note above)
   attr_t* tmpc = (attr_t*)ssd.dispColor ;// (see note above)
   //* Initialize the display data *
   int i = ZERO ;
   if ( (strncmp ( tnfPtr[i].fName, "..", MAX_FNAME )) == ZERO )
   {
      // NOTE: This is tricky: substitute a meaningful display string
      //       for display of parent directory name
      tmpt[i] = (char*)parentDir ;
      tmpc[i] = dirAttr ;
      ++i ;
   }
   for ( ; i < ssd.dispItems ; i++ )
   {
      tmpt[i] = tnfPtr[i].fName ;
      tmpc[i] = tnfPtr[i].fType == fmREG_TYPE ? srcAttr : dirAttr ;
   }
   return this->tnfCount ;

}  //* End imReadCWD() *

//*************************
//*      imAllocate       *
//*************************
//******************************************************************************
//* Allocate dynamic memory blocks for both an array of display-string pointers*
//* and an array of color attributes.                                          *
//*                                                                            *
//* If memory previously allocated to this object, release it.                 *
//*                                                                            *
//* Input  : ssd      : (by reference) ssetData-class data-control object      *
//*          itemCount: number of items to allocate for each array             *
//*                                                                            *
//* Returns: OK  if allocation is successful                                   *
//*              (any existing allocation will be released)                    *
//*          ERR if ssd.dispItems input parameter <= ZERO                      *
//*              (existing allocation will be undisturbed)                     *
//******************************************************************************

short SrcProf::imAllocate ( ssetData& ssd, UINT itemCount )
{
   short status = ERR ;

   if ( itemCount > ZERO )
   {
      //* If prevous allocation *
      this->imRelease ( ssd ) ;
   
      //* Allocate new data blocks *
      ssd.dispItems = itemCount ;
      ssd.dispText  = new const char*[itemCount] ; // pointers to display strings
      ssd.dispColor = new attr_t[itemCount] ;      // color attributes for display strings
      status = OK ;
   }
   return status ;

}  //* End imAllocate() *

//******************************
//*         imRelease          *
//******************************
//******************************************************************************
//* Release dynamic memory allocation attached to ssetData class object.       *
//* All data members set to default values.                                    *
//*                                                                            *
//* Input  : ssd : ssetData-class object (by reference)                        *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

void  SrcProf::imRelease ( ssetData& ssd )
{
   if ( ssd.dispText != NULL )
   {
      delete [] ssd.dispText ;
      ssd.dispText = NULL ;
   }
   if ( ssd.dispColor != NULL )
   {
      delete [] ssd.dispColor ;
      ssd.dispColor = NULL ;
   }
   ssd.dispItems = ZERO ;
   ssd.hlIndex   = ZERO ;
   ssd.hlShow    = true ;

}  //* End imRelease() *

//*************************
//*     imSave2File       *
//*************************
//******************************************************************************
//* Save currently-displayed statistics to a file.                             *
//* Prompt user for a target filename or path/filename                         *
//*                                                                            *
//* Input  : ctr : center of parent dialog window for positioning child dialog *
//*                                                                            *
//* Returns: osTEXT if text-format data written successfully                   *
//*          osHTML if HTML-format data written successfully                   *
//*          osNONE if operation cancelled or error writing to file            *
//******************************************************************************

OutSave SrcProf::imSave2File ( const winPos& ctr )
{
const char* Label[] = 
{
   "Enter ^name of output file:",            // text box label
   "Save as ^text",                          // 'text' radio button label
   "Save as ^HTML",                          // 'HTML' radio button label
   "^Append data to existing file (if any)", // 'append' radio button label
   "^Overwrite existing file (if any)",      // 'overwrite' radio button label
} ;
const char* Message[] = 
{
   "       Data written successfully!       ", 
   "            Data not saved.             ", 
   "  Unable to access target, not saved!   ", 
   " Invalid filename specified, try again. ", 
} ;
const short dialogROWS = 11,                 // display lines
            dialogCOLS = 44,                 // display columns
            controlsDEFINED = 7,             // # of controls in dialog
            ulY = ctr.ypos - dialogROWS / 2, // upper left corner in Y
            ulX = ctr.xpos - dialogCOLS / 2, // upper left corner in X
            warnLine = dialogROWS - 3,       // position for warning messages
            msgDelay = 2 ;                   // seconds to pause after message
const attr_t 
            tbFocus = nc.br,                 // focus color for text box
            tbNonFocus = nc.bw,              // non-focus color for text box
            goodColor = nc.grG,              // color for good-news messages
            warnColor = nc.reG ;             // color for warning messages
enum ctrlIndices : short { savPB = ZERO, canPB, fnTB, txtRB, htmRB, appRB, ovrRB } ;


InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'SAVE' pushbutton   - - - - - - - - - - - - - - - - - - - -    savPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 9),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "  ^SAVE  ",                  // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - -    canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[savPB].ulY,                // ulY:       upper left corner in Y
      short(dialogCOLS / 2 + 1),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " ^CANCEL ",                  // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[fnTB],                   // nextCtrl:  link in next structure
   },
   {  //* 'Filename' textbox   - - - - - - - - - - - - - - - - - - - - -  fnTB *
      dctTEXTBOX,                   // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      3,                            // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      short(dialogCOLS - 4),        // cols:      control columns
      this->logName.ustr(),         // dispText:  initial display text
      tbNonFocus,                   // nColor:    non-focus color
      tbFocus,                      // fColor:    focus color
      tbFileName,                   // filter:    valid filename characters only
      Label[0],                     // label:     
      -1,                           // labY:      
      ZERO,                         // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[txtRB]                    // nextCtrl:  link in next structure
   },
   {  //* 'Text' radiobutton    - - - - - - - - - - - - - - - - - - -    txtRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: angle bracket, 3 chars wide
      true,                         // rbSelect:  initially selected
      short(ic[fnTB].ulY + 1),      // ulY:       upper left corner in Y
      5,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Label[1],                     // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[htmRB],                   // nextCtrl:  link in next structure
   },
   {  //* 'HTML' radiobutton    - - - - - - - - - - - - - - - - - - -    htmRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: angle bracket, 3 chars wide
      false,                        // rbSelect:  initially un-selected
      ic[txtRB].ulY,                // ulY:       upper left corner in Y
      short(ic[txtRB].ulX + 19),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Label[2],                     // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[appRB],                   // nextCtrl:  link in next structure
   },
   {  //* 'Append' radiobutton  - - - - - - - - - - - - - - - - - - -    appRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: angle bracket, 3 chars wide
      true,                         // rbSelect:  initially selected
      short(ic[txtRB].ulY + 2),     // ulY:       upper left corner in Y
      2,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Label[3],                     // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[ovrRB],                   // nextCtrl:  link in next structure
   },
   {  //* 'Overwrite' radiobutton   - - - - - - - - - - - - - - - - -    ovrRB *
      dctRADIOBUTTON,               // type:      
      rbtS3a,                       // rbSubtype: angle bracket, 3 chars wide
      false,                        // rbSelect:  initially un-selected
      short(ic[appRB].ulY + 1),     // ulY:       upper left corner in Y
      ic[appRB].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      Label[4],                     // label:
      ZERO,                         // labY:      
      4,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL                          // nextCtrl:  link in next structure
   },
} ;

   //* Determine which button of the 'text/html' group is initially 'selected'.*
   if ( (this->logName.find( L".html" )) >= ZERO )
   {
      ic[txtRB].rbSelect = false ;
      ic[htmRB].rbSelect = true ;
   }

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Save Data To File  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Group 'text/html' radio buttons *
   short XorGroup[3] = { txtRB, htmRB, -1 } ;
   dp->GroupRadiobuttons ( XorGroup ) ;
   //* Group 'append/overwrite' radio buttons *
   XorGroup[0] = appRB ; XorGroup[1] = ovrRB ;
   dp->GroupRadiobuttons ( XorGroup ) ;

   OutSave outputSaved = osNONE ;      // return value

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = dp->EditPushbutton ( Info ) ;
            else                    // if we arrived here via hotkey
               Info.HotData2Primary () ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == savPB )
               {
                  //* Get filename and create path/filename *
                  gString gsFname, lfPath ;
                  dp->GetTextboxText ( fnTB, gsFname ) ;
                  this->spfCatPathFilename ( lfPath, this->cwDir, gsFname.gstr() ) ;
                  if ( gsFname.gschars() > 1 )
                  {
                     bool htmout = bool((dp->GetRbGroupSelection ( htmRB )) 
                                         == htmRB ? true : false ) ;
                     bool append = bool((dp->GetRbGroupSelection ( appRB )) 
                                         == appRB ? true : false ) ;
                     outputSaved = 
                        this->WriteLogFile ( lfPath, htmout, append ) ;
                     if ( outputSaved != osNONE )
                     {  //* Remind user that data file created *
                        // Programmer's Note: This is not an error, but we use the 
                        // error-reporting system to write the message to the console.
                        gString gsOut ;
                        gsOut.compose ( L"** Statistical data saved to file: '%S' **",
                                        gsFname.gstr() ) ;
                        if ( this->emCount < emMAX )
                           gsOut.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                        dp->WriteString ( warnLine, 2, Message[0], goodColor, true ) ;
                     }
                     else
                        dp->WriteString ( warnLine, 2, Message[2], warnColor, true ) ;
                     sleep ( msgDelay ) ;
                     done = true ;
                  }
                  else
                  {  //* Empty string *
                     dp->WriteString ( warnLine, 2, Message[3], warnColor, true ) ;
                     sleep ( msgDelay ) ;
                     dp->ClearLine ( warnLine, false ) ;
                  }
               }
               else if ( Info.ctrlIndex == canPB )
               {
                  dp->WriteString ( warnLine, 2, Message[1], warnColor, true ) ;
                  sleep ( msgDelay - 1 ) ;
                  done = true ;     // data not saved
               }
            }
            else
            { /* No button press, so nothing to do */ }
         }

         //*******************************************
         //* If focus is currently on a Text Box     *
         //*******************************************
         else if ( ic[icIndex].type == dctTEXTBOX )
         {
            //* Allow user to edit filename *
            Info.viaHotkey = false ;   // discard any hotkey data
            icIndex = dp->EditTextbox ( Info ) ;
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = dp->EditRadiobutton ( Info ) ;
            else                    // if we arrived here via hotkey
               Info.HotData2Primary () ;
            if ( Info.dataMod && (Info.ctrlIndex == txtRB || Info.ctrlIndex == htmRB) )
            {  //* Adjust the filename extension to *
               //* match the selected output format *
               short fmt = dp->GetRbGroupSelection ( txtRB ) ;
               gString newText ;
               wchar_t oldText[gsMAXCHARS] ;
               dp->GetTextboxText ( fnTB, oldText ) ;
               short i = ZERO ;
               while ( oldText[i] != NULLCHAR )    ++i ;       // end of string
               while ( i > ZERO && oldText[i] != PERIOD ) --i ;// filename extension
               if ( i > ZERO )      // if there is a real filename
               {
                  oldText[i] = NULLCHAR ;    // discard the old extension
                  newText = oldText ;
                  gString newExt ;
                  this->spfExtractFileExtension ( newExt, fmt == txtRB ? 
                                               spSaveTextStats : spSaveHtmlStats ) ;
                  newText.append ( newExt.gstr() ) ;
               }
               else                 // plug in the default name
               {
                  newText = fmt == txtRB ? spSaveTextStats : spSaveHtmlStats ;
               }
               dp->SetTextboxText ( fnTB, newText ) ;
               dp->RefreshWin () ;
            }
         }  // radio button

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while()
   }        // if((dp->OpenWindow())==OK)

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;
   return outputSaved ;

}  //* End imSave2File() *

//*************************
//*     imSortOption      *
//*************************
//******************************************************************************
//* Allow user to select the sort option for display of statistical data.      *
//*                                                                            *
//* 1) get selection                                                           *
//* 2) if new option selected                                                  *
//*    a) release old display data                                             *
//*    b) re-sort the statistical data                                         *
//*    c) create new display data                                              *
//*                                                                            *
//* Input  : ssd : (by reference) contains current display data which will be  *
//*                released, and replaced with the re-sorted data              *
//*          ctr : center of parent dialog window for positioning child dialog *
//*                                                                            *
//* Returns: 'true' if sort option changed, else 'false'                       *
//******************************************************************************

bool SrcProf::imSortOption ( ssetData& ssd, winPos& ctr )
{
const char* rbLabel[] = 
{
   "by filename ^Extension (default)",
   "by file^Name",
   "by maintainability inde^X",
   "by number of ^Source code lines",
} ;
const short dialogROWS = 12,                 // display lines
            dialogCOLS = 44,                 // display columns
            controlsDEFINED = 6,             // # of controls in dialog
            ulY = ctr.ypos - dialogROWS / 2, // upper left corner in Y
            ulX = ctr.xpos - dialogCOLS / 2 ;// upper left corner in X
enum ctrlIndices : short { okPB = ZERO, canPB, extRB, namRB, pctRB, srcRB } ;

   bool  status = false ;           // return value

   
InitCtrl ic[controlsDEFINED] =   // array of dialog control initialization objects
{
   {  //* 'OK' pushbutton - - - - - - - - - - - - - - - - - - - - - -     okPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 9),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      "   ^OK   ",                  // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[canPB],                   // nextCtrl:  link in next structure
   },
   {  //* 'CANCEL' pushbutton - - - - - - - - - - - - - - - - - - - -    canPB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      ic[okPB].ulY,                 // ulY:       upper left corner in Y
      short(dialogCOLS / 2 + 1),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      8,                            // cols:      control columns
      " ^CANCEL ",                  // dispText:  
      pbnColor,                     // nColor:    non-focus color
      pbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[extRB],                   // nextCtrl:  link in next structure
   },
   {  //* 'By Extension' radiobutton  - - - - - - - - - - - - - - - -    extRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: angle bracket, 5 chars wide
      false,                        // rbSelect:  initially un-selected
      2,                            // ulY:       upper left corner in Y
      3,                            // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[0],                   // label:
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[namRB],                   // nextCtrl:  link in next structure
   },
   {  //* 'By Filename'  radiobutton  - - - - - - - - - - - - - - - -    namRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: angle bracket, 5 chars wide
      false,                        // rbSelect:  initially un-selected
      short(ic[extRB].ulY + 2),     // ulY:       upper left corner in Y
      ic[extRB].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[1],                   // label:
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[pctRB],                   // nextCtrl:  link in next structure
   },
   {  //* 'By Maintainability' radiobutton  - - - - - - - - - - - - -    pctRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: angle bracket, 5 chars wide
      false,                        // rbSelect:  initially un-selected
      short(ic[namRB].ulY + 2),     // ulY:       upper left corner in Y
      ic[extRB].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[2],                   // label:
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[srcRB],                   // nextCtrl:  link in next structure
   },
   {  //* 'By Source Lines' radiobutton  - - - - - - - - - - - - - -     srcRB *
      dctRADIOBUTTON,               // type:      
      rbtS5a,                       // rbSubtype: angle bracket, 5 chars wide
      false,                        // rbSelect:  initially un-selected
      short(ic[pctRB].ulY + 2),     // ulY:       upper left corner in Y
      ic[extRB].ulX,                // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      0,                            // cols:      (n/a)
      NULL,                         // dispText:  (n/a) for standard button types
      rbnColor,                     // nColor:    non-focus color
      rbfColor,                     // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      rbLabel[3],                   // label:
      ZERO,                         // labY:      
      7,                            // labX       
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Determine current sort option and set   *
   //* corresonding radio button as 'selected'.*
   short oldSelection ;
   switch ( this->pd.fileSort )
   {
      case soExtension:   ic[extRB].rbSelect = true ; oldSelection = extRB ;  break ;
      case soFilename:    ic[namRB].rbSelect = true ; oldSelection = namRB ;  break ;
      case soMaintain:    ic[pctRB].rbSelect = true ; oldSelection = pctRB ;  break ;
      case soSourcelines: ic[srcRB].rbSelect = true ; oldSelection = srcRB ;  break ;
   }

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  Select Sort Option  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Group all radio buttons in the dialog *
   short XorGroup[] = { extRB, namRB, pctRB, srcRB, -1 } ;
   dp->GroupRadiobuttons ( XorGroup ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //*******************************************
         //* If focus is currently on a Pushbutton   *
         //*******************************************
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = dp->EditPushbutton ( Info ) ;
            else                    // if we arrived here via hotkey
               Info.HotData2Primary () ;

            //* If a Pushbutton was pressed *
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == okPB )
               {
                  //* Get user's selection *
                  short selMember = dp->GetRbGroupSelection ( extRB ) ;
                  if ( selMember != oldSelection )
                  {  //* Set the new sort option *
                     switch ( selMember )
                     {
                        case extRB: this->pd.fileSort = soExtension ;   break ;
                        case namRB: this->pd.fileSort = soFilename ;    break ;
                        case pctRB: this->pd.fileSort = soMaintain ;    break ;
                        case srcRB: this->pd.fileSort = soSourcelines ; break ;
                     }
                     //* If we currently have data to display, re-sort it.*
                     if ( ssd.dispItems > 1 )
                     {
                        //* Discard the old dialog-control display data *
                        this->imRelease ( ssd ) ;
                        //* Sort the analytical data according to *
                        //* specified option (this->pd.fileSort). *
                        this->SortFileData () ;
                        //* Format the data for display *
                        this->CreateDisplayStrings () ;
                        //* Write formatted display data to scrolling  *
                        //* control and alert caller to display it.    *
                        this->imFormatStatData ( ssd ) ;
                     }
                     status = true ;
                  }
                  else if ( Info.ctrlIndex == canPB )
                     { /* Display data unchanged */ }
               }
               else if ( Info.ctrlIndex == canPB )
                  { /* Display data unchanged */ }
               done = true ;
            }
            else
            { /* No button press, so nothing to do */ }
         }

         //*******************************************
         //* If focus is currently on a Radio Button *
         //*******************************************
         else if ( ic[icIndex].type == dctRADIOBUTTON )
         {
            if ( Info.viaHotkey == false )
               icIndex = dp->EditRadiobutton ( Info ) ;
            else                    // if we arrived here via hotkey
               Info.HotData2Primary () ;
         }  // radio button

         //* If user exited the control edit method via a hotkey,*
         //* then the new control already has focus. Otherwise,  *
         //* move input focus to next/previous control.          *
         if ( done == false && Info.viaHotkey == false )
         {
            if ( Info.keyIn == nckSTAB )
               icIndex = dp->PrevControl () ; 
            else
               icIndex = dp->NextControl () ;
         }
      }     // while()
   }        // if((dp->OpenWindow())==OK)

   if ( dp != NULL )                         // close the window
      delete ( dp ) ;
   
   return status ;

}  //* End imSortOption() *

//*************************
//*      HelpDialog       *
//*************************
//******************************************************************************
//* Display the application's Help window.                                     *
//*                                                                            *
//*                                                                            *
//* Input  : ulY : upper left position in Y                                    *
//*          ulX : upper left position in X                                    *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Please refer to SourceProfiler.texi and/or its Texinfo output:             *
//* srcprof.info for more complete documentation.                              *
//*                                                                            *
//******************************************************************************

void SrcProf::HelpDialog ( short ulY, short ulX, 
                           const char* ncdVersion, const char* nclVersion )
{
const short dialogROWS = 23 ;       // # of display lines
const short dialogCOLS = 78 ;       // # of display columns
const short controlsDEFINED = 2 ;   // # of controls in dialog
const short hdLINES = 141 ;         // # of text lines in helpData array
gString hdTitle, hdCopyright, hdncdVersion, hdnclVersion ;
hdTitle.compose( L" %S%S", AppTitle1, AppVersion ) ;
hdCopyright.compose( L" Copyright%S", AppTitle2 ) ;
hdncdVersion.compose( L"   NcDialog-class  v:%s", ncdVersion ) ;
hdnclVersion.compose( L"   ncurses library v:%s", nclVersion ) ;

const char* helpData[hdLINES] = 
{
   "                         Source Profiler Quick Help",
   " ",
   "Source Profiler (srcprof) is a source code analysis tool for determining",
   "the 'maintainability' of source code modules based on the ratio of",
   "comments and white space to the number of source statements.",
   "'srcprof' can be used to profile source code for high-level languages such",
   "as C, C++ and Java, as well as various assembly languages and scripting",
   "languages.",
   " ",
   "Operational Overview",
   " Source Profiler has two operating modes:",
   "  1) Interactive Mode (default)",
   "    - In this mode, analytical data for specified files are displayed in",
   "      a dialog window.",
   "    - If no source files were specified for analysis, then you may select",
   "      files to be processed from a list of recognized source code files",
   "      found in the directory tree.",
   "  2) Text-only Mode",
   "    - In this mode, the application acts like a simple, text-based",
   "      command-line utility and analytical data for specified files is",
   "      displayed as plain text.",
   "    - If no valid source files are specified for processing, the",
   "      application will exit with a warning message.",
   " ",
   " If invalid command-line parameters are specified or if one or more of the",
   " specified files is a) not found, b) is not accessible or c) is not a",
   " 'regular' file, then the application will display as much diagnostic",
   " information as possible to the console window, and then exit.",
   " See below for a complete list of command-line parameters.",
   " ",
   "Interactive Mode Controls",
   " 1) Scroll-window: Displays either profile data or file-selection data.",
   " 2) DONE: Exit the application.",
   " 3) SAVE: Save currently-displayed profile data to a file.",
   " 4) SORT: Select profile-data sort option.",
   " 5) HELP: Brief Help. For full documentation, use Texinfo: 'info srcprof'.",
   " 6) SELECT FILES: Switch from profile-data view to file-selection view.",
   " 7) 'Legend' (view only): Color codes for display of profile data.",
   " 8) 'Files Selected': number of files selected in current directory",
   "     (view only) visible only during file selection,",
   " 9) 'Total': total number of files selected in this session.",
   "     (view only) visible only during file selection,",
   " ",
   "Interactive Mode File Selection",
   " When 'SELECT FILES' pushbutton is selected, a list of files and",
   " subdirectories is displayed in the scroll window.",
   " - Navigate within the window using the scrolling keys:",
   "                 Up, Down, PageUp, PageDown, Home, End.",
   " - To change directories, highlight the directory name and press ENTER.",
   " - To select/deselect a file for analysis, highlight the file and press",
   "   SPACE key or ENTER.",
   " - Control+A: select or de-select ALL files in the current directory.",
   " - Control+C: Clear (discard) all selections in this session",
   " - Press the TAB key when selection is complete.",
   "   The selected files will be analyzed and the results displayed.",
   " ",
   "Maintainability Analysis",
   " Analysis of the source code is quite simple. Each line in the source file",
   " is categorized as one of the following:",
   "  1) a line which contains only source code",
   "  2) a line which contains both source code and comments",
   "  3) a line which contains only comments",
   "  4) a line which contains only whitespace (spaces, tabs, or no data)",
   " Standard source files created on Linux/UNIX systems end each source line",
   " with a newline character, but source files created on systems which end",
   " lines with a carriage-return/linefeed character combination are also",
   " recognized.",
   " ",
   " Source Profiler determines the 'maintainability' of the code based on the",
   " ratio of comments and whitespace to the total lines in the source module.",
   " ",
   " Guidelines for C/C++, Java and other high-level languages",
   "  Code written in a high-level language should contain at least",
   "  40% comments and whitespace, and never less than 25% comments.",
   " Guidelines for Assembly Languages",
   "  Code written in assembly languages should contain comments",
   "  on no less than 50% of the source file lines.",
   " ",
   "            \"Programs must be written for people to read,",
   "             and only incidentally for machines to execute.\"",
   "             -- Harold Abelson and Gerald Jay Sussman (MIT)",
   " ",
   "Supported Source-file Types",
   " The filename extension determines the way source files are scanned.",
   " The following filename extensions are recognized.",
   " (Recognition of filename extensions is not case-sensitive.)",
   "   .c    = C language source files",
   "   .h    = C header files",
   "   .cpp  = C++ language source files",
   "   .hpp  = C++ header files",
   "   .java = Java language source files",
   "   .js   = Javascript files",
   "   .as   = Flash ActionScript files",
   "   .m .mm= Objective-C source files",
   "   .cs   = C# source files",
   "   .asm  = Assembly language source files (Intel, Motorola, etc.)",
   "   .inc  = Assembly language header files",
   "   .m51  = 8051-family assembly language files",
   "   .s07  = 68HC11-family assembly language files",
   "   .s33  = 68HC12-family assembly language files",
   "   .py .pyw = Python script files",
//   "   .pl, .pm, .t = Perl script files",
//   "   .php .phtml .php3 .php4 .php5 .phps = PHP script files",
//   "   .rb .rbw = Ruby script files",
   "   .vb .vbs = Visual Basic source files and VBScript files",
//   "   .sql  = SQL Standard Query Language (ANSI/ISO, INFORMIX, SQL-99)",
   "   .texi .texinfo = Texinfo mark-up source files",
   "   .html .htm = HTML mark-up files",
   " If your source files do not use these extensions, but you believe that",
   " the syntax for comments is similiar to one of the above, simply rename",
   " your source file and give it a try.",
   " ",
   "Command-line Arguments",
   " Usage  : srcprof [OPTIONS] [FILENAMES]",
   "          Arguments may be specified in any order.",
   "          Any reasonable number of source filenames may be specified.",
   " EXAMPLES: srcprof GooeyCode.cpp HeadCode.hpp GoldCode.asm",
   "           srcprof -i *.[hc]pp -S=n",
   "           srcprof -ta",
   " -I, -i Start the application in interactive mode (default).",
   " -T, -t Start the application in text-only mode.",
   " -A, -a Profile all source code files in current directory.",
   " -L, -l Specify the name of a plain text file containing a list of source",
   "        files to be analyzed, one filename or path/filename per line.",
   "        Any reasonable number of source filenames may be specified.",
   "           EXAMPLE:  srcprof -L=source_file_list.txt",
   " -D, -d Profile all source code files in the specified directory.",
   "           EXAMPLE:  srcprof -D=./src/videoConvert",
   " -S, -s Sort option with sub-option [e | n | m | s] Extension (default),",
   "        Name, Maintainability index, Source code lines. EXAMPLE: -S=n",
   " -M, -m Maintainability thresholds. Specify alternate thresholds for code",
   "        evaluation. The values (range 0-100%) are interpreted as",
   "        'good high-level', 'fair high-level', 'good assembler' and",
   "        'fair assembler'.  EXAMPLE: -M=40,25,55,45  (default values)",
   " -H, -h --help  Command-line Help",
   " --version      Display version number and copyright notice",
   " ",
   hdTitle.ustr(),
   hdCopyright.ustr(),
   hdnclVersion.ustr(),
   hdncdVersion.ustr(),
   " Developed using GNU G++ under Fedora Linux,",
   " Software released under GNU General Public License(GPL v3+)",
   " Documentation released under GNU Free Documentation License(FDL v1.3+)",
   " For bugs or suggestions, visit: http://www.SoftwareSamurai.us/",
};//xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxxxxx0xxxxxx|
const attr_t hdtColor = nc.br, hdbColor = nc.bl, hdhColor = nc.ma, hdqColor = nc.gr ;
const attr_t hdAttr[hdLINES + 8] = 
{
   // Title and introduction
   hdtColor, 
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, 

   // Operational Overview
   hdhColor, 
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor,  
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor,  
   hdbColor, hdbColor, hdbColor, hdbColor, 

   // Interactive Mode Controls
   hdhColor,
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor,  
   hdbColor, hdbColor, hdbColor, hdbColor, 

   // Interactive Mode File Selection
   hdhColor,
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor,  
   hdbColor, hdbColor, hdbColor, hdbColor,  

   // Maintainability Analysis
   hdhColor,
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor,  
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor,  
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor,   

   // Quotation
   hdqColor, hdqColor, hdqColor,
   hdbColor,

   // Supported Source-file Types
   hdhColor, 
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, 
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, 
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, 
   hdbColor, //hdbColor, hdbColor, 

   // Command-line Arguments
   hdhColor, 
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, 
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, 
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, 

   // Development info
   hdtColor, hdtColor, hdqColor, hdqColor, hdtColor, hdtColor, hdtColor, hdtColor,

   // (safety margin)
   hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, hdbColor, 
} ;
enum ctrls { closePB = ZERO, helpSE } ;
attr_t dColor = nc.blR ;

InitCtrl ic[controlsDEFINED] =      // array of dialog control info
{
   {  //* 'CLOSE' pushbutton  - - - - - - - - - - - - - - - - - - -    closePB *
      dctPUSHBUTTON,                // type:      
      rbtTYPES,                     // rbSubtype: (n/a)
      false,                        // rbSelect:  (n/a)
      short(dialogROWS - 2),        // ulY:       upper left corner in Y
      short(dialogCOLS / 2 - 4),    // ulX:       upper left corner in X
      1,                            // lines:     (n/a)
      7,                            // cols:      control columns
      " CLOSE ",                    // dispText:  
      hdhColor,                     // nColor:    non-focus color
      (hdhColor | ncrATTR),         // fColor:    focus color
      tbPrint,                      // filter:    (n/a)
      "",                           // label:     (n/a)
      ZERO,                         // labY:      (n/a)
      ZERO,                         // labX       (n/a)
      ddBoxTYPES,                   // exType:    (n/a)
      1,                            // scrItems:  (n/a)
      1,                            // scrSel:    (n/a)
      NULL,                         // scrColor:  (n/a)
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      &ic[1],                       // nextCtrl:  link in next structure
   },
   {  //* 'HELP' Scroll Ext control - - - - - - - - - - - - - - - -     helpSE *
      dctSCROLLEXT,                 // type:      define a scrolling-data control
      rbtTYPES,                     // rbSubtype: (na)
      false,                        // rbSelect:  (n/a)
      ZERO,                         // ulY:       upper left corner in Y
      ZERO,                         // ulX:       upper left corner in X
      short(dialogROWS - 2),        // lines:     control lines
      dialogCOLS,                   // cols:      control columns
      NULL,                         // dispText:  n/a
      dColor,                       // nColor:    non-focus border color
      nc.cyG,                       // fColor:    focus border color
      tbPrint,                      // filter:    (n/a)
      NULL,                         // label:     
      ZERO,                         // labY:      offset from control's ulY
      ZERO,                         // labX       offset from control's ulX
      ddBoxTYPES,                   // exType:    (n/a)
      ZERO,                         // scrItems:  number of data items
      ZERO,                         // scrSel:    (n/a)
      NULL,                         // scrColor:  single-color data display
      NULL,                         // spinData:  (n/a)
      true,                         // active:    allow control to gain focus
      NULL,                         // nextCtrl:  link in next structure
   },
} ;

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       NULL,           // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       ic              // pointer to list of control definitions
                     ) ;

   //* Instantiate the dialog window *
   NcDialog* dp = new NcDialog ( dInit ) ;

   //* Open the dialog window *
   if ( (dp->OpenWindow()) == OK )
   {
      //* Connect Scroll Ext control to bordetrs *
      cdConnect cdConn ;
      cdConn.connection = true ;
      cdConn.ll2Left = cdConn.lr2Right = true ;
      dp->ConnectControl2Border ( helpSE, cdConn ) ;

      //* Initialize the display data
      ssetData sData( helpData, hdAttr, hdLINES, ZERO, false ) ;
      dp->SetScrollextText ( 1, sData ) ;

      dp->RefreshWin () ;           // make everything visible

      uiInfo Info ;                 // user interface data returned here
      short  icIndex = ZERO ;       // index of control with input focus
      bool   done = false ;         // loop control
      while ( ! done )
      {
         //* If focus is currently on a Pushbutton *
         if ( ic[icIndex].type == dctPUSHBUTTON )
         {
            icIndex = dp->EditPushbutton ( Info ) ;
            if ( Info.dataMod != false )
            {
               if ( Info.ctrlIndex == closePB )
                  done = true ;
            }
         }
         //* If focus is currently on a Scrollext control *
         else if ( ic[icIndex].type == dctSCROLLEXT )
         {
            icIndex = dp->ViewScrollext ( Info ) ;
         }

         if ( ! done )
         {
            icIndex = dp->NextControl () ;
         }
      }
   }
   if ( dp != NULL )
      delete ( dp ) ;                        // close the window

}  //* End HelpDialog() *

#if DEBUG_CODE != 0     // DEBUGGING METHODS
//*************************
//*       imDebug         *
//*************************
//******************************************************************************
//* DEBUG ONLY - Display debugging info.                                       *
//* NOTE: Terminal window must be >= 132 x 37.                                 *
//*                                                                            *
//* Input  : ssd     : (by reference) display data                             *
//*          stayOpen: (optional, false bt default) if 'true', stay open       *
//*          msg     : (optional, NULL by default) message                     *
//*                                                                            *
//* Returns: pointer to open dialog, or NULL* if dialog closed                 *
//******************************************************************************

NcDialog* SrcProf::imDebug ( const ssetData& ssd, bool stayOpen, const char* msg )
{
const char* ssdHead = "dispText :\ndispColor:\ndispItems:\nhlIndex  :\nhlShow   :" ;
const char* pdHead  = "fData      :\nfCount     :\n"
                      "dispData   :\ngoodAttr   :\nfairAttr   :\npoorAttr   :\n"
                      "dfltAttr   :\n"
                      "goodpctC   :\nfairpctC   :\ngoodpctA   :\nfairpctA   :\n"
                      "tLineCount :\ntSourceLine:\ntCommentLin:\ntMixedLines:\n"
                      "tBlankLines:\ntMaintIndex:\nsummaryData:" ;
const char* tnfHead = "tnfPtr  :\ntnfCount:" ;
const short dialogROWS = debugROWS ;   // # of display lines
const short dialogCOLS = debugCOLS ;   // # of display columns
const short ulY = ZERO ;               // dialog position
const short ulX = minTERMCOLS ;
const short clear2here = dialogROWS - 5 ; // clear-line limit
const attr_t dColor = nc.blR,          // dialog base colr
            hColor = nc.blG ;          // heading color
winPos      wp( 1, 2 ),                // data display position
            wpm( clear2here, 2 ) ;     // 'msg' display position
static NcDialog* dp =  NULL ;          // return value

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       " DEBUG DATA ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate and open the dialog window *
   if ( dp == NULL )
   {
      dp = new NcDialog ( dInit ) ;
      dp->OpenWindow() ;
   }
   //* Clear right half of existing window *
   else
   {
      wp = { 1, (dialogCOLS / 2) } ;
      for ( short y = wp.ypos ; y < clear2here ; y++ )
         dp->ClearLine ( y, false, wp.xpos ) ;
      ++wpm.ypos ;
   }
   if ( dp != NULL )
   {
      dp->WriteString ( wp.ypos++, wp.xpos, "ssetData ssd        ", (hColor | ncuATTR) ) ;
      dp->WriteParagraph ( wp.ypos, wp.xpos, ssdHead, hColor ) ;
      gString gsOut ;
      gsOut.compose( L"%p\n%p\n%d\n%d\n%hhd", 
            ssd.dispText, ssd.dispColor, &ssd.dispItems, &ssd.hlIndex, &ssd.hlShow ) ;
      dp->WriteParagraph ( wp.ypos, (wp.xpos + 11), gsOut.gstr(), dColor ) ;
      wp.ypos += 6 ;
      dp->WriteString ( wp.ypos++, wp.xpos, "ProfData pd         ", (hColor | ncuATTR) ) ;
      dp->WriteParagraph ( wp.ypos, wp.xpos, pdHead, hColor ) ;
      gsOut.compose( L"%p\n%u\n%p\n0x%08X\n0x%08X\n0x%08X\n0x%08X\n"
                      "%3.1lf\n%3.1lf\n%3.1lf\n%3.1lf\n"
                      "%4u\n%4u\n%4u\n%4u\n%4u\n%3.1lf\n", 
                     this->pd.fileData, &this->pd.fileCount, 
                     this->pd.dispData, 
                     &this->pd.goodAttr, &this->pd.fairAttr, 
                     &this->pd.poorAttr, &this->pd.dfltAttr, 
                     &this->pd.goodpctC, &this->pd.fairpctC, 
                     &this->pd.goodpctA, &this->pd.fairpctA,
                     &this->pd.tLineCount,  &this->pd.tSourceLines, &this->pd.tCommentLines,
                     &this->pd.tMixedLines, &this->pd.tBlankLines, &this->pd.tMaintIndex ) ;
      winPos wptmp = dp->WriteParagraph ( wp.ypos, (wp.xpos + 13), gsOut.gstr(), dColor ) ;
      gString gsSum( L"---" ) ;
      short sumCh = this->pd.summaryData.gschars() ;
      if ( sumCh >= maxFNAMECOLS )
      {
         gsSum = &this->pd.summaryData.gstr()[12] ;
         gsSum.limitColumns( 10 ) ;
         dp->WriteString ( wptmp.ypos, wptmp.xpos, gsSum, dColor ) ;
         ++wptmp.ypos ;
         wptmp.xpos = 3 ;
         gsSum = &this->pd.summaryData.gstr()[30] ;
      }
      dp->WriteString ( wptmp.ypos, wptmp.xpos, gsSum, dColor ) ;
      wp.ypos += 21 ;
      dp->WriteString ( wp.ypos++, wp.xpos, "tnFName tnfPtr      ", (hColor | ncuATTR) ) ;
      dp->WriteParagraph ( wp.ypos, wp.xpos, tnfHead, hColor ) ;
      gsOut.compose( L"%p\n%u", this->tnfPtr, &this->tnfCount ) ;
      dp->WriteParagraph ( wp.ypos, (wp.xpos + 13), gsOut.gstr(), dColor ) ;

      if ( msg != NULL )
      {
         dp->ClearLine ( wpm.ypos, false ) ;
         dp->WriteString ( wpm.ypos, wpm.xpos, msg, dColor ) ;
      }
      
      wp = { (dialogROWS - 2), (dialogCOLS / 2 - 7) } ;
      dp->WriteString ( wp.ypos, wp.xpos, " PRESS A KEY ", nc.reG ) ;
      dp->RefreshWin () ;
      if ( ! stayOpen )
         nckPause() ;
   }
   if ( !stayOpen && dp != NULL )
   {
      delete ( dp ) ;               // close the window
      dp = NULL ;
   }
   return dp ;

}  //* End imDebug() *

//*************************
//*     imDump_tnfPtr     *
//*************************
//******************************************************************************
//* DEBUG ONLY - Display debugging info.                                       *
//* NOTE: Terminal window must be >= 132 x 40.                                 *
//*                                                                            *
//* Input  : stayOpen: (optional, false bt default) if 'true', stay open       *
//*          msg     : (optional, NULL by default) message                     *
//*                                                                            *
//* Returns: pointer to open dialog, or NULL* if dialog closed                 *
//******************************************************************************

NcDialog* SrcProf::imDump_tnfPtr ( bool stayOpen, const char* msg )
{
const short dialogROWS = debugROWS ;   // # of display lines
const short dialogCOLS = debugCOLS ;   // # of display columns
const short ulY = ZERO ;               // dialog position
const short ulX = minTERMCOLS ;
const short clear2here = dialogROWS - 5 ; // clear-line limit
const attr_t dColor = nc.blR,          // dialog base colr
            hColor = nc.blG ;          // heading color
winPos      wp( 1, 2 ),                // data display position
            wpm( clear2here, 2 ) ;     // 'msg' display position
static NcDialog* dp =  NULL ;          // return value

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  DUMP tnfPtr  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate and open the dialog window *
   if ( dp == NULL )
   {
      dp = new NcDialog ( dInit ) ;
      dp->OpenWindow() ;
   }
   //* Clear right half of existing window *
   else
   {
      wp = { 1, (dialogCOLS / 2) } ;
      for ( short y = wp.ypos ; y < clear2here ; y++ )
         dp->ClearLine ( y, false, wp.xpos ) ;
      ++wpm.ypos ;
   }
   if ( dp != NULL )
   {
      gString gsOut ;
      gsOut.compose( L"Ptr:%p Count:%u", this->tnfPtr, &this->tnfCount ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, gsOut, (hColor | ncuATTR) ) ;

      gString gsName ;
      wchar_t dirType ;
      for ( UINT i = ZERO ; i < this->tnfCount ; i++ )
      {
         gsName = this->tnfPtr[i].fName ;
         gsName.limitColumns ( 18 ) ;
         dirType = this->tnfPtr[i].fType == fmDIR_TYPE ? L'D' : L' ' ;
         gsOut.compose ( L"%02d %C %S", &i, &dirType, gsName.gstr() ) ;
         dp->WriteString ( wp.ypos++, wp.xpos, gsOut, dColor ) ;
         if ( wp.ypos >= clear2here )
            break ;
      }

      if ( msg != NULL )
      {
         dp->ClearLine ( wpm.ypos, false ) ;
         dp->WriteString ( wpm.ypos, wpm.xpos, msg, dColor ) ;
      }
      
      wp = { (dialogROWS - 2), (dialogCOLS / 2 - 7) } ;
      dp->WriteString ( wp.ypos, wp.xpos, " PRESS A KEY ", nc.reG ) ;
      dp->RefreshWin () ;
      if ( ! stayOpen )
         nckPause() ;
   }
   if ( !stayOpen && dp != NULL )
   {
      delete ( dp ) ;               // close the window
      dp = NULL ;
   }
   return dp ;

}  //* End imDump_tnfPtr() *

//*************************
//*    imDump_fileData    *
//*************************
//******************************************************************************
//* DEBUG ONLY - Display debugging info.                                       *
//* NOTE: Terminal window must be >= 132 x 40.                                 *
//*                                                                            *
//* Input  : stayOpen: (optional, false bt default) if 'true', stay open       *
//*          msg     : (optional, NULL by default) message                     *
//*                                                                            *
//* Returns: pointer to open dialog, or NULL* if dialog closed                 *
//******************************************************************************

NcDialog* SrcProf::imDump_fileData ( bool stayOpen, const char* msg )
{
const short dialogROWS = debugROWS ;   // # of display lines
const short dialogCOLS = debugCOLS ;   // # of display columns
const short ulY = ZERO ;               // dialog position
const short ulX = minTERMCOLS ;
const short clear2here = dialogROWS - 5 ; // clear-line limit
const attr_t hColor = nc.blG ;         // heading color
winPos      wp( 1, 2 ),                // data display position
            wpm( clear2here, 2 ) ;     // 'msg' display position
static NcDialog* dp =  NULL ;          // return value

   //* Initial parameters for dialog window *
   InitNcDialog dInit( dialogROWS,     // number of display lines
                       dialogCOLS,     // number of display columns
                       ulY,            // Y offset from upper-left of terminal 
                       ulX,            // X offset from upper-left of terminal 
                       "  DUMP fileData  ", // dialog title
                       ncltSINGLE,     // border line-style
                       dColor,         // border color attribute
                       dColor,         // interior color attribute
                       NULL            // pointer to list of control definitions
                     ) ;

   //* Instantiate and open the dialog window *
   if ( dp == NULL )
   {
      dp = new NcDialog ( dInit ) ;
      dp->OpenWindow() ;
   }
   //* Clear right half of existing window *
   else
   {
      wp = { 1, (dialogCOLS / 2) } ;
      for ( short y = wp.ypos ; y < clear2here ; y++ )
         dp->ClearLine ( y, false, wp.xpos ) ;
      ++wpm.ypos ;
   }
   if ( dp != NULL )
   {
      gString gsOut ;
      gsOut.compose( L"P:%p C:%u S:%hd", this->pd.fileData, 
                     &this->pd.fileCount, &this->pd.fileSort ) ;
      dp->WriteString ( wp.ypos++, wp.xpos, gsOut, (hColor | ncuATTR) ) ;

      gString gsName ;
      for ( UINT i = ZERO ; i < this->pd.fileCount ; i++ )
      {
         this->spfExtractFilename ( gsName, this->pd.fileData[i].fName ) ;
         gsName.limitColumns ( 20 ) ;
         gsOut.compose ( L"%02d %S", &i, gsName.gstr() ) ;
         dp->WriteString ( wp.ypos++, wp.xpos, gsOut, dColor ) ;
         if ( wp.ypos >= clear2here )
            break ;
      }

      if ( msg != NULL )
      {
         dp->ClearLine ( wpm.ypos, false ) ;
         dp->WriteString ( wpm.ypos, wpm.xpos, msg, dColor ) ;
      }
      
      wp = { (dialogROWS - 2), (dialogCOLS / 2 - 7) } ;
      dp->WriteString ( wp.ypos, wp.xpos, " PRESS A KEY ", nc.reG ) ;
      dp->RefreshWin () ;
      if ( ! stayOpen )
         nckPause() ;
   }
   if ( !stayOpen && dp != NULL )
   {
      delete ( dp ) ;               // close the window
      dp = NULL ;
   }
   return dp ;

}  //* End imDump_fileData() *
#endif   // DEBUG_CODE
#undef DEBUG_CODE

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************


