//******************************************************************************
//* File       : SrcProf_List.cpp                                              *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 1998-2015 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in SrcProf.hpp           *
//* Date       : 22-Oct-2015                                                   *
//* Version    : (see AppVersion string in SrcProf.hpp)                        *
//*                                                                            *
//* Description: This module contains methods for creating and validating a    *
//* list of source code files to be analyzed. A temporary file is created in   *
//* the working directory where the application was invoked.                   *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Copyright Notice:                                                          *
//* This program is free software: you can redistribute it and/or modify it    *
//* under the terms of the GNU General Public License as published by the Free *
//* Software Foundation, either version 3 of the License, or (at your option)  *
//* any later version.                                                         *
//*                                                                            *
//* This program is distributed in the hope that it will be useful, but        *
//* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY *
//* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
//* for more details.                                                          *
//*                                                                            *
//* You should have received a copy of the GNU General Public License along    *
//* with this program.  If not, see <http://www.gnu.org/licenses/>.            *
//*                                                                            *
//*         Full text of the GPL License may be found in the Texinfo           *
//*         documentation for this program under 'Copyright Notice'.           *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "SrcProf.hpp"

//****************
//* Local Data   *
//****************

//*************************
//*   CreateFileList      *
//*************************
//******************************************************************************
//* Create a file containing a list of all files in the specified directory    *
//* which have a filename extension supported by this application.             *
//*                                                                            *
//* Input  : rawPath: path/dirname of directory to scan                        *
//*                                                                            *
//* Returns: OK   if list successfully created                                 *
//*          ERR  if error creating list or if no files in list                *
//******************************************************************************

short SrcProf::CreateFileList ( const char* rawPath )
{
   short status = ERR ;

   //* Convert provided path to full path specification *
   char fullPath[gsMAXBYTES] ;
   if ( (realpath ( rawPath, fullPath )) != NULL )
   {
      //* Verify that directory exists *
      gString dPath( fullPath ) ;
      if ( (this->spfTargetExists ( dPath, true )) )
      {
         //* Read the directory contents and build an array of file            *
         //* descriptions for all recognized source code files. Array is       *
         //* attached to this->tnfPtr and number of entries is this->tnfCount. *
         this->tnfRelease () ;      // release any previous allocation
         //* If directory contains at least one valid source code file, create *
         //* the temporary file containing list of source code filenames.      *
         if ( (this->spfReadDirectory ( dPath )) > ZERO )
         {
            status = this->CreateFileList ( dPath ) ;
         }
         else
         {
            gString gst, gsName ;
            this->spfExtractFilename ( gsName, dPath.ustr() ) ;
            gst.compose( badFileTemplate[bftEmptyDir], gsName.ustr() ) ;
            gst.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
         }
         //* Release the dynamically-allocated memory *
         this->tnfRelease () ;
      }
      else if ( this->emCount < emMAX )
      {
         gString gst, gsName ;
         this->spfExtractFilename ( gsName, fullPath ) ;
         gst.compose( badFileTemplate[bftDirNotFound], gsName.ustr() ) ;
         gst.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
      }
   }
   else if ( this->emCount < emMAX )
   {
      gString gst, gsName ;
      this->spfExtractFilename ( gsName, rawPath ) ;
      gst.compose( badFileTemplate[bftDirNotFound], gsName.ustr() ) ;
      gst.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
   }
   return status ;

}  //* End CreateFileList() *

//*************************
//*   CreateFileList      *
//*************************
//******************************************************************************
//* Create a file containing a list of all 'selected' files in the specified   *
//* directory which have a filename extension supported by this application.   *
//*                                                                            *
//* Input    dPath : full path specification for source directory              *
//*                   this->tnfPtr points to an array of file-description      *
//*                   objects, and this->tnfCount == number of elements        *
//*                   in the array                                             *
//*          scanCount: (optional, -1 by default)                              *
//*                   if specified, this value overrides this->tnfCount as the *
//*                   number of items in the array to be processed i.e.        *
//*                   process only the first scanCount items                   *
//*          append : (optional, false by default)                             *
//*                   if 'true' append new records to existing file if it      *
//*                             exists, else create the file                   *
//*                   if 'false' truncate exitsing file, if any, before        *
//*                             writing new records                            *
//*                                                                            *
//* Returns: OK   if list successfully created                                 *
//*          ERR  if error creating list or if no files in list                *
//******************************************************************************

short SrcProf::CreateFileList ( const gString& dPath, int scanCount, bool append )
{
   UINT fCount = (scanCount > ZERO && UINT(scanCount) <= this->tnfCount) ? 
                  scanCount : this->tnfCount ;
   short status = ERR ;

   //* If directory contains at least one valid source code file, create *
   //* the temporary file and write list of source code filenames.       *
   if ( fCount > ZERO && this->tnfPtr != NULL )
   {
      ofstream ofs ;
      if ( append != false )
         ofs.open( this->lstFile.ustr(), ofstream::out | ofstream::app ) ;
      else
         ofs.open( this->lstFile.ustr(), ofstream::out | ofstream::trunc ) ;
      if ( ofs.is_open() )             // if input file open
      {
         ofs << "# Source file list for:\n#  " << dPath.ustr() << endl ;

         //* Write the filenames to the list file.               *
         //* 'selected' files are grouped at the top of the list.*
         for ( UINT i = ZERO ; i < fCount ; i++ )
         {
            ofs << dPath.ustr() << "/" << this->tnfPtr[i].fName << endl ;
         }
         ofs.close() ;        // close the list file
         status = OK ;        // return the good news
      }
      else if ( this->emCount < emMAX )
      {
         gString gst( badFileTemplate[bftBadTempfile] ) ;
         gst.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
      }
   }
   return status ;

}  //* End CreateFileList() *

//*************************
//*     AddFileToList     *
//*************************
//******************************************************************************
//* Add one path/filename to the list of files to be analyzed.                 *
//* 1) Expand to full, path/filename spec.                                     *
//* 2) Verify that file exists.                                                *
//* 3) Verify that file is a supported source-code filetype.                   *
//* 4) Write the path/filename spec to the output file.                        *
//*                                                                            *
//* Input  : ofs     : pointer to the open file                                *
//*          srcFile : file name or partial path/filename of source file       *
//*          errorMsg: (optional, false by default)                            *
//*                    if true AND if file not added to list, then generate    *
//*                       an error message                                     *
//*                                                                            *
//* Returns: 'true'  if file successfully added to list                        *
//*          'false' if file not found or not validated                        *
//******************************************************************************

bool SrcProf::AddFileToList ( ofstream& ofs, const gString& srcFile, bool errorMsg )
{
   gString fileSpec, fileName, gs ;
   bool    status = false ;

   if ( ((this->spfRealpath ( srcFile.ustr(), fileSpec )) != false) &&
        ((this->spfTargetExists ( fileSpec )) != false) )
   {
      if ( (this->GetFileFamily ( fileSpec.ustr() )) != sffUnk )
      {
         if ( ofs.is_open() )
         {
            ofs << fileSpec.ustr() << endl ;
            status = true ;
         }
      }
      else if ( errorMsg != false ) // unsupported-file-type
      {
         if ( this->emCount < emMAX )
         {
            this->spfExtractFilename ( fileName, fileSpec ) ;
            gs.compose( badFileTemplate[bftUnsupported], fileName.gstr() ) ;
            gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
         }
      }
   }
   else if ( errorMsg != false )  // file-not-found
   {
      if ( this->emCount < emMAX )
      {
         this->spfExtractFilename ( fileName, srcFile ) ;
         gs.compose( badFileTemplate[bftNotFound], fileName.ustr() ) ;
         gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
      }
      status = false ;
   }
   return status ;

}  //* End AddFileToList() *

bool SrcProf::AddFileToList ( ofstream& ofs, const char* srcFile, bool errorMsg )
{

   gString gs( srcFile ) ;
   return ( this->AddFileToList ( ofs, gs, errorMsg ) ) ;

}  //* End AddFileToList() *

//*************************
//*   ValidateFileList    *
//*************************
//******************************************************************************
//* Read the file which contains a list of sourcefile path/filenames, one      *
//* path/filename for each line.                                               *
//* Verify that:                                                               *
//*  1) specified list file exists and that we have read access                *
//*  2) that each target file:                                                 *
//*     a) exists                                                              *
//*     b) is a 'regular' file                                                 *
//*     c) is a supported sourcefile type                                      *
//*  3) allocate analysis and display space                                    *
//*  4) initialize each file's path/filename (fileData::fPath)                 *
//*                           and   filename (fileData:fName)                  *
//*                                                                            *
//* File lines that begin with a C++ comment '//' are ignored.                 *
//*                                                                            *
//* Input  : none  (this->lstFile references the list file)                    *
//*                                                                            *
//* Returns: number of valid sourcefile names in the list                      *
//*          if > ZERO,  then this->pd.listCount == number of filename strings *
//*                      pd.fileData::fPath and ::fName have been initialized  *
//*          if == ZERO, then this->pd.listCount == ZERO                       *
//******************************************************************************

UINT SrcProf::ValidateFileList ( void )
{
   gString gsPath ;     // filename statistics and formatted output

   this->pdRelease () ; // release any old data (listCount == ZERO, listStrings == NULL)

   //* Verify that the list file exists *
   if ( this->spfTargetExists ( this->lstFile ) )
   {
      //* Open the list file and count the number of valid sourcefile names.*
      ifstream ifs ( this->lstFile.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )       // if input file open
      {
         char  tmp[gsMAXBYTES] ; // UTF-8 line input from file
         UINT  goodCount = ZERO, // number of validated filenames
               badCount = ZERO ; // number of un-validated filenames
         bool  done = false ;    // loop control
         while ( ! done )
         {
            //* Read a source line *
            ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;

            if ( ifs.good() )
            {
               //* If not a blank line AND not a comment *
               gsPath = tmp ;
               if ( (gsPath.gschars() > 1) && (*gsPath.gstr() != HASH) )
               {
                  //* If the target exists AND is a supported filetype *
                  if ( this->spfTargetExists ( gsPath ) )
                  {
                     if ( this->GetFileFamily ( gsPath.ustr()) != sffUnk )
                     {
                        ++goodCount ;     // target file validated
                     }
                     else
                     {
                        if ( this->emCount < emMAX )
                        {
                           gString gst, gsName ;
                           this->spfExtractFilename ( gsName, gsPath.ustr() ) ;
                           gst.compose( badFileTemplate[bftUnsupported], gsName.gstr() ) ;
                           gst.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                        }
                        ++badCount ;         // count files not validated
                     }
                  }
                  else
                  {
                     if ( this->emCount < emMAX )
                     {
                        gString gst, gsName ;
                        this->spfExtractFilename ( gsName, gsPath.ustr() ) ;
                        gst.compose( badFileTemplate[bftNotFound], gsName.ustr() ) ;
                        gst.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
                     }
                     ++badCount ;         // count files not found
                  }
               }
            }  // good line read
            else                    // end of file (or read error)
               done = true ;
         }     // while()

         //* If at least one valid file AND no invalid files *
         if ( goodCount > ZERO && badCount == ZERO )
         {
            //* Clear any error flags and return to top of input file.*
            ifs.clear() ;
            ifs.seekg( ZERO ) ;
            //* Allocate space for analysis and display *
            this->pdAllocate ( goodCount ) ;

            //* Read the file again, saving the target files' data.*
            done = false ;
            for ( UINT pdIndex = ZERO ; pdIndex < this->pd.fileCount ; )
            {
               //* Read a source line *
               ifs.getline ( tmp, gsMAXBYTES, NEWLINE ) ;
               gsPath = tmp ;

               //* If not a blank line AND not a comment *
               if ( (gsPath.gschars() > 1) && (*gsPath.gstr() != HASH) )
               {
                  this->pd.fileData[pdIndex].fPath = gsPath ;
                  this->spfExtractFilename (this->pd.fileData[pdIndex].fName, gsPath ) ;
                  ++pdIndex ;
               }
            }
         }
         ifs.close() ;                 // close the file
      }
      else if ( this->emCount < emMAX )
      {
         gString gs, gsName ;
         this->spfExtractFilename ( gsName, this->lstFile.ustr() ) ;
         gs.compose( badFileTemplate[bftListNotFound], gsName.ustr() ) ;
         gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
      }
   }
   else if ( this->emCount < emMAX )
   {
      gString gs, gsName ;
      this->spfExtractFilename ( gsName, this->lstFile.ustr() ) ;
      gs.compose( badFileTemplate[bftListNotFound], gsName.ustr() ) ;
      gs.copy( this->ErrorMsg[this->emCount++], minTERMCOLS ) ;
   }
   return this->pd.fileCount ;

}  //* End ValidateFileList()  *

//************************
//*    GetLocalTime      *
//************************
//******************************************************************************
//* Get the system timecode and convert it to localTime format                 *
//*                                                                            *
//* Input  : ft  : (by reference, initial values ignored)                      *
//*                on return, containss decoded date/time                      *
//*                                                                            *
//* Returns: true if successful, false if system call fails                    *
//******************************************************************************

bool SrcProf::GetLocalTime ( localTime& ft )
{
bool        success = false ;                // return value
  
   //* Get the system time code *
   if ( (ft.epoch = time ( NULL )) >= 0 )
   {
      //* Decode the system time *
      Tm    tm ;                             // Linux time structure
      if ( (localtime_r ( &ft.epoch, &tm )) != NULL )
      {
         //* Translate to localTime format *
         ft.day      = ZERO ;                // day-of-week is not initialized
         ft.date     = tm.tm_mday ;          // today's date
         ft.month    = tm.tm_mon + 1 ;       // month
         ft.year     = tm.tm_year + 1900 ;   // year
         ft.hours    = tm.tm_hour ;          // hour
         ft.minutes  = tm.tm_min ;           // minutes
         ft.seconds  = tm.tm_sec ;           // seconds
         success = true ;
      }
   }
   return success ;

}  //* End GetLocalTime() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

