//******************************************************************************
//* File       : TagAsf.cpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 16-Oct-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module supports parsing, display and modification of     *
//*              WMA (audio) and WMV (video) files. These files are            *
//*              encapsulated with an Advanced Systems Format (ASF) container. *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Notes on ASF formatting:                                                   *
//* ------------------------                                                   *
//* ASF (Advanced Systems Format, formerly Advanced Streaming Format or Active *
//* Streaming Format) is a Microsoft proprietary container format. Any use of  *
//* this container format without licensing will likely result in lawsuits     *
//* from the evil troll in Redmmond, WA. You have been warned.                 *
//*                                                                            *
//* For our purposes, we simply read the human-readable metadata, which so far *
//* as we know, is not illegal. However, Microsloth is without humor or        *
//* humanity when it comes to their patents and copyrights.                    *
//* This author was formerly a beta-tester for early versions of Windows(tm),  *
//* and it is difficult to express precisely how much we loathe Microsoft and  *
//* the horse it rode in on.                                                   *
//*                                                                            *
//* This code is based on ASF version 1.20.03 (Dec. 2004).                     *
//*                                                                            *
//* Each ASF object takes the format:                                          *
//* GUID: 128 bits (16 bytes)    This is the unique identifier for the object. *
//* Size:  64 bits ( 8 bytes)    24 + size of data in bytes.                   *
//* Data:                        'Size' - 24 bytes of data >= ZERO bytes.      *
//*                                                                            *
//* Numeric Format:                                                            *
//* ---------------                                                            *
//* A) All values are unsigned values.                                         *
//* B) All GUID values are stored as a sequence of three little-endian values  *
//*    followed by two big-endian values. This is just freaky-stupid i.e.      *
//*    the Microsloth way.                                                     *
//*    Example  : 75B22630-668E-11CF-A6D9-00AA0062CE6C                         *
//*    Stored as: 3026B275 - 8E66 - CF11 - A6D9 - 00AA0062CE6C                 *
//* C) 64-bit values (e.g. the object-size value)                              *
//* D) 32-bit values (e.g. the ECDO descriptor count and MO/MLO data size)     *
//* E) 16-bit values most data field byte counts are 16-bit values meaning     *
//*           that descriptor names and _most_ descriptor values will fit      *
//*           within a 64KiB buffer.                                           *
//*                                                                            *
//* Text Format:                                                               *
//* ------------                                                               *
//* Unless otherwise specified, human-readable text within the file will be    *
//* encoded as UTF-16 (little-endian, no byte-order mark), null terminated     *
//* unless otherwise specified. This is the Windoze standard for text data     *
//* because up until mid-2019 they seemed to believe that UTF-8 was a          *
//* Commie plot. Actually UTF-16 is considered to be a serious risk to         *
//* computer security, and Microsloth is slowly transitioning to UTF-8.        *
//*                                                                            *
//* Overall Structure:                                                         *
//* ------------------                                                         *
//* A) Each file begins with a top-level Header Object.                        *
//*    Only this Header Object may contain other ASF objects.                  *
//*    "Bibliographic data" are contained in a Content Description object.     *
//* B) A top-level Data Object follows the header.                             *
//*    This object contains the audio/video or other binary data.              *
//*    The format of these data are ignored in our context, and we blindly     *
//*    copy the audio streams unmodified.                                      *
//* C) An _optional_ Index Object (if present) must be the last object in      *
//*    the file. The format of these data are also ignored in our context.     *
//*                                                                            *
//* Header Object: (see asfHeader class)                                       *
//* ------------------------------------                                       *
//* 128 bits      GUID ASF_Header_Object                                       *
//*                    75B22630-668E-11CF-A6D9-00AA0062CE6C                    *
//*  64 bits      object size                                                  *
//*  32 bits      number of objects contained in header (not incl. this one)   *
//*   8 bits      reserved (always 0x01)                                       *
//*   8 bits      reserved (always 0x02)                                       *
//*                                                                            *
//* Content Description Object: (see asfConDesc class)                         *
//* --------------------------------------------------                         *
//* 128 bits      GUID ASF_Content_Description_Object                          *
//*                    75B22633-668E-11CF-A6D9-00AA0062CE6C                    *
//*  64 bits      object size (34 bytes minimum)                               *
//*  16 bits      Title length        (bytes)                                  *
//*  16 bits      Author length          "                                     *
//*  16 bits      Copyright length       "                                     *
//*  16 bits      Description length     "                                     *
//*  16 bits      Rating length          "                                     *
//*  varies       Title text          (UTF-16LE, no BOM)                       *
//*  varies       Author text            "                                     *
//*  varies       Copyright text         "                                     *
//*  varies       Description text       "                                     *
//*  varies       Rating text            "                                     *
//*       (optional, only one allowed)                                         *
//*                                                                            *
//* Extended Content Description Object: (see asfConDesc class)                *
//* -----------------------------------------------------------                *
//* 128 bits      GUID ASF_Extended_Content_Description_Object                 *
//*                    D2D0A440-E307-11D2-97F0-00A0C95EA850                    *
//*  64 bits      object size (26 bytes minimum)                               *
//*  16 bits      number of attached Content Descriptor objects (see below)    *
//*   varies      an array of Content Descriptor objects (see below)           *
//*       (optional, only one allowed)                                         *
//*                                                                            *
//* Content Descriptor:                                                        *
//* -------------------                                                        *
//* 16 bits       Descriptor-name length                                       *
//* varies        Descriptor Name (UTF-16 little-endian, no BOM)               *
//* 16 bits       Descriptor-value data type (see enum asfDataType)            *
//* 16 bits       Descriptor-value length                                      *
//* varies        Descriptor Value in the specified data format, UTF-16 text,  *
//*               numeric or binary data.                                      *
//*               (For a description of binary image descriptors,  )           *
//*               (please see the asfFormatBinaryPreamble() method.)           *
//*                                                                            *
//* Content Branding Object                                                    *
//* -----------------------                                                    *
//* 128 bits      GUID ASF_Content_Branding_Object                             *
//*  64 bits      object size (40 bytes minimum)                               *
//*  32 bits      banner image type                                            *
//*               00 00 00 00 == none                                          *
//*               01 00 00 00 == bitmap                                        *
//*               02 00 00 00 == JPEG                                          *
//*               03 00 00 00 == GIF                                           *
//*  32 bits      banner image data size (bytes)                               *
//*   varies      image data                                                   *
//*  32 bits      banner image URL length                                      *
//*   varies      banner image URL (ASCII)                                     *
//*  32 bits      copyright URL length                                         *
//*   varies      copyright URL (ASCII)                                        *
//*                                                                            *
//* Metadata Object                                                            *
//* ---------------                                                            *
//* 128 bits      GUID ASF_Metadata_Object                                     *
//*  64 bits      object size (greater than 26 bytes)                          *
//*  16 bits      number of attached descriptor records                        *
//*   varies      one or more descriptor records                               *
//*                                                                            *
//* Metadata-object descriptors                                                *
//* ---------------------------                                                *
//*  16 bits      reserved, must be 0x0000                                     *
//*  16 bits      stream number (0x0000==whole file, else 0x0001 - 0x007F)     *
//*  16 bits      name length                                                  *
//*  16 bits      data type (see enum asfDataType)                             *
//*  32 bits      data length (capped at 65,535, however the spec may be wrong)*
//*   varies      name (UTF-16LE, no BOM)                                      *
//*   varies      data                                                         *
//*                                                                            *
//* Metadata Library Object                                                    *
//* -----------------------                                                    *
//* 128 bits      GUID ASF_Metadata_Library_Object                             *
//*  64 bits      object size (at least 26 bytes)                              *
//*  16 bits      number of attached descriptor records                        *
//*   varies      zero or more descriptor records                              *
//*                                                                            *
//* Metadata-library-object descriptors                                        *
//* -----------------------------------                                        *
//*  16 bits      language-list index (index into Language List Object)        *
//*  16 bits      stream number (0x0000==whole file, else 0x0001 - 0x007F)     *
//*  16 bits      name length                                                  *
//*  16 bits      data type (see enum asfDataType)                             *
//*  32 bits      data length                                                  *
//*   varies      name (UTF-16LE, no BOM)                                      *
//*   varies      data                                                         *
//*                                                                            *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//* a) Currently only the CDO and ECDO objects are decoded/encoded.            *
//* b) Text fields in the MO, MLO) may be decoded in a future release.         *
//*    (Display and encoding of these fields would require major)              *
//*    (changes at the application level.                       )              *
//* c) Image data in the CBO object may be decoded in a future release.        *
//* See conditional compile flag: PARSE_EXTRA_METADATA                         *
//*                                                                            *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//* Certain ECDO descriptor names seem to be cannonical:                       *
//* (Names are with or without the "WM/" prefix, which seems )                 *
//* (to be an integral part of the Window(tm) Media player,  )                 *
//* (and by extension, the patented encoding tools.          )                 *
//* We scan for these common descriptor names. See the "afmb[]" array of       *
//* AsfFieldMap objects.                                                       *
//* See the full list at: <https://exiftool.org/TagNames/ASF.html>             *
//*                                                                            *
//* Other (non-canonical) descriptor names may be similar to tag names used    *
//* in other audio formats. For these, we scan for the substrings in the       *
//* "afmc[]" array.                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************


//****************
//* Header Files *
//****************
#include "Taggit.hpp"         // Taggit-class definitions and data,
                              // plus general definitions and NcDialogAPI definition
#include "TagDialog.hpp"      // for umwHeaders[][tfCOUNT] only

//**************************************
//* Set to non-zero for debugging only *
//**************************************
#define DEBUG_WMDATA (0)      // Debug WriteMetadata_ASF()
#define DEBUG_EMDATA (0)      // Debug ExtractMetadata_ASF()


//**************
//* Local data *
//**************

//* Parse CBO, MO and MLO objects   (1) *
//* Discard CBO, MO and MLO objects (0) *
#define PARSE_EXTRA_METADATA (0)

//* For some reason, the encoder often prepends the "WM/"  *
//* substring to the ECDO descriptor name. It is likely    *
//* that playback software looks for this prefix when      *
//* decoding metadata, but it is not necessary for display *
//* of the data within this application.                   *
//* Experimentally remove the prefix and subtract          *
//* 3 * sizeof(UTF-16 word) from 'nameSize' member.        *
#define STRIP_WM (0)
#if STRIP_WM != 0
const char* wmSubstring = "WM/" ;
#endif   // STRIP_WM

//* Types of metadata an ASF Container may hold.       *
//* These are defined in the ASF specification,        *
//* Revision: 01.20.03 December 2004                   *
//* (these constants declared 'extern' in TagAsf.hpp.) *
//* ASF_Header_Object *
const char* HEADER_GUID = "75B22630-668E-11CF-A6D9-00AA0062CE6C" ;
//* ASF_Content_Description_Object *
const char* CDO_GUID    = "75B22633-668E-11CF-A6D9-00AA0062CE6C" ;
//* ASF_Extended_Content_Description_Object *
const char* ECDO_GUID   = "D2D0A440-E307-11D2-97F0-00A0C95EA850" ;
//* ASF_Content_Branding_Object *
const char* CBO_GUID    = "2211B3FA-BD23-11D2-B4B7-00A0C955FC6E" ;
//* ASF_Metadata_Object *
const char* MO_GUID     = "C5F8CBEA-5BAF-4877-8467-AA8C44FA4CCA" ;
//* ASF_Metadata_Library_Object *
const char* MLO_GUID    = "44231C94-9498-49D1-A141-1D134E457054" ;

//* Binary-encoded sequences corresponding to the GUID above definitions.*
const uint8_t EncodedGUID[][asfGUID_BYTES] = 
{
   //* Content Description Object *
   { 0x33, 0x26, 0xB2, 0x75, 0x8E, 0x66, 0xCF, 0x11, 
     0xA6, 0xD9, 0x00, 0xAA, 0x00, 0x62, 0xCE, 0x6C },
   //* Extended Content Description Object *
   { 0x40, 0xA4, 0xD0, 0xD2, 0x07, 0xE3, 0xD2, 0x11, 
     0x97, 0xF0, 0x00, 0xA0, 0xC9, 0x5E, 0xA8, 0x50 },
   //* Content Branding Object *
   { 0xFA, 0xB3, 0x11, 0x22, 0x23, 0xBD, 0xD2, 0x11, 
     0xB7, 0xB4, 0x00, 0xA0, 0xC9, 0x55, 0xFC, 0x6E },
   //* Metadata Object *
   { 0xEA, 0xCB, 0xF8, 0xC5, 0xAF, 0x5B, 0x77, 0x48, 
     0x67, 0x84, 0xAA, 0x8C, 0x44, 0xFA, 0x4C, 0xCA },
   //* Metadata Library Object *
   { 0x94, 0x1C, 0x23, 0x44, 0x98, 0x94, 0xD1, 0x49,
     0x41, 0xA1, 0x1D, 0x13, 0x4E, 0x45, 0x70, 0x54 },
   //* Advanced Systems Format (ASF) Header *
   { 0x30, 0x26, 0xB2, 0x75, 0x8E, 0x66, 0xCF, 0x11, 
     0xA6, 0xD9, 0x00, 0xAA, 0x00, 0x62, 0xCE, 0x6C },
} ;

//* Display string which replaces non-image binary data *
//* found in ECDO descriptors.                          *
const char* NonImageBinary = "(non-image binary data)" ;

//* For display of source-data fields which cannot be mapped *
//* to a relevant display field, these characters are used   *
//* to delimit name/value pairs.                             *
const wchar_t DELIM_NAME  = L'名' ;
const wchar_t DELIM_VALUE = L'值' ;
const wchar_t DELIM_BOOL  = L'。' ;
const wchar_t DELIM_WORD  = L'：' ;
const wchar_t DELIM_DWORD = L'，' ;
const wchar_t DELIM_QWORD = L'；' ;

//* For a descriptor which contains an image, the *
//* 'descValue' field contains a preamble before  *
//* the actual image data.                        *
//* See notes in asfReadBinaryDescriptor().       *
const uint16_t PREAMBLE_BYTES = 29 ;

//* Size of dynamically-allocated input buffer *
const uint64_t KB64 = 0x010000 ;

class AsfFieldMap
{
   public:
   const wchar_t* const fName ;  // ASF field name or partial name (substring)
   const short fLen ;            // number of characters to compare
   const asfDataType fType ;     // data type of field contents
   const TagFields fIndex ;      // display-field index
} ;

//* Basic tag-field names in Content Description Object *
//* -- CDO tag field names are constant.                *
const short AFMA_COUNT = 5 ;     // number of pre-defined field names (CDO)
static AsfFieldMap afma[AFMA_COUNT] = 
{
   { L"TITLE",          5, asfdtUTF16, tfTit2 }, // Title
   { L"AUTHOR",         6, asfdtUTF16, tfTpe1 }, // Artist
   { L"COPYRIGHT",      9, asfdtUTF16, tfTcop }, // Copyright "YYYY xxx..."
   { L"DESCRIPTION",   11, asfdtUTF16, tfTxxx }, // Freeform comment
   { L"RATING",         6, asfdtUTF16, tfTsrc }, // Popularity rating:
                                                 // (data from  this field is 
                                                 //  saved to popMeter::popStar,
                                                 //  _not_ to tfTsrc field)
} ;


//* Tag-field names in Extended Content Description Object.*
//* This is a list of exact ECDO descriptor-name matches.  *
const short AFMB_COUNT = 15 ;
static AsfFieldMap afmb[AFMB_COUNT] = 
{
   { L"WM/Year",                  7, asfdtUTF16, tfTyer }, // year
   { L"WM/TrackNumber",          14, asfdtDWORD, tfTrck }, // track
   { L"WM/UniqueFileIdentifier", 23, asfdtUTF16, tfTofn }, // original filename
   { L"WM/Composer",             11, asfdtUTF16, tfTcom }, // composer
   { L"WM/Publisher",            12, asfdtUTF16, tfTpub }, // publisher, record label
   { L"WM/AlbumTitle",           13, asfdtUTF16, tfTalb }, // Album
   { L"WM/AlbumArtist",          14, asfdtUTF16, tfTope }, // original artist
   { L"WM/Provider",             11, asfdtUTF16, tfTown }, // file owner/licensee
   { L"WM/ProviderRating",       17, asfdtUTF16, tfTsrc }, // (data stored in Popularimeter)
   { L"WM/Genre",                 8, asfdtUTF16, tfTcon }, // genre
   { L"WM/ProviderStyle",        16, asfdtUTF16, tfTcon }, // genre
   { L"WM/AudioFileURL",         15, asfdtUTF16, tfTrso }, // radio station owner
   { L"WM/AudioSourceURL",       17, asfdtUTF16, tfTrsn }, // radio station name
   { L"WM/Lyrics",                9, asfdtUTF16, tfToly }, // original lyricist
   { L"WM/EncodingTime",         15, asfdtQWORD, tfTlen }, // playback time (mSec)

   //* Group of descriptors to be concatenated into the 'tfTflt' (File Type)  *
   //* field. See delimeter characters DELIM_NAME and DELIM_VALUE (above).    *
//   { L"WM/AuthorURL",            12, asfdtUTF16, tfTflt }, // 
//   { L"WM/MediaPrimaryClassID",  22, asfdtUTF16, tfTflt }, // 
//   { L"WMFSDKVersion",           13, asfdtUTF16, tfTflt }, // 
//   { L"WMFSDKNeeded",            12, asfdtUTF16, tfTflt }, // 
} ;

//* ECDO tag fields are user defined, but commonly      *
//* have several names that are similar or identical to *
//* tag names in other audio formats.                   *
const short AFMC_COUNT = 16 ;    // ECDO descriptor names (near-match)
static AsfFieldMap afmc[AFMC_COUNT] = 
{
   { L"ALBUM",          5, asfdtUTF16, tfTalb },   // Album
   { L"TRACK",          5, asfdtUTF16, tfTrck },   // Track
   { L"YEAR",           4, asfdtUTF16, tfTyer },   // Year recorded
   { L"GENRE",          5, asfdtUTF16, tfTcon },   // Genre
   { L"STYLE",          5, asfdtUTF16, tfTcon },   // Genre
   { L"PUBLISHER",      9, asfdtUTF16, tfTpub },   // Publisher, Record Label
   { L"COMPOSE",        7, asfdtUTF16, tfTcom },   // Composer(s)
   { L"LYRI",           4, asfdtUTF16, tfText },   // Lyricist/Lyrics
   { L"REMIX",          5, asfdtUTF16, tfTpe4 },   // Interpreted/Remixed
   { L"TIME",           4, asfdtUTF16, tfTlen },   // playback time
   { L"LENGTH",         6, asfdtUTF16, tfTlen },   // audio data size (bytes)
   { L"SIZE",           4, asfdtUTF16, tfTsiz },   //  "                 "
   { L"WEB",            3, asfdtUTF16, tfTrsn },   // Download site, internet radio station, etc.
   { L"NET",            3, asfdtUTF16, tfTrsn },   //  "                 "
   { L"PROVIDER",       8, asfdtUTF16, tfTrsn },   //  "                 "
//   { L"IsVBR",          5, asfdtBOOL,  tfT    },   // 
} ;

//* Descriptor names that may indicate an image descriptor *
static const short inCOUNT = 4 ;
static const char* const imageNames[inCOUNT] = 
{
   "WM/Picture",
   "Picture",
   "Image",
   "Photo",
} ;

//* Temporary storage for extracted image data *
class picList
{
   public:
   picList ( void )
   { this->reset () ; }

   void reset ( void )
   {
      *this->picPath = '\0' ;
      *this->mimType = '\0' ;
      this->picSize = 0 ;
      this->picType = 0x00 ;        // ("other")
   }

   char  picPath[gsMAXBYTES] ;      // filespec of image
   char  mimType[gsMAXBYTES] ;      // MIME type of image
   short picSize ;                  // size of image in bytes (<=64Kib)
   uint8_t picType ;                // picture-type code
} ;


//*************************
//* Non-member prototypes *
//*************************
static TagFields asfMapField ( const gString& fldName ) ;
static void asfFormatHeader ( const asfHeader& asfHdr, ofstream& ofs, ofstream& dbg ) ;
static short asfCreatePicList ( const EmbeddedImage* epPtr, picList*& plPtr ) ;
static void asfFormatBinaryPreamble ( asfConDesc& asfMeta, const picList* plPtr, 
                                      short dcIndx, ofstream& dbg ) ;
static bool asfRating2Popmeter ( const char* rating, popMeter& pop ) ;
static bool asfPopmeter2Rating ( popMeter& pop, gString& gsRating ) ;
static uint32_t asfDiscardDescriptor ( char *ibuff, uint32_t objSize, ifstream& ifs ) ;


//*************************
//*  ExtractMetadata_ASF  *
//*************************
//******************************************************************************
//* Read the media file and extract the metadata tag fields.                   *
//*                                                                            *
//* Input  : si  : index into list of source data                              *
//*                                                                            *
//* Returns: 'true' if valid ASF header,                                       *
//*          'false' invalid header record, file not found, access error, etc. *
//******************************************************************************

bool Taggit::ExtractMetadata_ASF ( short si )
{
   ofstream dbg ;

   #if DEBUG_EMDATA != 0
   const char* BadAsf = "Error: Invalid ASF/WMA format.\n" ;
   const char* noTagData =  "No WMA metadata available.\n" ;
   const short CNAME_COLS = 28 ; // output column alignment
   gString dbgFile( "%s/emd.txt", this->cfgOpt.appPath ) ;
   uint64_t srcBytes = ZERO ;    // number of source bytes read
   dbg.open( dbgFile.ustr(), 
                 ofstream::out | (si == ZERO ? ofstream::trunc : ofstream::app) ) ;
   if ( dbg.is_open() )
   {
      if ( si == ZERO )
         dbg << "Debug ExtractMetadata_ASF()\n"
                "---------------------------\n" ;
      dbg << "si(" << si << ") '" 
          << this->tData.sf[si].sfName << "'\n" << endl ;
   }
   #endif   // DEBUG_EMDATA

   char *ibuff = new char[KB64] ;// input buffer (64KiB dynamic allocation)
   short cnt ;                   // for validating header record
   uint64_t objSize ;            // size (bytes) of current object
   asfHeader   asfHdr ;          // ASF container header
   asfConDesc  asfMeta ;         // Content Desc. / Extended Content Desc. data
   gString gsOut, gstmp ;        // text formatting
   bool valid_hdr = false ;      // 'true' if file format verified
   bool status = false ;         // return value

   //* Open the source file *
   ifstream ifs( this->tData.sf[si].sfPath, ifstream::in ) ;
   if ( ifs.is_open() )
   {
      //* Read the ASF container header *
      ifs.read ( ibuff, asfASF_HDR_BYTES ) ;
      cnt = ifs.gcount() ;
      if ( cnt == asfASF_HDR_BYTES )
         valid_hdr = asfHdr.validateHeader ( ibuff ) ;

      #if DEBUG_EMDATA != 0
      //* Display the header information *
      gstmp.formatInt( asfHdr.size, 13, true ) ;
      gsOut.compose( "GUID: %s  (at 0x%06llX)\n"
                     "SIZE: %S\n"
                     "OBJS: %u\n"
                     "Res1: %hhd\n"
                     "Res2: %hhd   valid:%s\n",
                     asfHdr.guid, &srcBytes, gstmp.gstr(), &asfHdr.hdrobj,
                     &asfHdr.res1, &asfHdr.res2,
                     (char*)(valid_hdr ? "true" : "false")) ;
      dbg << gsOut.ustr() << endl ;
      srcBytes += cnt ;
      #endif   // DEBUG_EMDATA

      //* If source file format verified as ASF *
      if ( valid_hdr )
      {
         //* Validation of object header *
         ObjHdrType metaHdr ;
         //* Total number of objects in header (loop control) *
         uint32_t hdrObjs = asfHdr.hdrobj ;

         //* Scan the header objects and isolate    *
         //* 1) Content Description Object          *
         //* 2) Extended Content Description Object *
         while ( hdrObjs > ZERO )
         {
            ifs.read ( ibuff, (asfGUID_BYTES + asfSIZE_BYTES) ) ;

            #if DEBUG_EMDATA != 0
            srcBytes += ifs.gcount() ;
            #endif   // DEBUG_EMDATA

            if ( ifs.gcount() == (asfGUID_BYTES + asfSIZE_BYTES) )
            {
               if ( ((metaHdr = asfMeta.validateObjectHeader ( ibuff, objSize ))) != ohtNONE )
               {
                  //********************************************
                  //* If validated header is CDO, get tag data *
                  //********************************************
                  if ( metaHdr == ohtCDO )
                  {
                     //* There are five(5) 16-bit size value for the five     *
                     //* possible tag fields. Read and decode the size values.*
                     //*      (assumes that source bytes are available)       *
                     ifs.read( ibuff, (asfDESC_BYTES * 5) ) ;
                     #if DEBUG_EMDATA != 0
                     srcBytes += ifs.gcount() ;
                     #endif   // DEBUG_EMDATA
                     asfMeta.decodeTagSizes( ibuff ) ;

                     //* Read the tag fields *
                     if ( asfMeta.titleSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.titleSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.titleSize, gstmp ) ;
                        gstmp.copy( asfMeta.title, gsMAXBYTES ) ;
                     }
                     if ( asfMeta.authorSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.authorSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.authorSize, gstmp ) ;
                        gstmp.copy( asfMeta.author, gsMAXBYTES ) ;
                     }
                     if ( asfMeta.copyrtSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.copyrtSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.copyrtSize, gstmp ) ;
                        gstmp.copy( asfMeta.copyrt, gsMAXBYTES ) ;
                     }
                     if ( asfMeta.descSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.descSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.descSize, gstmp ) ;
                        gstmp.copy( asfMeta.desc, gsMAXBYTES ) ;
                     }
                     if ( asfMeta.ratingSize > ZERO )
                     {
                        ifs.read( ibuff, asfMeta.ratingSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.ratingSize, gstmp ) ;
                        gstmp.copy( asfMeta.rating, gsMAXBYTES ) ;
                     }

                     #if DEBUG_EMDATA != 0
                     //* Display the CDO information *
                     uint64_t gpos = srcBytes - ((asfGUID_BYTES + asfSIZE_BYTES)
                        + (2 * 5) + asfMeta.titleSize + asfMeta.authorSize 
                        + asfMeta.copyrtSize + asfMeta.descSize 
                        + asfMeta.ratingSize ) ;
                     gstmp.formatInt( asfMeta.cdoSize, 13, true ) ;
                     gsOut.compose( "CDO  GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n"
                                    "    Title: %02hu '%s'\n"
                                    "   Author: %02hu '%s'\n"
                                    "   Copyrt: %02hu '%s'\n"
                                    "     Desc: %02hu '%s'\n"
                                    "   Rating: %02hu '%s'\n",
                                    asfMeta.guid, &gpos, gstmp.gstr(), &asfMeta.cdoSize,
                                    &asfMeta.titleSize,  asfMeta.title,
                                    &asfMeta.authorSize, asfMeta.author,
                                    &asfMeta.copyrtSize, asfMeta.copyrt,
                                    &asfMeta.descSize,   asfMeta.desc,
                                    &asfMeta.ratingSize, asfMeta.rating ) ;
                     dbg << gsOut.ustr() << endl ;

                     //* Report the captured tag data *
                     if ( asfMeta.titleSize > ZERO )
                     {
                        gsOut = "Title " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        dbg << gsOut.ustr() << asfMeta.title << endl ;
                     }
                     if ( asfMeta.authorSize > ZERO )
                     {
                        gsOut = "Author " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        dbg << gsOut.ustr() << asfMeta.author << endl ;
                     }
                     if ( asfMeta.copyrtSize > ZERO )
                     {
                        gsOut = "Copyright " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        dbg << gsOut.ustr() << asfMeta.copyrt << endl ;
                     }
                     if ( asfMeta.descSize > ZERO )
                     {
                        gsOut = "Description " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        dbg << gsOut.ustr() << asfMeta.desc << endl ;
                     }
                     if ( asfMeta.ratingSize > ZERO )
                     {
                        gsOut = "Rating " ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        dbg << gsOut.ustr() << asfMeta.rating << endl ;
                     }
                     #endif   // DEBUG_EMDATA
                  }

                  //*****************************************************
                  //* If validated header is ECDO, get descriptor count *
                  //*     (assumes that source bytes are available)     *
                  //*****************************************************
                  else if ( metaHdr == ohtECDO )
                  {
                     ifs.read( ibuff, asfDESC_BYTES ) ;
                     asfMeta.descCount = asfMeta.decode16Bit ( ibuff ) ;

                     #if DEBUG_EMDATA != 0
                     //* Display the CDO information *
                     srcBytes += ifs.gcount() ;
                     uint64_t gpos = srcBytes - ((asfGUID_BYTES + asfSIZE_BYTES)
                              + asfDESC_BYTES) ;
                     gstmp.formatInt( asfMeta.ecdoSize, 13, true ) ;
                     gsOut.compose( "ECDO GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n"
                                    "     OBJS: %hu\n",
                                    asfMeta.eguid, &gpos, gstmp.gstr(), 
                                    &asfMeta.ecdoSize, &asfMeta.descCount ) ;
                     dbg << "\n" << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     //* Read the tag data.                 *
                     //* (do not overrun our storage array) *
                     for ( short descIndx = ZERO ; 
                           ((descIndx < asfMeta.descCount) &&
                            (descIndx < asfECD_MAX_COUNT)) ; ++descIndx )
                     {
                        //* Decode descriptor name length *
                        ifs.read( ibuff, asfDESC_BYTES ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.exDesc[descIndx].nameSize = asfMeta.decode16Bit ( ibuff ) ;

                        //* Decode descriptor name *
                        ifs.read( ibuff, asfMeta.exDesc[descIndx].nameSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.utf16Decode ( ibuff, asfMeta.exDesc[descIndx].nameSize, 
                                              gstmp ) ;
                        #if STRIP_WM != 0    //* Strip the "WM/" prefix *
                        if ( (gstmp.find( wmSubstring )) == ZERO )
                        {
                           gstmp.erase( wmSubstring ) ;
                           asfMeta.exDesc[descIndx].nameSize -= (3 * 2) ;
                        }
                        #endif   // STRIP_WM
                        gstmp.copy( asfMeta.exDesc[descIndx].descName, gsMAXBYTES ) ;

                        //* Decode descriptor value data type *
                        ifs.read( ibuff, asfDESC_BYTES ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.exDesc[descIndx].dataType = (asfDataType)asfMeta.decode16Bit ( ibuff ) ;
                        
                        //* Decode descriptor value length *
                        ifs.read( ibuff, asfDESC_BYTES ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        asfMeta.exDesc[descIndx].valueSize = asfMeta.decode16Bit ( ibuff ) ;

                        //* Decode descriptor value.             *
                        //* Descriptors guaranteed to be < 64Kib.*
                        ifs.read( ibuff, asfMeta.exDesc[descIndx].valueSize ) ;
                        #if DEBUG_EMDATA != 0
                        srcBytes += ifs.gcount() ;
                        #endif   // DEBUG_EMDATA
                        if ( asfMeta.exDesc[descIndx].dataType == asfdtUTF16 )
                           asfMeta.utf16Decode ( ibuff, 
                                      asfMeta.exDesc[descIndx].valueSize, gstmp ) ;

                        //* 8-bit byte array - determine type of source data *
                        else if ( asfMeta.exDesc[descIndx].dataType == asfdtBYTE )
                        {
                           //* Extract and save the image data *
                           if ( (this->asfReadBinaryDescriptor ( si, ibuff, 
                                                   asfMeta, descIndx, dbg )) )
                           {
                              #if DEBUG_EMDATA != 0
                              gstmp = asfMeta.exDesc[descIndx].descValue ;
                              #endif   // DEBUG_EMDATA
                           }
                           else     // not image data
                           {
                              //* The non-image binary data are not displayed.*
                              #if DEBUG_EMDATA != 0
                              gstmp = asfMeta.exDesc[descIndx].descValue ;
                              #endif   // DEBUG_EMDATA
                           }
                        }

                        //* Numeric values will be reported as ASCII decimal.  *
                        //* Common numeric values are for tracks, track number,*
                        //* runtime, etc.                                      *
                        else if ( (asfMeta.exDesc[descIndx].dataType == asfdtBOOL) ||
                                  (asfMeta.exDesc[descIndx].dataType == asfdtDWORD) )
                        {
                           uint32_t u32 = asfMeta.decode32Bit ( ibuff ) ;
                           gstmp.compose( "%u", &u32 ) ;
                        }
                        else if ( asfMeta.exDesc[descIndx].dataType == asfdtQWORD )
                        {
                           uint64_t u64 = asfMeta.decode64Bit ( ibuff ) ;
                           gstmp.compose( "%llu", &u64 ) ;
                        }
                        else if ( asfMeta.exDesc[descIndx].dataType == asfdtWORD )
                        {
                           uint16_t u16 = asfMeta.decode16Bit ( ibuff ) ;
                           gstmp.compose( "%hu", &u16 ) ;
                        }
                        gstmp.copy( asfMeta.exDesc[descIndx].descValue, gsMAXBYTES ) ;

                        #if DEBUG_EMDATA != 0
                        gsOut.compose( "%02hd) (%02hu) %s --- (type:%02hXh, %02hu) %s",
                                       &descIndx, 
                                       &asfMeta.exDesc[descIndx].nameSize,
                                       asfMeta.exDesc[descIndx].descName,
                                       &asfMeta.exDesc[descIndx].dataType,
                                       &asfMeta.exDesc[descIndx].valueSize,
                                       asfMeta.exDesc[descIndx].descValue ) ;
                        dbg << gsOut.ustr() << endl ;
                        #endif   // DEBUG_EMDATA
                     }

                     //* If some descriptors were not decoded due to local    *
                     //* buffer being filled. This is unlikely, but if it     *
                     //* happens, it will be reported in the debugging data   *
                     //* before the data are discarded.                       *
                     if ( asfMeta.descCount >= asfECD_MAX_COUNT )
                     {
                        #if DEBUG_EMDATA != 0
                        if ( dbg.is_open() )
                        {
                           dbg << "------------------------------------"
                                  "--------------------------\n"
                                  "CAUTION! asf.exDesc[] array filled. "
                                  "Some metadata may be lost." << endl ;
                        }
                        #endif   // DEBUG_EMDATA

                        //* Scan and discard remainder of ECDO object.*
                        uint16_t fieldsize ;
                        for ( short descIndx = asfECD_MAX_COUNT ;
                              descIndx < asfMeta.descCount ; ++descIndx )
                        {
                           ifs.read( ibuff, asfDESC_BYTES ) ;  // descName size
                           fieldsize = asfMeta.decode16Bit( ibuff ) ;
                           ifs.read( ibuff, fieldsize ) ;      // descName
                           #if DEBUG_EMDATA != 0
                           if ( dbg.is_open() )
                           {
                              gString gx ;
                              asfMeta.utf16Decode( ibuff, fieldsize, gx ) ;
                              gstmp.compose( "%2hd) Name Bytes :%3hu  '%s'\n", 
                                             &descIndx, &fieldsize, gx.ustr() ) ;
                              dbg << gstmp.ustr() ;
                           }
                           #endif   // DEBUG_EMDATA
                           ifs.read( ibuff, asfDESC_BYTES ) ;  // descValue datatype
                           ifs.read( ibuff, asfDESC_BYTES ) ;  // descValue size
                           fieldsize = asfMeta.decode16Bit( ibuff ) ;
                           ifs.read( ibuff, fieldsize ) ;      // descValue
                           #if DEBUG_EMDATA != 0
                           if ( dbg.is_open() )
                           {
                              gString gx ;
                              asfMeta.utf16Decode( ibuff, fieldsize, gx ) ;
                              gstmp.compose( "    Value Bytes:%4hu  '%s'", 
                                             &fieldsize, gx.ustr() ) ;
                              dbg << gstmp.ustr() << endl ;
                              if ( descIndx == (asfMeta.descCount - 1) )
                                 dbg << endl ;
                           }
                           #endif   // DEBUG_EMDATA
                        }
                        asfMeta.descCount = asfECD_MAX_COUNT ;
                     }

                     #if DEBUG_EMDATA != 0
                     //* Report the captured tag data *
                     for ( short descIndx = ZERO ; 
                              descIndx < asfMeta.descCount ; ++descIndx )
                     {
                        gsOut = asfMeta.exDesc[descIndx].descName ;
                        while ( gsOut.gschars() < CNAME_COLS )
                           gsOut.append( L'-' ) ;
                        gsOut.append( L' ' ) ;
                        dbg << gsOut.ustr() 
                            << asfMeta.exDesc[descIndx].descValue << endl ;
                     }
                     #endif   // DEBUG_EMDATA
                  }

                  // Programmer's Note: Currently we ignore the contents of 
                  // Content Branding Object, Metadata Object and 
                  // Metadata Library Object. If in future we find these may
                  // contain useful data, These hooks will get us started.
                  else if ( metaHdr == ohtCBO )
                  {
                     #if DEBUG_EMDATA != 0
                     //* Display the CBO information *
                     srcBytes += ifs.gcount() ;
                     uint64_t gpos = srcBytes - (asfGUID_BYTES + asfSIZE_BYTES) ;
                     gstmp.formatInt( asfMeta.cboSize, 13, true ) ;
                     gsOut.compose( "CBO  GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n",
                                    asfMeta.cboguid, &gpos, gstmp.gstr(), 
                                    &asfMeta.cboSize ) ;
                     dbg << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;
                     uint32_t bytesRead = asfDiscardDescriptor ( ibuff, objSize, ifs ) ;
                     #if DEBUG_EMDATA != 0
                     srcBytes += bytesRead ;
                     #endif   // DEBUG_EMDATA

                     if ( bytesRead != objSize )   // unexpected EOF
                     {
                        #if DEBUG_EMDATA != 0
                        if ( dbg.is_open() )
                           dbg << "\nERROR! Unexpected end-of-file "
                                  "(CBO object)" << endl ;
                        #endif   // DEBUG_EMDATA
   
                        break ;     // end of loop
                     }
                  }

                  else if ( metaHdr == ohtMO )
                  {
                     #if DEBUG_EMDATA != 0
                     //* Display the MO information *
                     srcBytes += ifs.gcount() ;
                     uint64_t gpos = srcBytes - (asfGUID_BYTES + asfSIZE_BYTES) ;
                     gstmp.formatInt( asfMeta.moSize, 13, true ) ;
                     gsOut.compose( "MO   GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n",
                                    asfMeta.moguid, &gpos, gstmp.gstr(), 
                                    &asfMeta.moSize ) ;
                     dbg << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;
                     uint32_t bytesRead = asfDiscardDescriptor ( ibuff, objSize, ifs ) ;
                     #if DEBUG_EMDATA != 0
                     srcBytes += bytesRead ;
                     #endif   // DEBUG_EMDATA

                     if ( bytesRead != objSize )   // unexpected EOF
                     {
                        #if DEBUG_EMDATA != 0
                        if ( dbg.is_open() )
                           dbg << "\nERROR! Unexpected end-of-file "
                                  "(MO object)" << endl ;
                        #endif   // DEBUG_EMDATA
   
                        break ;     // end of loop
                     }
                  }

                  else if ( metaHdr == ohtMLO )
                  {
                     #if DEBUG_EMDATA != 0
                     //* Display the MLO information *
                     srcBytes += ifs.gcount() ;
                     uint64_t gpos = srcBytes - (asfGUID_BYTES + asfSIZE_BYTES) ;
                     gstmp.formatInt( asfMeta.mloSize, 13, true ) ;
                     gsOut.compose( "MLO  GUID: %s  (at 0x%06llX)\n"
                                    "     SIZE: %S (%llXh)\n",
                                    asfMeta.mloguid, &gpos, gstmp.gstr(), 
                                    &asfMeta.mloSize ) ;
                     dbg << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;
                     uint32_t bytesRead = asfDiscardDescriptor ( ibuff, objSize, ifs ) ;
                     #if DEBUG_EMDATA != 0
                     srcBytes += bytesRead ;
                     #endif   // DEBUG_EMDATA

                     if ( bytesRead != objSize )   // unexpected EOF
                     {
                        #if DEBUG_EMDATA != 0
                        if ( dbg.is_open() )
                           dbg << "\nERROR! Unexpected end-of-file "
                                  "(MLO object)" << endl ;
                        #endif   // DEBUG_EMDATA
   
                        break ;     // end of loop
                     }
                  }
               }

               //* Discard contents of uninteresting object *
               else
               {
                  #if DEBUG_EMDATA != 0
                  uint64_t gpos = srcBytes - (asfGUID_BYTES + asfSIZE_BYTES) ;
                  gstmp.compose( "DISCARDED OBJECT: %s (size:%llX at 0x%06llX)", 
                                 asfMeta.xguid, &objSize, &gpos ) ;
                  dbg << gstmp.ustr() << endl ;
                  #endif   // DEBUG_EMDATA
                  objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;

                  uint32_t bytesRead = asfDiscardDescriptor ( ibuff, objSize, ifs ) ;
                  #if DEBUG_EMDATA != 0
                  srcBytes += bytesRead ;
                  #endif   // DEBUG_EMDATA
                  if ( bytesRead != objSize )   // unexpected EOF
                  {
                     #if DEBUG_EMDATA != 0
                     if ( dbg.is_open() )
                        dbg << "\nERROR! Unexpected end-of-file "
                               "(uninteresting object)" << endl ;
                     #endif   // DEBUG_EMDATA

                     break ;     // end of loop
                  }
               }
            }
            else     // unexpected EOF (unlikely)
            {
               dbg << "Unexpected End-Of-File! Metadata not found.\n" << endl ;
               break ;
            }
            --hdrObjs ;
         }     // while()
      }        // valid header
      #if DEBUG_EMDATA != 0
      else        // not a valid ASF container
         dbg << BadAsf << endl ;
      #endif   // DEBUG_EMDATA

      ifs.close() ;           // close the source file
   }

   //******************************************************
   //* Transfer the captured data to global data members. *
   //******************************************************
   if ( asfMeta.cdoVerified )
   {
      //* Title *
      if ( asfMeta.title[ZERO] != NULLCHAR )
      {
         gsOut = asfMeta.title ;
         gsOut.copy( this->tData.sf[si].sfTag.field[afma[0].fIndex], gsMAXCHARS ) ;
         status = true ;   // valid data
      }
      //* Artist *
      if ( asfMeta.author[ZERO] != NULLCHAR )
      {
         gsOut = asfMeta.author ;
         gsOut.copy( this->tData.sf[si].sfTag.field[afma[1].fIndex], gsMAXCHARS ) ;
         status = true ;   // valid data
      }
      //* Copyright *
      if ( asfMeta.copyrt[ZERO] != NULLCHAR )
      {
         gsOut = asfMeta.copyrt ;
         gsOut.copy( this->tData.sf[si].sfTag.field[afma[2].fIndex], gsMAXCHARS ) ;
         status = true ;   // valid data
      }
      //* Description *
      if ( asfMeta.desc[ZERO] != NULLCHAR )
      {
         gsOut = asfMeta.desc ;
         gsOut.copy( this->tData.sf[si].sfTag.field[afma[3].fIndex], gsMAXCHARS ) ;
         status = true ;   // valid data
      }
      //* Rating *
      if ( asfMeta.rating[ZERO] != NULLCHAR )
      {
         asfRating2Popmeter ( asfMeta.rating, this->tData.sf[si].sfTag.pop ) ;
         status = true ;   // valid data
      }
   }
   if ( asfMeta.ecdoVerified )
   {
      TagFields tfMap = tfCOUNT ;      // index into tagFields array
      for ( short indx = ZERO ; indx < asfMeta.descCount ; ++indx )
      {
         //* Compare the tag name with names of display fields, and *
         //* if a match, copy the tag value to the display field.   *
         //* If multiple tag names match the same display field,    *
         //* the first match will be reported.                      *
         gstmp = asfMeta.exDesc[indx].descName ;
         if ( ((tfMap = asfMapField ( gstmp )) != tfCOUNT) &&
              (this->tData.sf[si].sfTag.field[tfMap][ZERO] == NULLCHAR) &&
              !((tfMap == tfTsrc) && (this->tData.sf[si].sfTag.pop.popdata != false))
            )
         {
            //* Special case: convert ProviderRating to Popularimeter.*
            if ( tfMap == tfTsrc )
            {
               asfRating2Popmeter ( asfMeta.exDesc[indx].descValue, 
                                    this->tData.sf[si].sfTag.pop ) ;
            }
            else
            {
               gsOut = asfMeta.exDesc[indx].descValue ;
               gsOut.copy( this->tData.sf[si].sfTag.field[tfMap], gsMAXCHARS ) ;
            }
            status = true ;

            #if DEBUG_EMDATA != 0
            dbg << "ECDO MATCH : " << gstmp.ustr() << " --- " << gsOut.ustr() << endl ;
            #endif   // DEBUG_EMDATA
         }

         //* For tags that cannot be mapped to a relevant display field,   *
         //* we concatenate the field name/value into a little-used field  *
         //* "File Type" (tfTflt) used only by MPEG files. Display of this *
         //* field is disabled by default. When writing data to the target *
         //* file, these data will be reformatted and appended to the      *
         //* other records written to target file.                         *
         //* Note that numeric fields have been converted to ASCII numeric.*
         //* Note that fields containing binary data are not displayed.    *
         else
         {
            if ( asfMeta.exDesc[indx].dataType != asfdtBYTE )
            {
               gsOut = this->tData.sf[si].sfTag.field[tfTflt] ;
               gsOut.append( "%C%s%C%s", 
                             &DELIM_NAME, asfMeta.exDesc[indx].descName, 
                             &DELIM_VALUE, asfMeta.exDesc[indx].descValue ) ;

               //* For descriptors which will be converted back to integers *
               //* when written to target, set a flag to indicate data type.*
               if ( (asfMeta.exDesc[indx].dataType == asfdtBOOL)  ||
                    (asfMeta.exDesc[indx].dataType == asfdtWORD)  ||
                    (asfMeta.exDesc[indx].dataType == asfdtDWORD) ||
                    (asfMeta.exDesc[indx].dataType == asfdtQWORD) )
               {
                  short ix = gsOut.findlast( DELIM_VALUE ) ;
                  if ( asfMeta.exDesc[indx].dataType == asfdtBOOL )
                     gsOut.insert( DELIM_BOOL, ix ) ;
                  else if ( asfMeta.exDesc[indx].dataType == asfdtWORD )
                     gsOut.insert( DELIM_WORD, ix ) ;
                  else if ( asfMeta.exDesc[indx].dataType == asfdtDWORD )
                     gsOut.insert( DELIM_DWORD, ix ) ;
                  else if ( asfMeta.exDesc[indx].dataType == asfdtQWORD )
                     gsOut.insert( DELIM_QWORD, ix ) ;
               }
               gsOut.copy( this->tData.sf[si].sfTag.field[tfTflt], gsMAXBYTES ) ;
            }

            #if DEBUG_EMDATA != 0
            dbg << "ECDO CONCAT: " << asfMeta.exDesc[indx].descName 
                << " --- " << asfMeta.exDesc[indx].descValue << endl ;
            #endif   // DEBUG_EMDATA
         }
      }
      #if DEBUG_EMDATA != 0
      //* Display contents of concatenated field *
      if ( *this->tData.sf[si].sfTag.field[tfTflt] != NULLCHAR )
      {
         gsOut = this->tData.sf[si].sfTag.field[tfTflt] ;
         short indx = 1 ;
         while ( ( indx = gsOut.find( DELIM_NAME, indx )) > ZERO )
         { gsOut.insert( L'\n', indx ) ; indx += 2 ; }
         dbg << "Concatenated Field:\n-------------------\n" << gsOut.ustr() << endl ;
      }
      #endif   // DEBUG_EMDATA
   }

   delete [] ibuff ;             // release dynamic allocation

   #if DEBUG_EMDATA != 0
   if ( ! status )
      dbg << noTagData << endl ;
   dbg << endl ;
   dbg.close() ;                 // close the debugging file
   #endif   // DEBUG_EMDATA

   return status ;

}  //* End ExtractMetadata_ASF() *

//*************************
//*      asfMapField      *
//*************************
//******************************************************************************
//* Non-member method                                                          *
//* -----------------                                                          *
//* Given a user-defined field name captured from the Extended Content         *
//* Description Object (ECDO), determine whether:                              *
//* a) The descriptor name is a exact match to an element of the afmb[] array. *
//* b) The descriptor name contains a substring matching an element of the     *
//*    afmc[] array.                                                           *
//* c) If no match found, return 'tfTflt' where unmapped descriptors will be   *
//*    concatenated.                                                           *
//* d) If a binary descriptor (not an image), ?                                *
//*                                                                            *
//*                                                                            *
//* Input  : fldName : (by reference) tag-field name to be compared to         *
//*                    field-name list.                                        *
//*                                                                            *
//* Returns: member of enum TagFields which an index into the tagFields-class  *
//*          array of display fields                                           *
//*          returns tfCOUNT if no match found                                 *
//******************************************************************************

static TagFields asfMapField ( const gString& fldName )
{
   const short SOFFSET = 3 ;        // index after "WM/"
   TagFields tfMap = tfCOUNT ;      // return value

   for ( short indx = ZERO ; indx < AFMB_COUNT ; ++indx )
   {
      if ( ((fldName.compare( afmb[indx].fName )) == ZERO) || 
           ((fldName.compare( &afmb[indx].fName[SOFFSET])) == ZERO) )
      {
         tfMap = afmb[indx].fIndex ;
         break ;
      }
   }
   if ( tfMap == tfCOUNT )
   {
      for ( short indx = ZERO ; indx < AFMC_COUNT ; ++indx )
      {
         if ( (fldName.find( afmc[indx].fName )) >= ZERO )
         {
            tfMap = afmc[indx].fIndex ;
            break ;
         }
      }
   }

   return tfMap ;

}  //* End asfMapField() *

//**************************
//* asfReadImageDescriptor *
//**************************
//******************************************************************************
//* Called only by asfExtractMetadata().                                       *
//* Caller has determined that a descriptor within the ECDO object contains    *
//* binary (non-text) data.                                                    *
//* Determine whether the descriptor specifies an embedded image or non-image  *
//* data.                                                                      *
//*                                                                            *
//* Compare descriptor name with likely image descriptions.                    *
//* 1) If a match is found:                                                    *
//*    a) Extract the identifier byte, image byte count, reserved value and    *
//*       MIME type from input stream.                                         *
//*    b) Test the MIME type to determine whether it indicates an image.       *
//*    c) Index start of image data.                                           *
//*    d) Locate image-type identifier embedded within the image data.         *
//*       (not currently implemented).                                         *
//*    e) Create a node on the ePic list describing the image.                 *
//*    f) Write the image data to a temporary file.                            *
//*                                                                            *
//* 2) If not image data:                                                      *
//*    a) Copy an informational message into the 'descValue' field.            *
//*    b) Discard the binary data. (see notes below)                           *
//*                                                                            *
//*                                                                            *
//* Input  : si       : index of source file with focus                        *
//*          ibuff    : caller's input buffer containing any header data and   *
//*                     all of the descriptor's value data.                    *
//*          asfMeta  : (by reference) data already captured for descriptor    *
//*                                    and tools for decoding data             *
//*                     'exDesc' member :                                      *
//*                        descName  : descriptor name                         *
//*                        descValue : receives MIME type (for debugging only) *
//*                        valueLen  : number of bytes in 'ibuff'              *
//*          descIndx : index into asfMeta::exDesc array                       *
//*          dbg      : handle to open output file to hold debugging data      *
//*                                                                            *
//* Returns: 'true'  if image found and scanned                                *
//*          'false' if not image data                                         *
//*                  (informational message written to 'descValue')            *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* The structure for inserting an image into the ASF container is not well    *
//* documented. For this reason, we are forced to reverse engineer the         *
//* process from the sample data we have obtained.                             *
//*                                                                            *
//* The examples have 29 bytes of data following the 'dataType' field and      *
//* before the actual image data. These 29 bytes are consistent within our     *
//* test data. Because all example files use JPEG-encoded images, we do not    *
//* know if the preamble to GIF or Bitmap images is different.                 *
//*                                                                            *
//* a) offset 00: could be the code for image format: 03==JPEG                 *
//*               03 is also the id3v2 code for "cover (front)"                *
//* b) offset 01: 16-bit size of remaining data (29 bytes less than 'valueLen')*
//* c) offset 03: 16-bit reserved value (always 0x0000)                        *
//* d) offset 05: The remaining 24 bytes consist of a UTF-16LE(no BOM),        *
//*               null-terminated string that is the MIME type for the image.  *
//*               JPEG MIME type: "image/jpeg"                                 *
//*                                                                            *
//* This is followed by the image data, either JPEG or GIF, (or other format)  *
//* Note that this is based only example data. The specification seems too     *
//* quiet on the acceptable image formats.                                     *
//* Note that JPEG images begin with the sequence:                             *
//*    FF D8 FF E0 00 10 4A 46 49 46                                           *
//*                       J  F  I  F                                           *
//* Note that GIF images begin with the sequence:                              *
//*    47 49 46                                                                *
//*     G  I  F                                                                *
//* Note that BMP images begin with the sequence:                              *
//*    42 4D                                                                   *
//*     B  M                                                                   *
//*                                                                            *
//*   ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----  ----   *
//*                                                                            *
//* Non-image binary descriptor:                                               *
//* ----------------------------                                               *
//* If the descriptor contains non-image binary data, there is no place to     *
//* display it, and no convenient way to save it for re-integration during     *
//* write to the target file.                                                  *
//*                                                                            *
//* The most common non-image binary descriptor is "MCDI". These data are      *
//* not actually binary, but UTF-16 encoded ASCII-hex numeric data.            *
//* This is presumably a code used by the Millennium Compact Disc Industries.  *
//* MCDI is a compact-disc manufacturer in Vancouver. The meaning of this      *
//* code is unclear; however, its presence or absence has no affect on a file  *
//* which is not actually on a CD.                                             *
//* Example:                                                                   *
//*  11+96+7053+982A+B337+D01C+E3A7+10258+11BA3+13674+14D9C+170D8+1CAE4+2319F+ *
//*  29974+311BB+37521+3AF79+3F7AF                                             *
//* At a guess, these are probably sector numbers.                             *
//*                                                                            *
//******************************************************************************

bool Taggit::asfReadBinaryDescriptor ( short si, char* ibuff,
                                       asfConDesc& asfMeta, short descIndx, 
                                       ofstream& dbg )
{
   gString  gs( asfMeta.exDesc[descIndx].descName ), // text formatting
            gstmp ;
   uint16_t imageBytes,          // actual size of image data
            reserved ;           // reserved word s/b 0x0000
   uint8_t  imageType ;          // picture-type code (see pType[] array)
   bool isImage = false ;        // return value

   #if DEBUG_EMDATA != 0
   gstmp.compose( "BinaryDescriptor: descName: '%S' valueSize: %hu\n"
                  "Raw Hex: ", gs.gstr(), &asfMeta.exDesc[descIndx].valueSize ) ;
   for ( short i = ZERO ; i < 39 ; ++i )
      gstmp.append( "%02hhX ", &ibuff[i] ) ;
   dbg << gstmp.ustr() << endl ;
   #endif   // DEBUG_EMDATA

   //* Compare descriptor name with image-name options *
   for ( short i = ZERO ; i < inCOUNT ; ++i )
   {
      if ( (gs.find( imageNames[i] )) >= ZERO )
      { isImage = true ; break ; }
   }

   if ( isImage )
   {
      //* Extract identifier, byte count, reserved *
      //* value and MIME type (if any).            *
      imageType = ibuff[ZERO] ;
      imageBytes = asfMeta.decode16Bit ( &ibuff[1] ) ;
      reserved   = asfMeta.decode16Bit ( &ibuff[3] ) ;
      asfMeta.utf16Decode ( &ibuff[5], 24, gstmp ) ;

      if ( (imageBytes < asfMeta.exDesc[descIndx].valueSize) && 
           (reserved == ZERO) &&
           (((gstmp.find( "image" )) >= ZERO) ||
            ((gstmp.find( "jpeg" )) >= ZERO)  ||
            ((gstmp.find( "gif" )) >= ZERO))
         )
      {
         #if DEBUG_EMDATA != 0
         //* Copy MIME type to 'descValue' field (for debugging output only) *
         gstmp.copy( asfMeta.exDesc[descIndx].descValue, gsMAXBYTES ) ;
         #endif   // DEBUG_EMDATA

         //* Move the encoded image data to top of buffer *
         // Programmer's Note: This isn't technically necessary, but makes 
         // the following code conceptually cleaner, and clean code is happy code.
         for ( uint16_t trg = ZERO, src = PREAMBLE_BYTES ; trg < imageBytes ; ++trg, ++src )
            ibuff[trg] = ibuff[src] ;

         //* Create a node on the ePic linked list and transfer *
         //* the setup data.                                    *
         EmbeddedImage* eiPtr = this->tData.sf[si].sfTag.addPic() ;
         eiPtr->inUse = true ;                        // signal active node
         gstmp.copy( eiPtr->mp3img.mimType, gsMAXBYTES ) ; // MIME type text
         eiPtr->mp3img.picSize = imageBytes ;
         eiPtr->mp3img.picType = imageType ;
         if ( eiPtr->mp3img.picType < PIC_TYPES )
            gs = pType[this->cfgOpt.appLanguage][eiPtr->mp3img.picType] ;
         else
            gs = pType[this->cfgOpt.appLanguage][ZERO] ;
         gs.copy( eiPtr->mp3img.txtDesc, gsMAXBYTES ) ;

         //* Create a temporary file to hold the image. *
         //* Rename the temporary file so it can be     *
         //* read by an external image viewer.          *
         this->CreateTempname ( gs ) ;
         bool isgif = ((gstmp.find( "gif" )) >= ZERO ) ;
         gs.append( isgif ? ".gif" : ".jpg" ) ;
         gs.copy( eiPtr->mp3img.picPath, gsMAXBYTES ) ;

         #if DEBUG_EMDATA != 0
         gString gsdbg ;
         if ( dbg.is_open() )
         {
            gsdbg.compose( "\nASF Embedded Image\n"
                           "------------------\n"
                           "picPath  : %s\n"
                           "picType  : %02hhX\n"
                           "picSize  : %hu\n"
                           "mimType  : %s\n"
                           "txtDesc  : %s\n",
                           eiPtr->mp3img.picPath, &eiPtr->mp3img.picType,
                           &eiPtr->mp3img.picSize,
                           eiPtr->mp3img.mimType, 
                           eiPtr->mp3img.txtDesc
                         ) ;
            dbg << gsdbg.ustr() << endl ;
         }
         #endif   // DEBUG_EMDATA

         //* Write the image data to the temporary file *
         ofstream ofs( eiPtr->mp3img.picPath, (ofstream::out | ofstream::trunc) ) ;
         if ( ofs.is_open() )
         {
            ofs.write( ibuff, eiPtr->mp3img.picSize ) ;
            ofs.close() ;                          // close the file
         }  // (ofs.is_open())
      }
      else
         isImage = false ;
   }

   //* Non-image binary data *
   if ( ! isImage )
   {
      /* Non-image binary data descriptors are currently discarded. *
       * See notes above.                                           */
      gstmp = NonImageBinary ;
      gstmp.copy( asfMeta.exDesc[descIndx].descValue, gsMAXBYTES ) ;

      #if DEBUG_EMDATA != 0
      if ( (dbg.is_open()) &&
           #if STRIP_WM == 0
           ((gs.compare( "WM/MCDI" )) == ZERO) )
           #else
           ((gs.compare( "MCDI" )) == ZERO) )
           #endif // STRIP_WM
      {
         asfMeta.utf16Decode( ibuff, asfMeta.exDesc[descIndx].valueSize, gstmp ) ;
         dbg << "Decoded: '" << gstmp.ustr() << "'" << endl ;
      }
      #endif   // DEBUG_EMDATA
   }

   return isImage ;

}  //* End asfReadBinaryDescriptor() *

//*************************
//*   WriteMetadata_ASF   *
//*************************
//******************************************************************************
//* Write edited metadata to target file, or alternatively write only the      *
//* headers and audio data.                                                    *
//*                                                                            *
//* Input  : si       : index into list of source data                         *
//*          filter   : (member of enum WriteFilter)                           *
//*                     determines which metadata will be written to target    *
//*          preserve : (optional, 'false' by default)                         *
//*                     If specified, retain the backup file after the target  *
//*                     has been successfully written.                         *
//*                                                                            *
//* Returns: 'true' if file successfully written                               *
//*          'false' if: a) invalid source header                              *
//*                      b) unable to encode the data                          *
//*                      c) file not found, access error, out-of-storage-space,*
//*                         etc.                                               *
//******************************************************************************

bool Taggit::WriteMetadata_ASF ( short si, WriteFilter filter, bool preserve )
{
   char *ibuff = new char[KB64] ;// input buffer (64KiB dynamic allocation)
   asfHeader   asfHdr ;          // ASF container header
   asfConDesc  asfMeta ;         // Content Desc. / Extended Content Desc. data
   ObjHdrType  metaHdr ;         // Validation of object header
   uint64_t hdrObjs = ZERO ;     // Total number of objects in source header
   uint64_t objSize ;            // size (bytes) of current object
   uint64_t audioSize = ZERO ;   // non-header bytes written to target
   gString gs, gstmp ;           // data formatting
   ofstream dbg ;                // handle for debugging output
   bool status = true ;          // Return value

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   //* For debugging, do not overwrite the source file *
   static short anSequence = ZERO ;
   char altName[MAX_FNAME] = "" ;
   #if 1    // DEBUG - ENABLED BY DEFAULT
   gString gsdbg( "wmd%02hd.%02hd.wma", &si, &anSequence ) ;
   gsdbg.copy( altName, MAX_FNAME ) ;
   ++anSequence ;
   #endif   // DEBUG - ENABLED BY DEFAULT
   #endif   // DEBUG_WMDATA

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   gString dbgFile( "%s/wmd.txt", this->cfgOpt.appPath ) ;
   dbg.open( dbgFile.ustr(), 
                 ofstream::out | (si == ZERO ? ofstream::trunc : ofstream::app) ) ;
   if ( dbg.is_open() )
   {
      if ( si == ZERO )
         dbg << "Debug WriteMetadata_ASF()\n"
                "-------------------------\n" ;
      dbg << "si(" << si << ")  filter(" 
          << (filter == wfALL_DATA ? "wfALL_DATA" : 
              filter == wfALL_ACTDATA ? "wfALL_ACTDATA" :
              filter == wfALL_ACTIVE ? "wfALL_ACTIVE" : "wfAUDIO_ONLY")
          << ")  preserve(" << preserve << ")\n"
          << " sfName: '" << this->tData.sf[si].sfName << "'\n" ;
      if ( *altName != NULLCHAR )
          dbg << "altName: '" << altName << "'\n" ;
      dbg << endl ;
   }
   #endif   // DEBUG_WMDATA

   gString srcPath( this->tData.sf[si].sfPath ),      // source filespec
           bacPath( "%S~", srcPath.gstr() ),          // backup filespec
           trgPath = srcPath,                         // target filespec
           tmpPath ;                                  // temp storage filespec

   //* Create a temporary file to contain the file header information.*
   this->CreateTempname ( tmpPath ) ;

   //* If user has specified a different target filename,   *
   //* adjust the target filespec. Because the source file  *
   //* will not be modified, backup filename is unnecessary.*
   if ( this->tData.sf[si].sfMod != false )
   {
      short fnameIndex = (trgPath.findlast( L'/' )) + 1 ;
      trgPath.limitChars( fnameIndex ) ;
      trgPath.append( this->tData.sf[si].sfName ) ;
   }

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   //* If alternate target filename *
   if ( *altName != NULLCHAR )
   {
      short indx = ((trgPath.findlast( L'/')) + 1) ;
      trgPath.limitChars( indx ) ;
      trgPath.append( altName ) ;

      if ( dbg.is_open() )
         dbg << "         Source backup not needed." << endl ;
   }
   else
   #endif   // DEBUG_WMDATA
   {
      //* Rename the source file as a backup  *
      //* If backup already exists, delete it *
      short bac = OK ;
      fmFType fType ;
      if ( (this->TargetExists ( bacPath, fType )) != false )
      {
         bac = this->DeleteFile ( bacPath ) ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {  if ( bac == OK )
               dbg << "         Existing backup deleted." << endl ;
            else
               dbg << "         Existing backup delete error!" << endl ;
         }
         #endif                  // DEBUG_WMDATA
      }

      //* Rename the source file *
      if ( bac == OK )
      {
         bac = this->RenameFile ( srcPath, bacPath ) ;
         srcPath = bacPath ;  // original source no longer exists
      }
      if ( bac != OK )        // access error, cannot continue
      {
         status = false ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            dbg << "Rename to backup failed. Abort.\n" << endl ;
         }
         #endif                  // DEBUG_WMDATA
      }
   }

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   if ( dbg.is_open() )
   {
      dbg << "srcPath: '" << srcPath.ustr() << "'\n"
          << "bacPath: '" << bacPath.ustr() << "'\n"
          << "trgPath: '" << trgPath.ustr() << "'\n"
          << "tmpPath: '" << tmpPath.ustr() << "'\n" << endl ;
   }
   #endif                  // DEBUG_WMDATA

   //***************************************************
   //* If setup successful, open source and temp files *
   //***************************************************
   // Programmer's Note: Because the source file has already been validated,
   // we assume that the source data are valid.
   if ( status != false )
   {
      ifstream ifs( srcPath.ustr(), ifstream::in ) ;
      ofstream ofs( tmpPath.ustr(), ofstream::trunc ) ;
      if ( (ifs.is_open() != false) && (ofs.is_open() != false) )
      {
         //* Read the ASF container header *
         ifs.read ( ibuff, asfASF_HDR_BYTES ) ;
         asfHdr.validateHeader ( ibuff ) ;
         hdrObjs = asfHdr.hdrobj ;     // used for loop control (see below)

         #if DEBUG_WMDATA != 0
         //* Display the header information *
         gstmp.formatInt( asfHdr.size, 13, true ) ;
         gs.compose( "GUID: %s\n"
                     "SIZE: %S\n"
                     "OBJS: %u\n"
                     "Res1: %hhd\n"
                     "Res2: %hhd\n",
                     asfHdr.guid, gstmp.gstr(), &asfHdr.hdrobj,
                     &asfHdr.res1, &asfHdr.res2 ) ;
         dbg << gs.ustr() << endl ;
         #endif   // DEBUG_EMDATA

         //* Reset the header byte count to zero.*
         //* This is the byte-output accumulator.*
         asfHdr.size = ZERO ;
         //* Reset the header-object count.      *
         //* This is the output-object counter.  *
         asfHdr.hdrobj = ZERO ;

         //* Capture the display data to the asfMeta object. *
         this->asfPrescanMetadata ( si, filter, asfMeta, ibuff, dbg ) ;

         //* If CDO and/or ECDO data captured, format the *
         //* data passed by the data filter, and write    *
         //* the encoded data to the temp file. Number of *
         //* bytes written is added to header accumulator.*
         if ( asfMeta.cdoVerified || asfMeta.ecdoVerified )
         {
            //* Adjust header-object count *
            if ( asfMeta.cdoVerified )    ++asfHdr.hdrobj ;
            if ( asfMeta.ecdoVerified )   ++asfHdr.hdrobj ;
            asfHdr.size += this->asfFormatMetadata ( si, asfMeta, ofs, dbg ) ;
         }

         asfMeta.reset() ;             // reset our work area
         while ( hdrObjs > ZERO )
         {
            ifs.read ( ibuff, (asfGUID_BYTES + asfSIZE_BYTES) ) ;
            if ( ifs.gcount() == (asfGUID_BYTES + asfSIZE_BYTES) )
            {
               metaHdr = asfMeta.validateObjectHeader ( ibuff, objSize ) ;
               //* If an existing metadata header, discard it     *
               //* and do not include its size in the byte total. *
               //* For all other objects, copy the unmodified     *
               //* data to the temp file and add size to total.  *
               if ( (metaHdr != ohtCDO) && (metaHdr != ohtECDO) )
               {
                  ofs.write( ibuff, (asfGUID_BYTES + asfSIZE_BYTES) ) ;
                  asfHdr.size += (asfGUID_BYTES + asfSIZE_BYTES) ;
               }
               objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;
               uint64_t readBlock ;
               while ( objSize > ZERO )
               {
                  if ( objSize <= KB64 )
                  { readBlock = objSize ; objSize = ZERO ; }
                  else
                  { readBlock = KB64 ; objSize -= KB64 ; }
                  ifs.read( ibuff, readBlock ) ;
                  if ( (metaHdr != ohtCDO) && (metaHdr != ohtECDO) )
                  {
                     ofs.write( ibuff, readBlock ) ;
                     asfHdr.size += readBlock ;
                     ++asfHdr.hdrobj ;
                  }
               }
            }
            else     // unexpected EOF (unlikely)
            {
               dbg << "Unexpected End-Of-File! Metadata not found.\n" << endl ;
               break ;
            }
            --hdrObjs ;
         }     // while()

         //* Close the temporary file *
         //**************************************************
         //* Important Note: The index into the source file *
         //* now references the data following the header.  *
         //**************************************************
         ofs.close() ;

         //* Add the size of the file-header object itself.*
         asfHdr.size += asfASF_HDR_BYTES ;

         #if DEBUG_WMDATA!= 0
         if ( dbg.is_open() )
         {
            tnFName fStats ;
            this->GetFileStats ( tmpPath, fStats ) ;
            gstmp.compose( "\nVerify Header Size:\n-------------------"
                           "\nasfHdr.size:%06llX (%llu) vs. fStat.fBytes:%06llX (%llu) "
                           "(header s/b %hd bytes larger) asfHdr.hdrobj:%u", 
                           &asfHdr.size, &asfHdr.size, 
                           &fStats.fBytes, &fStats.fBytes, 
                           &asfASF_HDR_BYTES, &asfHdr.hdrobj ) ;
            dbg << gstmp.ustr() << endl ;
         }
         #endif   // DEBUG_WMDATA


         //* Open the target file *
         ofs.open( trgPath.ustr(), (ofstream::out | ofstream::trunc) ) ;
         if ( ofs.is_open() )
         {
            //* Format the ASF Header and write it to target *
            asfFormatHeader ( asfHdr, ofs, dbg ) ;

            //* Add the temp-file (metadata and non-metadata *
            //* header info) to target file.                 *
            ifstream tmp_ifs( tmpPath.ustr(), ifstream::in ) ;
            if ( tmp_ifs.is_open() )
            {
               tmp_ifs.read( ibuff, KB64 ) ;
               while ( tmp_ifs.gcount() > ZERO )
               {
                  ofs.write( ibuff, tmp_ifs.gcount() ) ;
                  tmp_ifs.read( ibuff, KB64 ) ;
               }
               tmp_ifs.close() ;
            }

            //* Append the audio-stream data to target *
            ifs.read( ibuff, KB64 ) ;
            while ( ifs.gcount() > ZERO )
            {
               ofs.write( ibuff, ifs.gcount() ) ;
               audioSize += ifs.gcount() ;
               ifs.read( ibuff, KB64 ) ;
            }
            ofs.close() ;     // close the target file
         }

         #if DEBUG_WMDATA!= 0
         if ( dbg.is_open() )
         {
            dbg << "Summary\n-------\n" ;
            //* Source-file size *
            tnFName fStats ;
            this->GetFileStats ( srcPath, fStats ) ;
            gstmp.formatInt( fStats.fBytes, 11, true ) ;
            gs.compose( "Source size: %S\n", gstmp.gstr() ) ;
            dbg << gs.ustr() ;

            //* Header's output byte accumulator. *
            //* (does not include the 30 bytes of *
            //* the ASF header itself)            *
            gstmp.formatInt( asfHdr.size, 11, true ) ;
            gs.compose( "asfHdr.size: %S\n", gstmp.gstr() ) ;
            dbg << gs.ustr() ;

            //* Number of byte of non-header data written (audio stream, etc.) *
            gstmp.formatInt( audioSize, 11, true ) ;
            gs.compose( "Audio data : %S\n", gstmp.gstr() ) ;
            dbg << gs.ustr() ;

            //* Calculated size of target file.               *
            //* Header object + header data + non-header data *
            fStats.fBytes = asfASF_HDR_BYTES + asfHdr.size + audioSize ;
            gstmp.formatInt( fStats.fBytes, 11, true ) ;
            gs.compose( "Target s/b : %S\n", gstmp.gstr() ) ;
            dbg << gs.ustr() ;

            //* Actual size of target file *
            this->GetFileStats ( trgPath, fStats ) ;
            gstmp.formatInt( fStats.fBytes, 11, true ) ;
            gs.compose( "Target size: %S\n", gstmp.gstr() ) ;
            dbg << gs.ustr() << endl ;
         }
         #endif   // DEBUG_WMDATA
      }
      if ( ifs.is_open() )    // close source file
         ifs.close() ;
      if ( ofs.is_open() )    // close target file
         ofs.close() ;
   }

   delete [] ibuff ;             // release dynamic allocation

   #if DEBUG_WMDATA != 0
   dbg << endl ;
   dbg.close() ;              // close the debugging file
   #endif   // DEBUG_WMDATA

   return status ;

}  //* End WriteMetadata_ASF() *

//*************************
//*  asfPrescanMetadata   *
//*************************
//******************************************************************************
//* Format the edited metadata in preparation for writing to target file.      *
//*                                                                            *
//* Input  : si     : index of source data to scan                             *
//*          filter : (member of enum WriteFilter)                             *
//*                   determines which metadata will be written to target      *
//*          asfMeta: (by reference) receives user-modified metadata           *
//*                   'cdoVerified' will be set if CDO data included           *
//*                   'ecdoVerified' will be set if ECDO data included         *
//*          ibuff  : caller's input buffer                                    *
//*          dbg    : handle for debugging output stream                       *
//*                  (file is opened by caller if debugging output is required)*
//*                                                                            *
//* Returns: 'true'  if formatted metadata returned                            *
//*          'false' if filter == wfAUDIO_ONLY (discard all metadata)          *
//******************************************************************************

bool Taggit::asfPrescanMetadata ( short si, WriteFilter filter, asfConDesc& 
                                  asfMeta, char* ibuff, ofstream& dbg )
{
   #define DEBUG_VERBOSE (0)        // additional debug info

   gString gs, gstmp ;              // text formatting
   bool hasData,                    // 'true' if field contains non-null data
        status = true ;             // return value - assume that at least 
                                    // one field contains metadata

   //* Initialize caller's data object *
   asfMeta.reset() ;

   //*******************************************************
   //* If audio only is to be saved, do not scan the data. *
   //*              NOTE THE EARLY RETURN                  *
   //*******************************************************
   if ( filter == wfAUDIO_ONLY )
      return false ;

   //***************************************************
   //* Scan the fields that may contain                *
   //* Content Description Object (CDO) or             *
   //* Extended Content Description Object (ECDO) data.*
   //***************************************************
   for ( short tfIndx = ZERO ; tfIndx < tfCOUNT ; ++tfIndx )
   {
      //* Apply the output filter:                                      *
      //* 1) wfALL_DATA:    All fields which contain data.              *
      //* 2) wfALL_ACTIVE:  All active fields (with or without data).   *
      //* 3) wfALL_ACTDATA: All active fields which contain data.       *
      //* 4) Special Case:  Active popularimeter (rating) data available*
      hasData = (bool)(this->tData.sf[si].sfTag.field[tfIndx][0] != NULLCHAR) ;
      if ( (filter == wfALL_DATA && hasData) ||
           (filter == wfALL_ACTIVE && this->tData.sffDisp[si]) ||
           (filter == wfALL_ACTDATA && this->tData.sffDisp[si] && hasData) ||
           (tfIndx == afma[4].fIndex && this->tData.sf[si].sfTag.pop.popdata )
         )
      {
         //***********************************
         //* Fixed-label fields (CDO object) *
         //***********************************
         if ( tfIndx == afma[0].fIndex )        // TITLE
         {
            gs = this->tData.sf[si].sfTag.field[afma[0].fIndex] ;
            gs.copy( asfMeta.title, gsMAXBYTES ) ;
            if ( *asfMeta.title != NULLCHAR )
               asfMeta.cdoVerified = true ; // indicate non-null data in CDO
         }
         else if ( tfIndx == afma[1].fIndex )   // ARTIST (author)
         {
            gs = this->tData.sf[si].sfTag.field[afma[1].fIndex] ;
            gs.copy( asfMeta.author, gsMAXBYTES ) ;
            if ( *asfMeta.author != NULLCHAR )
               asfMeta.cdoVerified = true ; // indicate non-null data in CDO
         }
         else if ( tfIndx == afma[2].fIndex )   // COPYRIGHT
         {
            gs = this->tData.sf[si].sfTag.field[afma[2].fIndex] ;
            gs.copy( asfMeta.copyrt, gsMAXBYTES ) ;
            if ( *asfMeta.copyrt != NULLCHAR )
               asfMeta.cdoVerified = true ; // indicate non-null data in CDO
         }
         else if ( tfIndx == afma[3].fIndex )   // DESCRIPTION
         {
            gs = this->tData.sf[si].sfTag.field[afma[3].fIndex] ;
            gs.copy( asfMeta.desc, gsMAXBYTES ) ;
            if ( *asfMeta.desc != NULLCHAR )
               asfMeta.cdoVerified = true ; // indicate non-null data in CDO
         }
         else if ( tfIndx == afma[4].fIndex )   // RATING
         {
            //* If the user has set the Popularimeter values,     *
            //* convert string data. Otherwise, leave field blank.*
            asfPopmeter2Rating ( this->tData.sf[si].sfTag.pop, gs ) ;
            gs.copy( asfMeta.rating, gsMAXBYTES ) ;
            if ( *asfMeta.rating != NULLCHAR )
               asfMeta.cdoVerified = true ; // indicate non-null data in CDO
         }

         //*******************************************
         //* User-defined-label fields (ECDO object) *
         //* (exclude field with concatenated items) *
         //*******************************************
         else if ( tfIndx != tfTflt )
         {
            if ( asfMeta.descCount < asfECD_MAX_COUNT )
            {
               short ix = ZERO ;
               for (  ; ix < AFMB_COUNT ; ++ix )
               {
                  if ( afmb[ix].fIndex == tfIndx )
                  {
                     //* Get descriptor name and value data type *
                     gs = afmb[ix].fName ;
                     asfMeta.exDesc[asfMeta.descCount].dataType = afmb[ix].fType ;
                     break ;
                  }
               }
               //* If display field does not map to a known ASF descriptor *
               //* name, use the display-field name. Assumes text data.    *
               if ( ix == AFMB_COUNT )
               {
                  gs = umwHeaders[this->cfgOpt.appLanguage][tfIndx] ;
                  gs.strip() ;
                  asfMeta.exDesc[asfMeta.descCount].dataType = asfdtUTF16 ;
               }
               gs.copy( asfMeta.exDesc[asfMeta.descCount].descName, gsMAXBYTES ) ;
               gs = this->tData.sf[si].sfTag.field[tfIndx] ;
               gs.copy( asfMeta.exDesc[asfMeta.descCount].descValue, gsMAXBYTES ) ;
               ++asfMeta.descCount ;
               asfMeta.ecdoVerified = true ; // indicate non-null data in ECDO
            }
         }
      }     // filter
   }        // for(;;)

   //**************************************************
   //* Unpack any miscelaneous data fields which are  *
   //* concatenated in the "File Type" (tfTflt) field.*
   //**************************************************
   gs = this->tData.sf[si].sfTag.field[tfTflt] ;
   short bindx = gs.after( DELIM_NAME ),
         eindx,
         igindx ;
   while ( (bindx >= ZERO) && (asfMeta.descCount < asfECD_MAX_COUNT) )
   {
      if ( (eindx = gs.find( DELIM_VALUE, bindx )) >= bindx )
      {
         asfMeta.ecdoVerified = true ; // indicate non-null data in ECDO
         gs.substr( gstmp, bindx, (eindx - bindx) ) ;

         //* Most descriptors in the concatenated field are UTF-16 text.   *
         //* However, some descriptors may have been numeric in the source *
         //* file, but were converted to text during extraction. These     *
         //* have been tagged for display according to data type. Set the  *
         //* specified data type and remove the data-type tag character.   *
         // Programmer's Note: This is not a very elegant way to track 
         // the data type. We continue to search for a better solution.
         if ( (igindx = gstmp.find( DELIM_BOOL )) > ZERO )
            asfMeta.exDesc[asfMeta.descCount].dataType = asfdtBOOL ;
         else if ( (igindx = gstmp.find( DELIM_WORD )) > ZERO )
            asfMeta.exDesc[asfMeta.descCount].dataType = asfdtWORD ;
         else if ( (igindx = gstmp.find( DELIM_DWORD )) > ZERO )
            asfMeta.exDesc[asfMeta.descCount].dataType = asfdtDWORD ;
         else if ( (igindx = gstmp.find( DELIM_QWORD )) > ZERO )
            asfMeta.exDesc[asfMeta.descCount].dataType = asfdtQWORD ;
         else
            asfMeta.exDesc[asfMeta.descCount].dataType = asfdtUTF16 ;
         if ( igindx > ZERO )
            gstmp.erase( igindx, 1 ) ;
         gstmp.copy( asfMeta.exDesc[asfMeta.descCount].descName, gsMAXBYTES ) ;

         bindx = eindx + 1 ;
         eindx = gs.find( DELIM_NAME, bindx ) ;
         if ( eindx < ZERO )     // if end-of-data
            eindx = gs.gschars() ;
         gs.substr( gstmp, bindx, (eindx - bindx) ) ;
         gstmp.copy( asfMeta.exDesc[asfMeta.descCount].descValue, gsMAXBYTES ) ;
         ++asfMeta.descCount ;
      }
      bindx = gs.after( DELIM_NAME, eindx ) ;
   }

   //***********************************************************
   //* Scan the source file for ECDO descriptors which contain *
   //* non-image binary data. This is an expensive operation,  *
   //* and we COULD simply discard the binary descriptors, but *
   //* in the interest of completeness we extract the data from*
   //* the source file so it may be inserted into the target.  *
   //***********************************************************
   #if DEBUG_WMDATA != 0 && DEBUG_VERBOSE != 0
   if ( dbg.is_open() )
   {
      dbg << "\nExtract Non-image Binary Data"
             "\n-----------------------------" << endl ;
   }
   #endif   // DEBUG_WMDATA && DEBUG_VERBOSE

   //* Open the source file *
   ifstream ifs( this->tData.sf[si].sfPath, ifstream::in ) ;
   if ( ifs.is_open() )
   {
      //* Read the ASF container header.       *
      //* (It is assumed that data are valid.) *
      asfHeader asfHdr ;            // ASF container header
      ifs.read ( ibuff, asfASF_HDR_BYTES ) ;
      asfHdr.validateHeader ( ibuff ) ;

      //* Locate the ECDO object *
      asfConDesc asfm ;
      uint64_t   objSize ;
      uint32_t   hdrObjs = asfHdr.hdrobj ;
      bool       isimg ;
      while ( hdrObjs > ZERO )
      {
         ifs.read ( ibuff, (asfGUID_BYTES + asfSIZE_BYTES) ) ;
         if ( ((asfm.validateObjectHeader ( ibuff, objSize ))) == ohtECDO )
         {
            ifs.read ( ibuff, asfDESC_BYTES ) ; // number of descriptors in ECDO
            asfm.descCount = asfm.decode16Bit ( ibuff ) ;

            //* Scan the ECDO descriptors for non-image binary data *
            for ( uint16_t dix = ZERO ; dix < asfm.descCount ; ++dix )
            {
               isimg = false ;      // set 'true' if type is binary image

               //* Decode:                      *
               //* a) descriptor-name length    *
               //* b) descriptor name           *
               //* c) descriptor-value data type*
               //* d) descriptor-value length.  *
               ifs.read( ibuff, asfDESC_BYTES ) ;
               asfm.exDesc[0].nameSize = asfm.decode16Bit ( ibuff ) ;
               ifs.read( ibuff, asfm.exDesc[0].nameSize ) ;
               asfm.utf16Decode ( ibuff, asfm.exDesc[0].nameSize, gstmp ) ;
               gstmp.copy( asfm.exDesc[0].descName, gsMAXBYTES ) ;
               ifs.read( ibuff, asfDESC_BYTES ) ;
               asfm.exDesc[0].dataType = 
                                    (asfDataType)asfm.decode16Bit ( ibuff ) ;
               ifs.read( ibuff, asfDESC_BYTES ) ;
               asfm.exDesc[0].valueSize = asfm.decode16Bit ( ibuff ) ;

               //* Read the descriptor value into the buffer.*
               //* (always <= KB64)                          *
               ifs.read( ibuff, asfm.exDesc[0].valueSize ) ;

               //* If the data type is asfdtBYTE, AND if not an image, *
               //* AND if data will fit in 'descValue', AND if caller's*
               //* 'exDesc[]' array is not full, save the entire       *
               //* descriptor to caller's object.                      *
               //* Otherwise, discard the descriptor.                  *
               // Programmer's Note: In the unlikely event that data are larger
               // than gsMAXBYTES, the data will be discarded. If this becomes
               // an issue, we will deal with it in a future release.
               if ( (asfm.exDesc[0].dataType == asfdtBYTE) &&
                    (asfMeta.descCount < asfECD_MAX_COUNT) )
               {
                  //* Compare descriptor name with image-name options *
                  for ( short i = ZERO ; i < inCOUNT ; ++i )
                  {
                     if ( (gstmp.find( imageNames[i] )) >= ZERO )
                     { isimg = true ; break ; }
                  }
                  if ( !isimg && 
                       (asfm.exDesc[0].valueSize <= (gsMAXBYTES - (asfDESC_BYTES * 2))) )
                  {
                     asfMeta.exDesc[asfMeta.descCount].nameSize = 
                                                   asfm.exDesc[0].nameSize ;
                     #if STRIP_WM != 0    //* Strip the "WM/" prefix *
                     if ( (gstmp.find( wmSubstring )) == ZERO )
                     {
                        gstmp.erase( wmSubstring ) ;
                        asfMeta.exDesc[asfMeta.descCount].nameSize -= (3 * 2) ;
                     }
                     #endif   // STRIP_WM

                     gstmp.copy( asfMeta.exDesc[asfMeta.descCount].descName, gsMAXBYTES ) ;
                     asfMeta.exDesc[asfMeta.descCount].dataType = 
                                                   asfm.exDesc[0].dataType ;
                     asfMeta.exDesc[asfMeta.descCount].valueSize = 
                                                   asfm.exDesc[0].valueSize ;
                     for ( short i = ZERO ; i < asfm.exDesc[0].valueSize ; ++i )
                        asfMeta.exDesc[asfMeta.descCount].descValue[i] = ibuff[i] ;
                     ++asfMeta.descCount ;

                     #if DEBUG_WMDATA != 0 && DEBUG_VERBOSE != 0
                     if ( dbg.is_open() )
                     {
                        uint16_t dix = asfMeta.descCount - 1 ;
                        gstmp.compose( "%02hu) nameSize:%hu dataType:%02hhX valueSize:%hu\n"
                                       "    descName : '%s'\n"
                                       "    descValue: ",
                                       &asfMeta.descCount, 
                                       &asfMeta.exDesc[dix].nameSize,
                                       &asfMeta.exDesc[dix].dataType,
                                       &asfMeta.exDesc[dix].valueSize,
                                       &asfMeta.exDesc[dix].descName ) ;
                        for ( short i = ZERO ; i < 33 ; ++i )
                        {
                           gstmp.append( "%02hhX", 
                                         &asfMeta.exDesc[dix].descValue[i] ) ;
                        }
                        dbg << gstmp.ustr() << endl ;
                     }
                     #endif   // DEBUG_WMDATA && DEBUG_VERBOSE
                  }
               }
            }
            #if DEBUG_WMDATA != 0 && DEBUG_VERBOSE != 0
            if ( dbg.is_open() )
            { dbg << endl ; }
            #endif   // DEBUG_WMDATA && DEBUG_VERBOSE

            break ;     // scan is complete
         }
         else
         {
            objSize -= (asfGUID_BYTES + asfSIZE_BYTES) ;
            asfDiscardDescriptor ( ibuff, objSize, ifs ) ;
         }
         --hdrObjs ;
      }     // hdrObjs
      ifs.close() ;     // close the source file
   }        // is_open()

   #if DEBUG_WMDATA != 0
   if ( dbg.is_open() )
   {
      short records = 5 + asfMeta.descCount ;
      gstmp.compose( "asfPrescanMetadata(%hd records)\n"
                     "---------------------------------\n", &records ) ;
      dbg << gstmp.ustr()
          << "Title ------------------------ " 
          << ((*asfMeta.title != '\0') ? asfMeta.title : "(no data)") << "\n"
          << "Author ----------------------- " 
          << ((*asfMeta.author != '\0') ? asfMeta.author : "(no data)") << "\n"
          << "Copyright -------------------- " 
          << ((*asfMeta.copyrt != '\0') ? asfMeta.copyrt : "(no data)") << "\n"
          << "Description ------------------ " 
          << ((*asfMeta.desc != '\0') ? asfMeta.desc : "(no data)") << "\n"
          << "Rating ----------------------- " 
          << ((*asfMeta.rating != '\0') ? asfMeta.rating : "(no data)") << "\n" ;

          for ( short i = ZERO ; i < asfMeta.descCount ; ++i )
          {
             gstmp.compose( "%s -", asfMeta.exDesc[i].descName ) ;
             while ( (gstmp.gscols()) < 30 )
                gstmp.append( L'-' ) ;
             gstmp.append( L' ' ) ;
             if ( asfMeta.exDesc[i].dataType != asfdtBYTE )
                dbg << gstmp.ustr() << asfMeta.exDesc[i].descValue << "\n" ;
             else
                dbg << gstmp.ustr() << "[BINARY DATA]\n" ;
          }

         //* Report any saved image files *
         if ( this->tData.sf[si].sfTag.ePic.inUse )
         {
            dbg << endl ;
            EmbeddedImage* epPtr = &this->tData.sf[si].sfTag.ePic ;
            for ( short picnum = 1 ; epPtr != NULL ; ++picnum )
            {
               gs.compose( "Image %02hd   : %s\n", &picnum, epPtr->mp3img.picPath ) ;
               gstmp.formatInt( epPtr->mp3img.picSize, 6, true ) ;
               dbg << gs.ustr()
                   << "Description: '" << epPtr->mp3img.txtDesc << "'\n"
                   << "MIME Type  : '" << epPtr->mp3img.mimType << "'\n"
                   << "Explanation: '" << epPtr->mp3img.picExpl << "'\n"
                   << "Bytes      : "  << gstmp.ustr() << "\n" ;
               epPtr = epPtr->next ;
            }
         }
         dbg << endl ;
   }
   #endif   // DEBUG_WMDATA

   return status ;

   #undef DEBUG_VERBOSE
}  //* End asfPrescanMetadata() *

//*************************
//*   asfFormatMetadata   *
//*************************
//******************************************************************************
//* Convert the user-modified tag data to ASF/WMA format and write the data    *
//* to the target file.                                                        *
//*                                                                            *
//* Input  : si     : index of source data to scan                             *
//*          asfMeta: (by reference) receives user-modified metadata           *
//*          ofs    : handle for open output file (temp file)                  *
//*          dbg    : handle for debugging output stream                       *
//*                  (file is opened by caller if debugging output is required)*
//*                                                                            *
//* Returns: number of bytes written to the target file                        *
//******************************************************************************

uint64_t Taggit::asfFormatMetadata ( short si, asfConDesc& asfMeta, 
                                     ofstream& ofs, ofstream& dbg )
{
   gString  gs, gstmp ;                   // text formatting
   uint8_t  guidTemp[asfGUID_BYTES+1] ;   // formatting buffer for GUID
   uint8_t  sizeTemp[asfSIZE_BYTES+1] ;   // formatting buffer for 64-bit size
   uint8_t  dataTemp[asfDESC_BYTES*5+1] ; // formatting for five(5) descriptor size values
   uint8_t  dcntTemp[asfDESC_BYTES+1] ;   // formatting buffer for 16-bit desc count
   uint64_t bytesWritten = ZERO ;         // bytes written to output (return value)

   //* For processing image information *
   picList  *pList = NULL ;               // pointer to temp-data array
   char     picBuff[gsMAXBYTES] ;         // buffer for transferring image data
   short    sizeSize = ZERO,              // bytes in five(5) encoded descriptor sizes
            picCount = ZERO,              // number of images attached to file
            picWrite = ZERO ;             // number of images written to output


   //**********************************************************
   //* Encode the CDO UTF-8 text fields to UTF-16LE (no BOM). *
   //**********************************************************
   gstmp = asfMeta.title ;    // Title
   asfMeta.titleSize = asfMeta.utf16Encode( asfMeta.title, gstmp ) ;
   gstmp = asfMeta.author ;   // Author (artist)
   asfMeta.authorSize = asfMeta.utf16Encode( asfMeta.author, gstmp ) ;
   gstmp = asfMeta.copyrt ;   // Copyright
   asfMeta.copyrtSize = asfMeta.utf16Encode( asfMeta.copyrt, gstmp ) ;
   gstmp = asfMeta.desc ;     // Description
   asfMeta.descSize = asfMeta.utf16Encode( asfMeta.desc, gstmp ) ;
   gstmp = asfMeta.rating ;   // Rating
   asfMeta.ratingSize = asfMeta.utf16Encode( asfMeta.rating, gstmp ) ;

   //******************************************************
   //* Encode the lengths of the CDO UTF-16 sequences     *
   //* above to an array of 16-bit, little-endian values. *
   //******************************************************
   sizeSize += asfMeta.encode16Bit( asfMeta.titleSize, (char*)&dataTemp[sizeSize] ) ;
   sizeSize += asfMeta.encode16Bit( asfMeta.authorSize, (char*)&dataTemp[sizeSize] ) ;
   sizeSize += asfMeta.encode16Bit( asfMeta.copyrtSize, (char*)&dataTemp[sizeSize] ) ;
   sizeSize += asfMeta.encode16Bit( asfMeta.descSize, (char*)&dataTemp[sizeSize] ) ;
   sizeSize += asfMeta.encode16Bit( asfMeta.ratingSize, (char*)&dataTemp[sizeSize] ) ;

   //***********************************************
   //* Encode the CDO GUID identifier and CDO size *
   //***********************************************
   gString gsGuid( CDO_GUID ) ;
   asfMeta.encodeGUID( gsGuid, guidTemp ) ;
   asfMeta.cdoSize = asfGUID_BYTES + asfSIZE_BYTES + sizeSize + 
                     asfMeta.titleSize + asfMeta.authorSize + 
                     asfMeta.copyrtSize + asfMeta.descSize + 
                     asfMeta.ratingSize ;
   asfMeta.encode64Bit( asfMeta.cdoSize, (char*)sizeTemp ) ;

   //* Append the resulting CDO byte streams to the output file.*
   ofs.write( (char*)guidTemp, asfGUID_BYTES ) ;
   ofs.write( (char*)sizeTemp, asfSIZE_BYTES ) ;
   bytesWritten += asfGUID_BYTES + asfSIZE_BYTES ;

   ofs.write( (char*)dataTemp, sizeSize ) ;
   bytesWritten += sizeSize ;

   ofs.write( asfMeta.title, asfMeta.titleSize ) ;
   bytesWritten += asfMeta.titleSize ;

   ofs.write( asfMeta.author, asfMeta.authorSize ) ;
   bytesWritten += asfMeta.authorSize ;

   ofs.write( asfMeta.copyrt, asfMeta.copyrtSize ) ;
   bytesWritten += asfMeta.copyrtSize ;

   ofs.write( asfMeta.desc, asfMeta.descSize ) ;
   bytesWritten += asfMeta.descSize ;

   ofs.write( asfMeta.rating, asfMeta.ratingSize ) ;
   bytesWritten += asfMeta.ratingSize ;

   //**********************************************************
   //* Encode the list of user-defined name/description pairs.*
   //**********************************************************
   uint16_t u16 ;
   uint32_t u32 ;
   uint64_t u64 ;
   for ( short i = ZERO ; i < asfMeta.descCount ; ++i )
   {
      //* Descriptor name and name length                *
      //* (UTF-8 length used to calculate UTF-16 length) *
      gstmp = asfMeta.exDesc[i].descName ;
      asfMeta.exDesc[i].nameSize = 
         asfMeta.utf16Encode( &asfMeta.exDesc[i].descName[asfDESC_BYTES], gstmp ) ;
      asfMeta.encode16Bit( asfMeta.exDesc[i].nameSize, asfMeta.exDesc[i].descName ) ;
      asfMeta.exDesc[i].nameSize += asfDESC_BYTES ;

      //* Data-type code, descriptor length and descriptor value *
      if ( asfMeta.exDesc[i].dataType == asfdtUTF16 )       // text data
      {
         gstmp = asfMeta.exDesc[i].descValue ;
         asfMeta.exDesc[i].valueSize = 
            asfMeta.utf16Encode( &asfMeta.exDesc[i].descValue[asfDESC_BYTES * 2], gstmp ) ;
      }
      else if ( (asfMeta.exDesc[i].dataType == asfdtBOOL) || // boolean integer (32-bit)
               (asfMeta.exDesc[i].dataType == asfdtDWORD) )  // 32-bit integer
      {
         gstmp = asfMeta.exDesc[i].descValue ;
         if ( (gstmp.gscanf( "%u", &u32 )) != 1 )
            u32 = ZERO ;      // conversion failed
         //* Allow boolean values of 0 or 1 only *
         if ( (asfMeta.exDesc[i].dataType == asfdtBOOL) && (u32 > 1) )
            u32 = 1 ;
         asfMeta.exDesc[i].valueSize = 
            asfMeta.encode32Bit( u32, &asfMeta.exDesc[i].descValue[asfDESC_BYTES * 2] ) ;
      }
      else if ( asfMeta.exDesc[i].dataType == asfdtQWORD )  // 64-bit integer
      {
         gstmp = asfMeta.exDesc[i].descValue ;
         if ( (gstmp.gscanf( "%llu", &u64 )) != 1 )
            u64 = ZERO ;      // conversion failed
         asfMeta.exDesc[i].valueSize = 
            asfMeta.encode64Bit( u64, &asfMeta.exDesc[i].descValue[asfDESC_BYTES * 2] ) ;
      }
      else if ( asfMeta.exDesc[i].dataType == asfdtWORD )   // 16-bit integer
      {
         gstmp = asfMeta.exDesc[i].descValue ;
         if ( (gstmp.gscanf( "%hu", &u16 )) != 1 )
            u16 = ZERO ;      // conversion failed
         asfMeta.exDesc[i].valueSize = 
            asfMeta.encode16Bit( u16, &asfMeta.exDesc[i].descValue[asfDESC_BYTES * 2] ) ;
      }
      else                                                  // binary data
      {
         //* Shift the binary data right by four(4) bytes *
         //* to make room for 'dataType' and 'valueSize'. *
         uint16_t srci = asfMeta.exDesc[i].valueSize - 1,
                  trgi = srci + (asfDESC_BYTES * 2) ;
         for ( uint16_t b = ZERO ; b < asfMeta.exDesc[i].valueSize ; ++b )
            asfMeta.exDesc[i].descValue[trgi--] = asfMeta.exDesc[i].descValue[srci--] ;
      }
      asfMeta.encode16Bit( asfMeta.exDesc[i].dataType, 
                           asfMeta.exDesc[i].descValue ) ;
      asfMeta.encode16Bit( asfMeta.exDesc[i].valueSize, 
                           &asfMeta.exDesc[i].descValue[asfDESC_BYTES] ) ;
      //* Add size of 'dataType' and 'valueSize'
      asfMeta.exDesc[i].valueSize += (asfDESC_BYTES * 2) ;
   }

   //* Test for stored images and gather information for each. *
   picCount = asfCreatePicList ( &this->tData.sf[si].sfTag.ePic, pList ) ;

   //* Integrate the image data into the descriptor array *
   if ( picCount > ZERO )
   {
      //* Short names for targets to unclutter the code *
      asfExDesc* edPtr ;   // pointer to target descriptor
      short dc ;        // descriptor index
      for ( short pc = ZERO ; pc < picCount ; ++pc )
      {
         if ( (dc = asfMeta.descCount) < asfECD_MAX_COUNT )
         {
            //* Set the descriptor name and name length *
            edPtr = &asfMeta.exDesc[dc] ;
            gstmp = imageNames[ZERO] ;
            edPtr->nameSize = 
                  asfMeta.utf16Encode( &edPtr->descName[asfDESC_BYTES], gstmp ) ;
            //* Reported 'nameSize' is the combined size of name and size    *
            //* value, while the _encoded_ size is the size of the name only.*
            edPtr->nameSize += asfMeta.encode16Bit( edPtr->nameSize, edPtr->descName ) ;

            //* Create the image-data preamble in 'descValue' field. *
            asfFormatBinaryPreamble ( asfMeta, &pList[pc], dc, dbg ) ;
            //* Binary data are written to the output file FOLLOWING *
            //* the data stored in 'descValue'. 'valueSize' contains *
            //* the TOTAL bytes for the field including image size.  *

            ++asfMeta.descCount ;      // increment descriptor count
         }
      }
   }


   //*************************************************************************
   //* Encode the ECDO GUID identifier, ECDO size and number of descriptors. *
   //*************************************************************************
   gsGuid = ECDO_GUID ;
   asfMeta.encodeGUID( gsGuid, guidTemp ) ;
   asfMeta.ecdoSize = asfGUID_BYTES ;

   //* Add sizes of descriptor name/value pairs *
   for ( short i = ZERO ; i < asfMeta.descCount ; ++i )
      asfMeta.ecdoSize += asfMeta.exDesc[i].nameSize + asfMeta.exDesc[i].valueSize ;
   asfMeta.ecdoSize += 
      asfMeta.encode16Bit ( asfMeta.descCount, (char*)dcntTemp ) ;
   asfMeta.ecdoSize += asfSIZE_BYTES ;    // size of 'ecdoSize' value
   asfMeta.encode64Bit ( asfMeta.ecdoSize, (char*)sizeTemp ) ;

   //* Append the resulting ECDO byte streams to the output file.*
   ofs.write( (char*)guidTemp, asfGUID_BYTES ) ;
   ofs.write( (char*)sizeTemp, asfSIZE_BYTES ) ;
   ofs.write( (char*)dcntTemp, asfDESC_BYTES ) ;
   bytesWritten += asfGUID_BYTES + asfSIZE_BYTES + asfDESC_BYTES ;

   for ( short i = ZERO ; i < asfMeta.descCount ; ++i )
   {
      ofs.write( asfMeta.exDesc[i].descName, asfMeta.exDesc[i].nameSize ) ;
      bytesWritten += asfMeta.exDesc[i].nameSize ;

      //* If descriptor contains UTF-16LE text _OR_ encoded integer data *
      if ( asfMeta.exDesc[i].dataType != asfdtBYTE )
      {
         ofs.write( asfMeta.exDesc[i].descValue, asfMeta.exDesc[i].valueSize ) ;
         bytesWritten += asfMeta.exDesc[i].valueSize ;
      }

      //* If descriptor contains binary data *
      else if ( asfMeta.exDesc[i].dataType == asfdtBYTE )
      {
         //* Test for an image preamble sequence.                           *
         //* The magic offset of '2' is location of encoded descriptor size.*
         //* The magic offset of '5' is location of encoded image size.     *
         uint16_t descSize = asfMeta.decode16Bit( &asfMeta.exDesc[i].descValue[2] ),
                  imgSize  = asfMeta.decode16Bit( &asfMeta.exDesc[i].descValue[5] ) ;

         //* If this descriptor controls an embedded *
         //* image append the image to output.       *
         // Programmer's Note: We assume that the image file exists AND 
         // that we have full access to it.
         if ( descSize == (imgSize + PREAMBLE_BYTES) )
         {
            //* Write the image preamble *
            ofs.write( asfMeta.exDesc[i].descValue, 
                       (asfDESC_BYTES + asfDESC_BYTES + PREAMBLE_BYTES) ) ;
            bytesWritten += (asfDESC_BYTES + asfDESC_BYTES + PREAMBLE_BYTES) ;

            //* Open the image file *
            ifstream img_ifs( pList[picWrite].picPath, ifstream::in ) ;
            if ( img_ifs.is_open() )
            {
               img_ifs.read( picBuff, gsMAXBYTES ) ;
               while ( img_ifs.gcount() > ZERO )
               {
                  ofs.write( picBuff, img_ifs.gcount() ) ;
                  img_ifs.read( picBuff, gsMAXBYTES ) ;
               }
               img_ifs.close() ;
               bytesWritten += pList[picWrite].picSize ;
               ++picWrite ;
            }
            #if DEBUG_WMDATA != 0
            else
            {
               if ( dbg.is_open() )
               { dbg << "ERROR! Unable to open image file.\n" << endl ; }
            }
            #endif   // DEBUG_WMDATA
         }

         //* For non-image binary data, write the binary stream *
         else     // (asfMeta.exDesc[i].dataType == asfdtBYTE)
         {
            //* Write the binary stream *
            ofs.write( asfMeta.exDesc[i].descValue, asfMeta.exDesc[i].valueSize ) ;
            bytesWritten += asfMeta.exDesc[i].valueSize ;
         }
      }
   }


   #if DEBUG_WMDATA != 0
   if ( dbg.is_open() )
   {
      uint8_t *u8Ptr ;
      const short MAXCOLS = 64 ;
      dbg << "CDO\n----------\n" ;
      gstmp.clear() ;
      for ( short i = ZERO ; i < asfGUID_BYTES ; ++i )               // GUID
         gstmp.append( "%02hhX ", &guidTemp[i] ) ;
      dbg << gstmp.ustr() << endl ;
      gstmp.compose( "TITLE  %04hX : ", &asfMeta.titleSize ) ;       // TITLE
      for ( short i = ZERO ; i < asfMeta.titleSize ; ++i )
      {
         //if ( i == 2 ) gstmp.append( L'-' ) ;
         gstmp.append( "%02hhX", &asfMeta.title[i] ) ;
      }
      gstmp.append( "\n         " ) ;
      u8Ptr = (uint8_t*)&asfMeta.titleSize ;
      gstmp.append( "%02hhX%02hhX-", &u8Ptr[0], &u8Ptr[1] ) ;
      for ( short i = ZERO ; i < asfMeta.titleSize ; i += 2 )
         gstmp.append( "%c   ", &asfMeta.title[i] ) ;
      dbg << gstmp.ustr() << endl ;

      gstmp.compose( "AUTHOR %04hX : ", &asfMeta.authorSize ) ;      // AUTHOR
      for ( short i = ZERO ; i < asfMeta.authorSize ; ++i )
      {
         gstmp.append( "%02hhX", &asfMeta.author[i] ) ;
      }
      gstmp.append( "\n         " ) ;
      u8Ptr = (uint8_t*)&asfMeta.authorSize ;
      gstmp.append( "%02hhX%02hhX-", &u8Ptr[0], &u8Ptr[1] ) ;
      for ( short i = ZERO ; i < asfMeta.authorSize ; i += 2 )
         gstmp.append( "%c   ", &asfMeta.author[i] ) ;
      dbg << gstmp.ustr() << endl ;

      gstmp.compose( "COPYRT %04hX : ", &asfMeta.copyrtSize ) ;      // COPYRIGHT
      for ( short i = ZERO ; i < asfMeta.copyrtSize ; ++i )
      {
         gstmp.append( "%02hhX", &asfMeta.copyrt[i] ) ;
      }
      gstmp.append( "\n         " ) ;
      u8Ptr = (uint8_t*)&asfMeta.copyrtSize ;
      gstmp.append( "%02hhX%02hhX-", &u8Ptr[0], &u8Ptr[1] ) ;
      for ( short i = ZERO ; i < asfMeta.copyrtSize ; i += 2 )
         gstmp.append( "%c   ", &asfMeta.copyrt[i] ) ;
      dbg << gstmp.ustr() << endl ;

      gstmp.compose( "DESCRI %04hX : ", &asfMeta.descSize ) ;     // DESCRIPTION
      for ( short i = ZERO ; i < asfMeta.descSize ; ++i )
      {
         gstmp.append( "%02hhX", &asfMeta.desc[i] ) ;
         if ( i > MAXCOLS ) break ;
      }
      gstmp.append( "\n         " ) ;
      u8Ptr = (uint8_t*)&asfMeta.descSize ;
      gstmp.append( "%02hhX%02hhX-", &u8Ptr[0], &u8Ptr[1] ) ;
      for ( short i = ZERO ; i < asfMeta.descSize ; i += 2 )
      {
         gstmp.append( "%c   ", &asfMeta.desc[i] ) ;
         if ( i > MAXCOLS ) break ;
      }
      dbg << gstmp.ustr() << endl ;

      gstmp.compose( "RATING %04hX : ", &asfMeta.ratingSize ) ;   // RATING
      for ( short i = ZERO ; i < asfMeta.ratingSize ; ++i )
      {
         gstmp.append( "%02hhX", &asfMeta.rating[i] ) ;
      }
      gstmp.append( "\n         " ) ;
      u8Ptr = (uint8_t*)&asfMeta.ratingSize ;
      gstmp.append( "%02hhX%02hhX-", &u8Ptr[0], &u8Ptr[1] ) ;
      for ( short i = ZERO ; i < asfMeta.ratingSize ; i += 2 )
         gstmp.append( "%c   ", &asfMeta.rating[i] ) ;
      dbg << gstmp.ustr() << endl ;

      dbg << "ECDO\n----------\n" ;
      gstmp.clear() ;
      for ( short i = ZERO ; i < asfMeta.descCount ; ++i )
      {
         gstmp = "NAME : " ;
         for ( short j = ZERO ; j < asfMeta.exDesc[i].nameSize ; ++j )
         {
            if ( j == 2 )
               gstmp.append( L'-' ) ;
            gstmp.append( "%02hhX", &asfMeta.exDesc[i].descName[j] ) ;
         }
         gstmp.append( "\n            " ) ;
         for ( short k = 2 ; k < asfMeta.exDesc[i].nameSize ; k += 2 )
            gstmp.append( "%c   ", &asfMeta.exDesc[i].descName[k] ) ;
         dbg << gstmp.ustr() << endl ;

         gstmp = "VALUE: " ;
         for ( short j = ZERO ; j < asfMeta.exDesc[i].valueSize ; ++j )
         {
            if ( j == 2 || j == 4 )
               gstmp.append( L'-' ) ;
            gstmp.append( "%02hhX", &asfMeta.exDesc[i].descValue[j] ) ;
            if ( (asfMeta.exDesc[i].dataType == asfdtBYTE) && (j > 39) ) break ;
            else if ( j > MAXCOLS ) break ;
         }
         if ( asfMeta.exDesc[i].dataType == asfdtUTF16 )
         {
            gstmp.append( "\n                 " ) ;
            for ( short k = 4 ; k < asfMeta.exDesc[i].valueSize ; k += 2 )
            {
               gstmp.append( "%c   ", &asfMeta.exDesc[i].descValue[k] ) ;
               if ( k > MAXCOLS ) break ;
            }
         }
         else if ( asfMeta.exDesc[i].dataType == asfdtBYTE )   // binary data
         {
            char achar = asfMeta.exDesc[i].descValue[9] ;
            if ( (achar >= 'a' && achar <= 'z') ||
                 (achar >= 'A' && achar <= 'Z') )
            {
               gstmp.append( "\n                           " ) ; 
               for ( short k = 9 ; k < 33 ; k += 2 )
               {
                  gstmp.append( "%c   ", &asfMeta.exDesc[i].descValue[k] ) ;
                  if ( k > MAXCOLS ) break ;
               }
            }
            else
               ;  // 'descValue' contains no text data
         }
         else if ( asfMeta.exDesc[i].dataType == asfdtQWORD )
         {
            uint64_t i64 = asfMeta.decode64Bit( &asfMeta.exDesc[i].descValue[asfDESC_BYTES * 2] ) ;
            gstmp.append( "\n                 %llu (0x%06llX)", &i64, &i64 ) ;
         }
         else if ( (asfMeta.exDesc[i].dataType == asfdtDWORD) ||
                   (asfMeta.exDesc[i].dataType == asfdtBOOL) )
         {
            uint32_t i32 = asfMeta.decode32Bit( &asfMeta.exDesc[i].descValue[asfDESC_BYTES * 2] ) ;
            if ( asfMeta.exDesc[i].dataType == asfdtBOOL )
               gstmp.append( "\n                 %u (%s)", &i32, (i32 == ZERO ? "false" : "true") ) ;
            else
               gstmp.append( "\n                 %u (0x%06X)", &i32, &i32 ) ;
         }
         else if ( asfMeta.exDesc[i].dataType == asfdtWORD )
         {
            uint16_t i16 = asfMeta.decode16Bit( &asfMeta.exDesc[i].descValue[asfDESC_BYTES * 2] ) ;
            gstmp.append( "\n                 %hu (0x%06hX)", &i16, &i16 ) ;
         }
         dbg << gstmp.ustr() << endl ;

         //* If this is binary descriptor, report *
         //* number of binary bytes written.      *
         if ( asfMeta.exDesc[i].dataType == asfdtBYTE )
         {
            uint16_t binbytes = asfMeta.exDesc[i].valueSize - (asfDESC_BYTES * 2) ;
            gs.formatInt( binbytes, 6, true ) ;
            gstmp.compose( "       %S bytes of binary data inserted.", gs.gstr() ) ;
            dbg << gstmp.ustr() << endl ;
         }
      }     // for(;;)
      gs.formatInt( bytesWritten, 11, true ) ;
      gstmp.compose( "\n*-------------------------"
                     "\n* bytesWritten: %S", gs.gstr() ) ;
      dbg << gstmp.ustr() << endl ;
   }
   #endif   // DEBUG_WMDATA

   if ( pList != NULL )          // release dynamic allocation
      delete [] pList ;

   return bytesWritten ;

}  //* End asfFormatMetadata() *

//*************************
//*    asfFormatHeader    *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Format the given ASF Header data and write it to the target file.          *
//* This is the first object written to target file.                           *
//*                                                                            *
//* Input  : asfHdr : (by reference) contains human-readable header info       *
//*          ofs    : handle for open output file (target file)                *
//*          dbg    : handle for debugging output stream                       *
//*                  (file is opened by caller if debugging output is required)*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void asfFormatHeader ( const asfHeader& asfHdr, ofstream& ofs, ofstream& dbg )
{
   uint8_t  guidTemp[asfGUID_BYTES+1] ;   // formatting buffer for GUID
   uint8_t  sizeTemp[asfSIZE_BYTES+1] ;   // formatting buffer for 64-bit size
   uint8_t  hobjTemp[asfHOBJ_BYTES+1] ;   // formatting buffer for 32-bit obj count
   const short resBYTES = 2 ;             // number of reserved bytes
   uint8_t resBytes[resBYTES] = { asfRES1, asfRES2 } ; // reserved bytes

   asfHdr.encodeGUID( guidTemp ) ;
// BUG! - SIZE OF FILE-HEADER OBJECT IS NOT INCLUDED!
   asfHdr.encode64Bit( asfHdr.size, (char*)sizeTemp ) ;
   asfHdr.encode32Bit( asfHdr.hdrobj, (char*)hobjTemp ) ;

   ofs.write( (char*)guidTemp, asfGUID_BYTES ) ;
   ofs.write( (char*)sizeTemp, asfSIZE_BYTES ) ;
   ofs.write( (char*)hobjTemp, asfHOBJ_BYTES ) ;
   ofs.write( (char*)resBytes, resBYTES ) ;

   #if DEBUG_WMDATA != 0
   if ( dbg.is_open() )
   {
      gString gs( "\nasfFormatHeader\n"
                  "---------------\n"
                  "GUID : " ) ;
      for ( short i = ZERO ; i < asfGUID_BYTES ; ++i )
         gs.append( "%02hhX ", &guidTemp[i] ) ;
      gs.append( "\nSIZE : " ) ;
      for ( short i = ZERO ; i < asfSIZE_BYTES ; ++i )
         gs.append( "%02hhX ", &sizeTemp[i] ) ;
      gs.append( "\nHOBJ : " ) ;
      for ( short i = ZERO ; i < asfHOBJ_BYTES ; ++i )
         gs.append( "%02hhX ", &hobjTemp[i] ) ;
      gs.append( "\nRESB : " ) ;
      for ( short i = ZERO ; i < resBYTES ; ++i )
         gs.append( "%02hhX ", &resBytes[i] ) ;
      dbg << gs.ustr() << "\n" << endl ;
   }
   #endif   // DEBUG_WMDATA

}  //* End asfFormatHeader()

//*************************
//*   asfCreatePicList    *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Given a linked list of EmbeddedImage objects, create an array of picList   *
//* objects. This consists of information on each image contained in the       *
//* source audio file.                                                         *
//*                                                                            *
//* Important Note: This method allocates dynamic memory. It is the caller's   *
//*                 responsibility to release the allocation.                  *
//*                                                                            *
//* Input  : epPtr : pointer to head of linked list                            *
//*          plPtr : (initially a NULL pointer)                                *
//*                  receives a pointer to dynamically-allocated array of      *
//*                  picList objects. (NULL pointer if epPtr list is empty)    *
//*                                                                            *
//* Returns: number of images identified (size of picList array)               *
//******************************************************************************

static short asfCreatePicList ( const EmbeddedImage* epPtr, picList*& plPtr )
{
   const EmbeddedImage *ep = epPtr ;   // for traversing the linked list
   gString gstmp ;                     // text formatting
   short   picCount ;                  // number of images identified (return value)

   if ( epPtr->inUse )                 // if one or more images in list
   {
      //* Count the number of images in the linked list.*
      for ( picCount = 0 ; ep != NULL ; ++picCount )
         ep = ep->next ;
      ep = epPtr ;               // reset the pointer

      //* Create the dynamic array *
      plPtr = new picList[picCount] ;

      //* Copy interesting data from linked list to array *
      for ( short pli = ZERO ; pli < picCount ; ++pli )
      {
         gstmp = ep->mp3img.picPath ;
         gstmp.copy( plPtr[pli].picPath, gsMAXBYTES ) ;
         gstmp = ep->mp3img.mimType ;
         gstmp.copy( plPtr[pli].mimType, gsMAXBYTES ) ;
         plPtr[pli].picSize = ep->mp3img.picSize ;
         plPtr[pli].picType = ep->mp3img.picType ;
      }
   }
   return picCount ;

}  //* End asfCreatePicList() *

//***************************
//* asfFormatBinaryPreamble *
//***************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* For an asfExDesc descriptor which will contain binary (image) data,        *
//* create the encoded image-data preamble.                                    *
//*                                                                            *
//* The preamble is the setup data that is located in the 'descValue' member   *
//* just before the actual image data. The preamble consists of:               *
//* OFFSET DATA                                                                *
//* ------ ---                                                                 *
//* 01-01  16-bit encoded data type (member of enum asfDataType i.e. asfdtBYTE)*
//* 02-03  16-bit encoded length of descriptor preamble PLUS size of image     *
//*                MINUS the size of the encoded 'dataType' and 'valueSize'    *
//*                words. (This encoded value == (valueSize - 4).              *
//* 04     8-bit  image code: These are the same values used for MPEG encoding.*
//*               See the "pType[]" array of picture types.                    *
//*               The most commonly used type is 0x03 ("Cover front")          *
//* 05-06  16-bit encoded image size (actual number of image-data bytes)       *
//* 07-08  16-bit reserved value (always 0x0000)                               *
//* 09-32         MIME type string encoded as UTF-16LE (no BOM)                *
//*                (padded with null values as necessary)                      *
//* 33            image data will be inserted at this point                    *
//*                                                                            *
//* Other members of target descriptor initialized:                            *
//* 'dataType'    receives the enum asfDataType value: asfdtBYTE               *
//* 'valueSize'   actual number of bytes encoded in 'descValue'                *
//*               This includes the data type, descriptor size value, preamble *
//*               size and the size of the binary image data.                  *
//*                                                                            *
//* Input  : asfMeta: (by reference) audio-file metadata                       *
//*          plPtr  : contains information about the image to be embedded      *
//*          dcIndx : index into the descriptor array                          *
//*          dbg    : handle for debugging output stream                       *
//*                  (file is opened by caller if debugging output is required)*
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void asfFormatBinaryPreamble ( asfConDesc& asfMeta, const picList* plPtr, 
                                      short dcIndx, ofstream& dbg )
{
   const char*    DFLT_MIME = "image/" ;        // generic MIME type
   const uint16_t RESERVED_VALUE = 0x0000 ;     // reserved value

   asfExDesc *exPtr = &asfMeta.exDesc[dcIndx] ; // point to target descriptor
   gString gstmp ;                              // text formatting
   short dvi = ZERO,                            // index into 'descValue' member
         dvisizeIndx,                           // offset for encoded 'descSize'
         mtBytes ;                              // number of bytes in encoded MIME type

   //* Encode data type at offset zero *
   exPtr->dataType = asfdtBYTE ;
   dvi += asfMeta.encode16Bit( exPtr->dataType, &exPtr->descValue[dvi] ) ;
   dvisizeIndx = dvi ;     // remember where the encoded descriptor size goes
   dvi += asfDESC_BYTES ;

   //* Insert picture-type code *
   exPtr->descValue[dvi++] = plPtr->picType ;

   //*  Encode the actual image size *
   dvi += asfMeta.encode16Bit( plPtr->picSize, &exPtr->descValue[dvi] ) ;

   //* Encode the reserved value *
   // Programmer's Note: We do this the hard way, just in case this value changes.
   dvi += asfMeta.encode16Bit( RESERVED_VALUE, &exPtr->descValue[dvi] ) ;

   //* Encode the MIME type if provided, else "image" *
   if ( *plPtr->mimType != NULLCHAR )
      gstmp = plPtr->mimType ;
   else
      gstmp = DFLT_MIME ;
   mtBytes = asfMeta.utf16Encode( &exPtr->descValue[dvi], gstmp ) ;

   //* Pad the encoded data to full preamble length *
   uint16_t encodedBytes = dvi + mtBytes ;
   while ( encodedBytes < (PREAMBLE_BYTES + (asfDESC_BYTES * 2)) )
      exPtr->descValue[encodedBytes++] = 0x00 ;

   //* Calculate and encode the full length of encoded data *
   exPtr->valueSize = PREAMBLE_BYTES + plPtr->picSize ;
   asfMeta.encode16Bit( exPtr->valueSize, &exPtr->descValue[dvisizeIndx] ) ;
   //* Add the bytes in data-type and descriptor-size (4 bytes, 2 words) *
   exPtr->valueSize += (asfDESC_BYTES * 2) ;

   #if DEBUG_WMDATA != 0
   if ( dbg.is_open() )
   {
      uint16_t sizeDiff = exPtr->valueSize - (asfDESC_BYTES * 2) - plPtr->picSize ;
      gString gsfmt( exPtr->valueSize, 6, true ) ;
      gstmp.compose( "dataType : %04hX\n" 
                     "valueSize: %S (0x%04X)\n", 
                     &exPtr->dataType, gsfmt.gstr(), &exPtr->valueSize ) ;
      dbg << gstmp.ustr() ;
      gsfmt.formatInt( plPtr->picSize, 6, true ) ;
      gstmp.compose( "picSize  : %S (0x%04hX)\n"
                     "sizeDiff : %hu (s/b %hu)\n"
                     "Encoded  : ",
                     gsfmt.gstr(), &plPtr->picSize, &sizeDiff, &PREAMBLE_BYTES ) ;
      for ( dvi = ZERO ; dvi < (PREAMBLE_BYTES + (asfDESC_BYTES * 2)) ; ++dvi )
      {
         if ( (dvi == 2) || (dvi == 4) )
            gstmp.append( L'-' ) ;
         gstmp.append( "%02hhX", &exPtr->descValue[dvi] ) ;
      }
      gstmp.append( "\n           ^--- ^--- ^-^---^---" ) ;
      for ( dvi = (asfDESC_BYTES * 4 + 1) ; 
            dvi < (PREAMBLE_BYTES + (asfDESC_BYTES * 2)) ; dvi += 2 )
         gstmp.append( "%c   ", &exPtr->descValue[dvi] ) ;
      dbg << gstmp.ustr() << endl ;
   }
   #endif   // DEBUG_WMDATA
}  //* End asfFormatBinaryPreamble() *

//*************************
//*  asfRating2Popmeter   *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Caller has identified a popularity rating in one of these fields:          *
//*   a) CDO 'rating' descriptor                                               *
//*   b) ECDO 'ProviderRating' descriptor                                      *
//* Convert the string data and initialize the Popularimeter fields.           *
//*                                                                            *
//* Input  : rating  : UTF-8 rating string                                     *
//*                    (should begin with ASCII integer which may be  )        *
//*                    (followed by an email address or other message.)        *
//*          pop     : (by reference)                                          *
//*                    'popStar' member receives rating (rating: 0-255)        *
//*                    'popEmail' member receives email addr. or other message *
//*                    'popdata' member set 'true'                             *
//*                                                                            *
//* Returns: state of 'popdata' flag                                           *
//******************************************************************************

static bool asfRating2Popmeter ( const char* rating, popMeter& pop )
{
   uint8_t rcode = ZERO ;
   gString gstmp = rating ;

   //* Convert the WMA x-of-10 rating to the MPEG x-of-255 rating.*
   //* If already in x-of-255 (or zero), do not convert.          *
   if ( (gstmp.gscanf( L"%hhu", &rcode )) > ZERO )
   {
      if ( (rcode > ZERO) && (rcode <= 10) )
         rcode = uint8_t((float)rcode / 10.0 * 255.0) ;
      pop.popStar = rcode ;

      //* If a message follows the star rating, capture it. *
      //* If this application wrote it, it might look like: *
      //*           "160/255 ☻☻☻½ - Cool tune!"             *
      //* Otherwise capture data following first space.     *
      short indx = gstmp.after( L" - " ) ;
      if ( indx < ZERO )
         indx = gstmp.after( L' ' ) ;
      if ( indx > ZERO )
         gstmp.substr( pop.popEmail, indx, gsMAXBYTES ) ;
      pop.popdata = true ;
   }
   return pop.popdata ;

}  //* End asfRating2Popmeter() *

//*************************
//*  asfPopmeter2Rating   *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* If the Popularimeter has been initialized, convert the data to a string    *
//* value.                                                                     *
//*                                                                            *
//* The "Popularimeter" (popularity rating) is an MPEG concept.                *
//* See "enum starRating" in TagMenu.cpp.                                      *
//*                                                                            *
//* Input  :  pop     : (by reference)                                         *
//*                     'popStar' member contains rating (0-255)               *
//*                     'popEmail' member optionally contains a message        *
//*                     'popdata' member indicates whether object contains data*
//*           gsRating: receives formatted rating string (cleared if no data)  *
//*                                                                            *
//* Returns: state of 'popdata' flag                                           *
//*          'true'  if Popularimeter data converted to rating                 *
//*          'false' if Popularimeter is empty (gsRating cleared)              *
//******************************************************************************

static bool asfPopmeter2Rating ( popMeter& pop, gString& gsRating )
{
   gsRating.clear() ;         // initialize caller's buffer

   if ( pop.popdata )
   {
      uint8_t popstar = pop.popStar ;

      gsRating.compose( "%hhu/255 %s", &popstar,
                        ((popstar == ZERO) ? "" :
                         (popstar >=   1 && popstar <=  25) ? "½" :
                         (popstar >=  26 && popstar <=  51) ? "☻" :
                         (popstar >=  52 && popstar <=  76) ? "☻½" :
                         (popstar >=  77 && popstar <= 102) ? "☻☻" :
                         (popstar >= 103 && popstar <= 127) ? "☻☻½" :
                         (popstar >= 128 && popstar <= 153) ? "☻☻☻" :
                         (popstar >= 154 && popstar <= 178) ? "☻☻☻½" :
                         (popstar >= 179 && popstar <= 204) ? "☻☻☻☻" :
                         (popstar >= 205 && popstar <= 229) ? "☻☻☻☻½" : "☻☻☻☻☻") ) ;
      if ( *pop.popEmail != NULLCHAR )
         gsRating.append( " - %s", pop.popEmail ) ;
   }
   return pop.popdata ;

}  //* End asfPopmeter2Rating() *

//*************************
//* asfDiscardDescriptor  *
//*************************
//******************************************************************************
//* Non-member Method                                                          *
//* -----------------                                                          *
//* Read specified number of bytes from the input stream. (data are not saved) *
//* Used to step over data from the input stream which will not be decoded.    *
//*                                                                            *
//* Input  : ibuff   : input buffer                                            *
//*          objSize : number of bytes to read                                 *
//*          ifs     : handle of open input stream                             *
//*                                                                            *
//* Returns: number of bytes actually read                                     *
//*          caller should compare this value with objSize                     *
//******************************************************************************

static uint32_t asfDiscardDescriptor ( char *ibuff, uint32_t objSize, ifstream& ifs )
{
   uint32_t readBlock,           // bytes to read each time through the loop
            bytesRead = ZERO ;   // total bytes read (return value)

   while ( objSize > ZERO )
   {
      if ( objSize <= KB64 )
      {
         readBlock = objSize ;
         objSize = ZERO ;
      }
      else
      {
         readBlock = KB64 ;
         objSize -= KB64 ;
      }
      ifs.read( ibuff, readBlock ) ;
      bytesRead += ifs.gcount() ;

      if ( ifs.gcount() < readBlock )  // unexpected end-of-file
         break ;
   }
   return bytesRead ;

}  //* End asfDiscardDescriptor() *

#undef DEBUG_WMDATA     // (definition belongs to this module only)
#undef DEBUG_EMDATA     // (definition belongs to this module only)

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

