//******************************************************************************
//* File       : TagConfig.cpp                                                 *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 16-Dec-2019                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module contains the methods related to reading the       *
//* configuration file and setting the application's working parameters.       *
//*                                                                            *
//*                                                                            *
//******************************************************************************
//* Version History (see Taggit.cpp).                                          *
//******************************************************************************


//****************
//* Header Files *
//****************
#include "Taggit.hpp"         // Taggit-class definitions and data
                              // plus general definitions and NcDialogAPI definition
#include <fcntl.h>            // for low-level file-descriptor operations
//#include <sys/wait.h>         // for the waitpid() function


//**************
//* Local data *
//**************
const wchar_t SLASH = L'/' ;
const wchar_t EQUAL = L'=' ;
//const wchar_t DOLLAR = L'$' ;

//* Comparison Strings *
// Usage Note: Not all members are used in all circumstances.
//       'str' used for each instance
//       'len' used for each instance (needed for comparisons)
//    'argcnt' used only in CfgParm[] : indicates number of args in attached array
//      'dflt' used only in CfgParm[] : indicates default member in attached array
//      'args' used only in CfgParm[] : attach point for argument array
class CfgComp
{
   public:
   const wchar_t* str ;          // pointer to command string
   short len ;                   // command-string length (not incl. NULLCHAR)
   short argcnt ;                // number of valid arguments for command
   short dflt ;                  // 'default' member of the argument array
   const CfgComp* args ;         // pointer to linked-list of arg descriptions
} ;

//* Names for accessing members of CfgParm[] array *
enum cciArguments : short 
{ 
   cciLA = ZERO,  // Language
   cciLO,   // Locale
   cciCS,   // ColorScheme
   cciEM,   // EnableMouse
   cciSO,   // SortOrder
   cciPS,   // PreserveSource
   cciSM,   // StripMetadata
   cciTE,   // TextEncoding
   //* 'Enable_(field)' Group *
   cciTIT2, cciTPE1, cciTALB, cciTRCK, cciTYER, cciTCON, cciTPE2, cciTPUB,
   cciTCOM, cciTEXT, cciTPE4, cciTCOP, cciTOWN, cciTXXX, cciTDAT, cciTIME,
   cciTRDA, cciTLEN, cciTSIZ, cciTBPM, cciTIT1, cciTIT3, cciTPE3, cciTPOS,
   cciTKEY, cciTLAN, cciTOPE, cciTOAL, cciTOFN, cciTOLY, cciTORY, cciTRSN,
   cciTRSO, cciTSRC, cciTSEE, cciTFLT, cciTDLY, cciTENC, cciTMED, 
   cciEC,   // EndConfigFile
   cciArgs  // (number of records in array)
} ;


static const CfgComp laArgs[] =           // 'Language' arguments
{  //* NOTE: These are in the same order as 'enum AppLang' *
   { L"English",    7, 0, 0, NULL },
   { L"Espanol",    7, 0, 0, NULL },
   { L"Zhongwen",   8, 0, 0, NULL },
   { L"TiengViet",  9, 0, 0, NULL },
   { L"locale",     6, 0, 0, NULL },      // (default)
} ;

static const CfgComp loArgs[] =           // 'Locale' arguments
{
   { L"locale",     6, 0, 0, NULL },      // (default)
   // Otherwise, a valid locale in the form: 'nn_NN.utf8'
} ;

static const CfgComp csArgs[] =           // 'ColorScheme' arguments
{
   { L"Default",   7, 0, 0, NULL },       // (default)
   { L"Black",     5, 0, 0, NULL },
   { L"Red",       3, 0, 0, NULL },
   { L"Green",     5, 0, 0, NULL },
   { L"Brown",     5, 0, 0, NULL },
   { L"Blue",      4, 0, 0, NULL },
   { L"Magenta",   7, 0, 0, NULL },
   { L"Cyan",      4, 0, 0, NULL },
   { L"Gray",      4, 0, 0, NULL },
} ;

const CfgComp emArgs[] =                  // 'EnableMouse' arguments
{
   { L"enable",  6, 0, 0, NULL },         // (default)
   { L"disable", 7, 0, 0, NULL } 
} ;

const CfgComp soArgs[] =                  // 'SortOrder' arguments
{
   { L"Filename",  8, 0, 0, NULL },       // (default)
   { L"Title",     5, 0, 0, NULL },
   { L"Track",     5, 0, 0, NULL },
   { L"Album",     5, 0, 0, NULL },
   { L"Artist",    6, 0, 0, NULL },
   { L"None",      4, 0, 0, NULL } 
} ;

static const CfgComp psArgs[] =           // 'PreserveSource' arguments
{
   { L"Preserve",  8, 0, 0, NULL },
   { L"Overwrite", 9, 0, 0, NULL } 
} ;

static const CfgComp smArgs[] =           // 'StripMetadata' arguments
{
   { L"Strip",     5, 0, 0, NULL },
   { L"Retain",    6, 0, 0, NULL } 
} ;

static const CfgComp teArgs[] =           // 'TextEncoding' arguments
{
   { L"ASCII_UTF16", 11, 0, 0, NULL },
   { L"UTF16",        5, 0, 0, NULL },
   { L"UTF16BE",      7, 0, 0, NULL }, 
   { L"UTF8",         4, 0, 0, NULL } 
} ;

static const CfgComp tfArgs[] =           // 'true/false' arguments
{
   { L"true",   4, 0, 0, NULL },
   { L"false",  5, 0, 0, NULL } 
} ;

static const CfgComp CfgParm[cciArgs] =   // Configuration Parameters
{
   { L"Language",              8,  5, 0, laArgs },
   { L"Locale",                6,  1, 0, loArgs },
   { L"ColorScheme",          11,  9, 0, csArgs },
   { L"EnableMouse",          11,  2, 0, emArgs },
   { L"SortOrder",             9,  6, 0, soArgs },
   { L"PreserveSource",       14,  2, 1, psArgs },
   { L"StripMetadata",        13,  2, 1, smArgs },
   { L"TextEncoding",         12,  4, 0, teArgs },
   { L"Enable_Title",         12,  2, 0, tfArgs }, // TIT2 (equivalent MP3 codes)
   { L"Enable_Artist",        13,  2, 0, tfArgs }, // TPE1
   { L"Enable_Album",         12,  2, 0, tfArgs }, // TALB
   { L"Enable_Track",         12,  2, 0, tfArgs }, // TRCK
   { L"Enable_Year",          11,  2, 0, tfArgs }, // TYER
   { L"Enable_Genre",         12,  2, 0, tfArgs }, // TCON
   { L"Enable_GuestArtist",   18,  2, 0, tfArgs }, // TPE2
   { L"Enable_RecordLabel",   18,  2, 0, tfArgs }, // TPUB
   { L"Enable_Composer",      15,  2, 0, tfArgs }, // TCOM
   { L"Enable_Lyricist",      15,  2, 0, tfArgs }, // TEXT
   { L"Enable_RemixedBy",     16,  2, 0, tfArgs }, // TPE4
   { L"Enable_Copyright",     16,  2, 0, tfArgs }, // TCOP
   { L"Enable_ContentOwner",  19,  2, 1, tfArgs }, // TOWN
   { L"Enable_Comment",       14,  2, 0, tfArgs }, // TXXX
   { L"Enable_RecDate",       14,  2, 1, tfArgs }, // TDAT
   { L"Enable_RecTime",       14,  2, 1, tfArgs }, // TIME
   { L"Enable_DateSupp",      15,  2, 1, tfArgs }, // TRDA
   { L"Enable_Length",        13,  2, 0, tfArgs }, // TLEN
   { L"Enable_Size",          11,  2, 0, tfArgs }, // TSIZ
   { L"Enable_BeatsPerMin",   18,  2, 1, tfArgs }, // TBPM
   { L"Enable_ContentGroup",  19,  2, 0, tfArgs }, // TIT1
   { L"Enable_Subtitle",      15,  2, 0, tfArgs }, // TIT3
   { L"Enable_Conductor",     16,  2, 0, tfArgs }, // TPE3
   { L"Enable_PartOfSet",     16,  2, 0, tfArgs }, // TPOS
   { L"Enable_InitialKey",    17,  2, 1, tfArgs }, // TKEY
   { L"Enable_Language",      15,  2, 0, tfArgs }, // TLAN
   { L"Enable_OrigArtist",    17,  2, 0, tfArgs }, // TOPE
   { L"Enable_OrigAlbum",     16,  2, 0, tfArgs }, // TOAL
   { L"Enable_OrigFilename",  19,  2, 0, tfArgs }, // TOFN
   { L"Enable_OrigLyricist",  19,  2, 0, tfArgs }, // TOLY
   { L"Enable_OrigYear",      15,  2, 0, tfArgs }, // TORY
   { L"Enable_NetStation",    17,  2, 0, tfArgs }, // TRSN
   { L"Enable_NetOwner",      15,  2, 0, tfArgs }, // TRSO
   { L"Enable_ISRC_Code",     16,  2, 1, tfArgs }, // TSRC
   { L"Enable_Settings",      15,  2, 1, tfArgs }, // TSEE
   { L"Enable_FileType",      15,  2, 1, tfArgs }, // TFLT
   { L"Enable_PlayDelay",     16,  2, 1, tfArgs }, // TDLY
   { L"Enable_EncodedBy",     16,  2, 1, tfArgs }, // TENC
   { L"Enable_MediaType",     16,  2, 1, tfArgs }, // TMED
   { L"EndConfigFile",        13,  1, 0, NULL   },
} ;                               

//*************************
//*    ReadConfigFile     *
//*************************
//******************************************************************************
//* Read the configuration file and initialize the associated member variables.*
//*                                                                            *
//*                                                                            *
//* Input  : altConfig: (optional, empty string by default)                    *
//*                     alternate path/filename for configuration file         *
//*          cfgAttr : color attribute for display of config options/arguments *
//*          goodAttr: color attribute for 'good' messages                     *
//*          badAttr : color attribute for display of syntax errors            *
//*          nameAttr: color attribute for display of config-file name         *
//*                                                                            *
//* Returns: OK  if valid configuration file                                   *
//*          ERR if not a valid config file, file contains serious syntax      *
//*              errors or if config file not found                            *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) Configuration file:                                                     *
//*    a) Default is 'Taggit.cfg' in application directory                     *
//*       Path: this->cfgOpt.appPath/Taggit.cfg                                *
//*    b) alternate path/filename specified on command line                    *
//*       passed in 'altConfig' parameter                                      *
//* 'cfgOpt' receives initial parameters from its instantiation                *
//*          receives secondary parameters from config file (if any)           *
//*          receives tertiary parameters from command-line options            *
//*          'tfPath' from system call                                         *
//*          'appPath' from argv[0]                                            *
//*          'srcPath' is either the CWD or alt path from command-line argument*
//*   language is from: a) environment, b) config file, c) comm-line           *
//*   cscheme is from: a) default, b) config file, c) comm-line                *
//*   mouse is from: a) default, b) config file, c) comm-line                  *
//*   sort is from: a) default, b) config file, c) comm-line(not impl)         *
//*   preserve src from: a) default(false), b) config file, c)comm-line(notimpl*
//*   strip meta from: a) default(false), b) config file, c) comm-line         *
//*   text encode from: a) config file only                                    *
//*   tag fields from: a) default(all), b) config file                         *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

short Taggit::ReadConfigFile ( const char* altConfig, attr_t cfgAttr, 
                               attr_t goodAttr, attr_t badAttr, attr_t nameAttr )
{
   gString cfgPath,           // config file path/filename
           cfgName,           // config file name
           gs, gsa, gsv ;     // work buffers
   short status = OK ;        // return value

   //* If alternate configuration file specified *
   if ( *altConfig != NULLCHAR )
   {
      char tmpPath[MAX_PATH] ;
      if ( (realpath ( altConfig, tmpPath )) != NULL )
      {
         cfgPath = tmpPath ;
         short fnOffset = (cfgPath.findlast( L'/' )) + 1 ;
         cfgName.compose( "\"%S\"", &cfgPath.gstr()[fnOffset] ) ;
      }
      else              // specified file not found
      {
         cfgName.compose( "\"%s\"", altConfig ) ;
         status = ERR ;
      }
   }
   //* Else use default configuration file *
   else
   {
      cfgPath.compose( "%s/%s", this->cfgOpt.appPath, ConfigTaggit ) ;
      cfgName.compose( "\"%s\"", ConfigTaggit ) ;
      fmFType ft ;
      if ( !(this->TargetExists ( cfgPath, ft )) )
         status = ERR ;
   }
   this->DiagMsg ( "Read configuration file: ", goodAttr, false ) ;
   this->DiagMsg ( cfgName.ustr(), nameAttr, bool(status == OK) ) ;
   if ( status != OK )
      this->DiagMsg ( "  File not found!", badAttr ) ;

   //* If config file exists *
   if ( status == OK )
   {
      char  lineData[gsMAXBYTES] ;  // raw UTF-8 input
      short readStatus,             // status of line read
            syntaxErrors = ZERO,    // count the syntax error found
            syntaxMax = 2,          // max syntax errors before abort
            lineNum = ZERO ;        // source-line counter
      bool  syntax = false,         // internal syntax-error flag
            done = false ;          // loop control

      //* Open the file *
      ifstream ifs ( cfgPath.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         //* Verify that file is a valid Taggit config file *
         ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
         if ( ifs.good() && (!(strncmp ( lineData, ConfigFileID, 5 ))) )
         {
            ++lineNum ;
         }
         else
         {
            this->DiagMsg ( "                         "
                            "Invalid configuration file format!", badAttr ) ;
            status = ERR ;
            done = true ;
         }

         //* Read the parameters and store values in member variables *
         const wchar_t* wPtr ;      // for scanning command string
         short ccIndex = ZERO,      // index into command array
               aaIndex,             // index into argument array
               aIndex ;             // index into command string
         while ( ! done )
         {
            if ( (readStatus = this->rcfReadSourceLine ( ifs, lineNum, gs )) > ZERO )
            {
               if ( this->suVerbose )
               {
                  gsv.compose( "LINE %3hd: %S", &lineNum, gs.gstr() ) ;
                  this->DiagMsg ( gsv.ustr(), cfgAttr ) ;
               }

               syntax = false ;        // reset the syntax flag

               //* Search for the config command in list of valid commands *
               for ( ccIndex = ZERO ; ccIndex < cciArgs ; ++ccIndex )
               {
                  if ( (gs.compare( CfgParm[ccIndex].str, true, 
                                    CfgParm[ccIndex].len )) == ZERO )
                  {
                     break ;
                  }
               }

               //* If a valid configuration command *
               if ( ccIndex < cciArgs )
               {
                  //* If end of config file reached *
                  if ( ccIndex == cciEC )
                  {
                     done = true ;
                  }

                  //* Else, index the command argument *
                  else
                  {
                     if ( (aIndex = gs.find( EQUAL )) >= ZERO )
                     {
                        wPtr = &gs.gstr()[aIndex] ;
                        aIndex = ZERO ;
                        while ( wPtr[++aIndex] == SPACE ) ;
                        gsa = &wPtr[aIndex] ;   // copy the argument

                        syntax = true ;      // expect the worst
                        for ( aaIndex = ZERO ; aaIndex < CfgParm[ccIndex].argcnt ; ++aaIndex )
                        {  //* (This test IS NOT case sensitive *
                           if ( (gsa.compare( CfgParm[ccIndex].args[aaIndex].str,
                                     false, CfgParm[ccIndex].args[aaIndex].len )) == ZERO )
                           {
                              syntax = false ;
                              break ;     // found a match
                           }
                        }

                        //* If a valid command argument was found set the  *
                        //* parameter. (Note that cciLO may have a valid   *
                        //* free-form argument.)                           *
                        if ( ! syntax || (ccIndex == cciLO) )
                        {
                           if ( ccIndex == cciLA )       // Language
                           {
                              switch ( aaIndex )
                              {
                                 case 0:  this->cfgOpt.appLanguage = enLang ;  break ; // English
                                 case 1:  this->cfgOpt.appLanguage = esLang ;  break ; // Espanol
                                 case 2:  this->cfgOpt.appLanguage = zhLang ;  break ; // Zhongwen
                                 case 3:  this->cfgOpt.appLanguage = viLang ;  break ; // TiengViet
                                 default: this->cfgOpt.appLanguage = locLang ; break ; // Default
                              } ;
                              
                           }
                           else if ( ccIndex == cciLO )  // Locale
                           {
                              if ( ! syntax )
                                 ;  // if default value, do nothing
                              else
                              {  //* If we arrive here, a non-default locale   *
                                 //* has been specified. Caller must verify it.*
                                 gsa.copy( this->cfgOpt.appLocale, MAX_FNAME ) ;
                                 syntax = false ;
                              }
                           }
                           else if ( ccIndex == cciCS )  // ColorScheme
                           {
                              switch ( aaIndex )
                              {
                                 case 1: this->cs.scheme = ncbcBK ;      break ; // Black
                                 case 2: this->cs.scheme = ncbcRE ;      break ; // Red
                                 case 3: this->cs.scheme = ncbcGR ;      break ; // Green
                                 case 4: this->cs.scheme = ncbcBR ;      break ; // Brown
                                 case 5: this->cs.scheme = ncbcBL ;      break ; // Blue
                                 case 6: this->cs.scheme = ncbcMA ;      break ; // Magenta
                                 case 7: this->cs.scheme = ncbcCY ;      break ; // Cyan
                                 case 8: this->cs.scheme = ncbcGY ;      break ; // Gray
                                 default: this->cs.scheme = ncbcCOLORS ; break ; // Default
                              } ;
                           }
                           else if ( ccIndex == cciEM )  // EnableMouse
                           {
                              this->cfgOpt.enableMouse = ((aaIndex == ZERO) ? true : false) ;
                           }
                           else if ( ccIndex == cciSO )  // SortOrder
                           {
                              switch ( aaIndex )
                              {
                                 case 1: this->cfgOpt.sortBy = sbTITLE ;   break ;
                                 case 2: this->cfgOpt.sortBy = sbTRACK ;   break ;
                                 case 3: this->cfgOpt.sortBy = sbALBUM ;   break ;
                                 case 4: this->cfgOpt.sortBy = sbARTIST ;  break ;
                                 case 5: this->cfgOpt.sortBy = sbNONE ;    break ;
                                 default: this->cfgOpt.sortBy = sbFNAME ;  break ;
                              } ;
                           }
                           else if ( ccIndex == cciPS )  // PreserveSource
                           {
                              if ( aaIndex == ZERO )     // 'Preserve'
                                 this->cfgOpt.preserve = true ;
                              else                       // 'Overwrite'
                                 this->cfgOpt.preserve = false ;
                           }
                           else if ( ccIndex == cciSM )  // StripMetadata
                           {
                              if ( aaIndex == ZERO )     // 'Strip'
                                 this->cfgOpt.ignore = true ;
                              else                       // 'Retain'
                                 this->cfgOpt.ignore = false ;
                           }
                           else if ( ccIndex == cciTE )  // TextEncoding
                           {
                              if ( aaIndex == ZERO )     // 'ASCII_UTF16'
                                 this->cfgOpt.txtEncode = ENCODE_ASCII ;
                              else if ( aaIndex == 1 )   // 'UTF16'
                                 this->cfgOpt.txtEncode = ENCODE_UTF16 ;
                              else if ( aaIndex == 2 )   // 'UTF16BE'
                                 this->cfgOpt.txtEncode = ENCODE_UTF16BE ;
                              else                       // 'UTF8'
                                 this->cfgOpt.txtEncode = ENCODE_UTF8 ;
                           }
                           //* Else, field is one of the "Enable_xxx" group.*
                           else
                           {
                              bool state = (aaIndex == ZERO) ? true : false ;
                              switch ( ccIndex )
                              {
                                 case cciTIT2:     // Title
                                    //* Ignore parameter and force field to display *
                                    this->tData.sffDisp[tfTit2] = true ;
                                    break ;
                                 case cciTPE1:     // Artist
                                    this->tData.sffDisp[tfTpe1] = state ;
                                    break ;
                                 case cciTALB:     // Album
                                    this->tData.sffDisp[tfTalb] = state ;
                                    break ;
                                 case cciTRCK:     // Track
                                    this->tData.sffDisp[tfTrck] = state ;
                                    break ;
                                 case cciTYER:     // Year
                                    this->tData.sffDisp[tfTyer] = state ;
                                    break ;
                                 case cciTCON:     // Genre
                                    this->tData.sffDisp[tfTcon] = state ;
                                    break ;
                                 case cciTPE2:     // GuestArtist
                                    this->tData.sffDisp[tfTpe2] = state ;
                                    break ;
                                 case cciTPUB:     // RecordLabel
                                    this->tData.sffDisp[tfTpub] = state ;
                                    break ;
                                 case cciTCOM:     // Composer
                                    this->tData.sffDisp[tfTcom] = state ;
                                    break ;
                                 case cciTEXT:     // Lyricist
                                    this->tData.sffDisp[tfText] = state ;
                                    break ;
                                 case cciTPE4:     // RemixedBy
                                    this->tData.sffDisp[tfTpe4] = state ;
                                    break ;
                                 case cciTCOP:     // Copyright
                                    this->tData.sffDisp[tfTcop] = state ;
                                    break ;
                                 case cciTOWN:     // ContentOwner
                                    this->tData.sffDisp[tfTown] = state ;
                                    break ;
                                 case cciTXXX:     // Comment
                                    this->tData.sffDisp[tfTxxx] = state ;
                                    break ;
                                 case cciTDAT:     // RecDate
                                    this->tData.sffDisp[tfTdat] = state ;
                                    break ;
                                 case cciTIME:     // RecTime
                                    this->tData.sffDisp[tfTime] = state ;
                                    break ;
                                 case cciTRDA:     // DateSupp
                                    this->tData.sffDisp[tfTrda] = state ;
                                    break ;
                                 case cciTLEN:     // Length
                                    this->tData.sffDisp[tfTlen] = state ;
                                    break ;
                                 case cciTSIZ:     // Size
                                    this->tData.sffDisp[tfTsiz] = state ;
                                    break ;
                                 case cciTBPM:     // BeatsPerMin
                                    this->tData.sffDisp[tfTbpm] = state ;
                                    break ;
                                 case cciTIT1:     // ContentGroup
                                    this->tData.sffDisp[tfTit1] = state ;
                                    break ;
                                 case cciTIT3:     // Subtitle
                                    this->tData.sffDisp[tfTit3] = state ;
                                    break ;
                                 case cciTPE3:     // Conductor
                                    this->tData.sffDisp[tfTpe3] = state ;
                                    break ;
                                 case cciTPOS:     // PartOfSet
                                    this->tData.sffDisp[tfTpos] = state ;
                                    break ;
                                 case cciTKEY:     // InitialKey
                                    this->tData.sffDisp[tfTkey] = state ;
                                    break ;
                                 case cciTLAN:     // Language
                                    this->tData.sffDisp[tfTlan] = state ;
                                    break ;
                                 case cciTOPE:     // OrigArtist
                                    this->tData.sffDisp[tfTope] = state ;
                                    break ;
                                 case cciTOAL:     // OrigAlbum
                                    this->tData.sffDisp[tfToal] = state ;
                                    break ;
                                 case cciTOFN:     // OrigFilename
                                    this->tData.sffDisp[tfTofn] = state ;
                                    break ;
                                 case cciTOLY:     // OrigLyricist
                                    this->tData.sffDisp[tfToly] = state ;
                                    break ;
                                 case cciTORY:     // OrigYear
                                    this->tData.sffDisp[tfTory] = state ;
                                    break ;
                                 case cciTRSN:     // NetStation
                                    this->tData.sffDisp[tfTrsn] = state ;
                                    break ;
                                 case cciTRSO:     // NetOwner
                                    this->tData.sffDisp[tfTrso] = state ;
                                    break ;
                                 case cciTSRC:     // ISRC_Code
                                    this->tData.sffDisp[tfTsrc] = state ;
                                    break ;
                                 case cciTSEE:     // Settings
                                    this->tData.sffDisp[tfTsee] = state ;
                                    break ;
                                 case cciTFLT:     // FileType
                                    this->tData.sffDisp[tfTflt] = state ;
                                    break ;
                                 case cciTDLY:     // PlayDelay
                                    this->tData.sffDisp[tfTdly] = state ;
                                    break ;
                                 case cciTENC:     // EncodedBy
                                    this->tData.sffDisp[tfTenc] = state ;
                                    break ;
                                 case cciTMED:     // MediaType
                                    this->tData.sffDisp[tfTmed] = state ;
                                    break ;

                                 default:
                                    //* Input is controlled, so execution *
                                    //* should never reach this point.    *
                                    break ;
                              } ;
                           }
                        }  // valid options
                     }     // malformed option
                     else
                        syntax = true ;
                  }
               }
               else        // unknown option
                  syntax = true ;


               if ( syntax )
               {
                  gsv.compose( L"Syntax error on line %hd ", &lineNum ) ;
                  this->DiagMsg ( gsv.ustr(), badAttr ) ;
                  if ( ! this->suVerbose )
                  {
                     gs.insert( L" \"" ) ;
                     gs.append( L'"' ) ;
                     this->DiagMsg ( gs.ustr(), cfgAttr ) ;
                  }
                  if ( ++syntaxErrors >= syntaxMax )
                  {
                     this->DiagMsg ( " Multiple Errors! Exiting application.", badAttr ) ;
                     status = ERR ;
                     done = true ;
                  }
                  else
                     this->DiagMsg ( " Continuing with default value...", badAttr ) ;
               }
            }
            else if ( readStatus == ZERO )
               { /* ignores comment or blank line */ }
            else     // (readStatus < ZERO) end-of-file
               done = true ;
         }
         ifs.close() ;                 // close the source file
      }
   }

   return status ;

}  //* End ReadConfigFile() *

//*************************
//*   rcfReadSourceLine   *
//*************************
//******************************************************************************
//* Read and verify a line of data from the source file.                       *
//*                                                                            *
//* Input  : ifs     : handle to open input stream                             *
//*          lineNum : most-recently-read line of source file                  *
//*          gs      : (by reference, initial contents ignored)                *
//*                    receives input data                                     *
//*                                                                            *
//* Returns: > 0  if successful read of a line containing non-comment data     *
//*          ==0  if successful read of comment OR an empty line               *
//*          < 0  if end-of-file (or read error)                               *
//******************************************************************************

short Taggit::rcfReadSourceLine ( ifstream& ifs, short& lineNum, gString& gs )
{
   char  lineData[gsMAXBYTES] ;     // raw UTF-8 input
   short readStatus = (-1) ;        // return value

   ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;

   if ( ifs.good() || ifs.gcount() > ZERO )
   {
      ++lineNum ;             // line counter
      readStatus = 1 ;        // assume non-empty line

      //* Skip leading whitespace and convert to 'wide' characters *
      short i = ZERO ;
      while ( lineData[i] == nckSPACE || lineData[i] == nckTAB )
         ++i ;
      gs = &lineData[i] ;     // result to caller's buffer

      //* Comment or empty line? *
      short lineLen ;
      const wchar_t* wptr = gs.gstr( lineLen ) ;
      if ( (lineLen == 1) || 
           (lineLen > 1 && (wptr[0] == SLASH && wptr[1] == SLASH)) )
         readStatus = ZERO ;
   }
   return readStatus ;

}  //* End rcfReadSourceLine() *

//*************************
//*    Set_UI_Language    *
//*************************
//******************************************************************************
//* Set language for user interface.                                           *
//*                                                                            *
//* Input  : clLang   : user's choice of available language from command line  *
//*                                                                            *
//* Returns: OK  if language set according to user choice or from locale       *
//*          ERR if selection mismatch between language and locale             *
//******************************************************************************
//* Logic:                                                                     *
//*  1) The locale in which we are operating has already been set:             *
//*     (this->cfgOpt.appLocale)                                               *
//*        a) taken from the terminal environment                              *
//*        b) taken from the configuration file                                *
//*        c) supplied as a command-line argument                              *
//*     For full functionality, the locale must support BOTH the target        *
//*     language AND UTF-8 encoding. If however, the locale supports UTF-8     *
//*     encoding, the user may not notice if the locale does not fully support *
//*     the target language. Exceptions might be timestamps or other date and  *
//*     number-related formatting.                                             *
//*                                                                            *
//*  1) If user did not specify a language,                                    *
//*        (cfgOpt.appLanguage == locLang && clLang == locLang),               *
//*     then use the language from the locale IF it is supported, ELSE default *
//*     to English.                                                            *
//*  2) If configuration file specified a non-default UI language,             *
//*        (cfgOpt.appLanguage != locLang),                                    *
//*     use specified language, overriding the language specified by locale.   *
//*  3) If user specified a language as a command-line option,                 *
//*        (clLang != locLang),                                                *
//*     use specified language, overriding both the language specified by the  *
//*     locale and any language specified in the config file.                  *
//*                                                                            *
//******************************************************************************

short Taggit::Set_UI_Language ( AppLang clLang )
{
   gString loco( this->cfgOpt.appLocale ) ;  // copy of locale name
   short status = OK ;

   //* If non-default value specified as a command-line option *
   if ( clLang != locLang )
   {
      this->cfgOpt.appLanguage = clLang ;
   }

   //* If config file specified that language be taken from the locale *
   else if ( this->cfgOpt.appLanguage == locLang )
   {
      if ( (loco.compare( alStrings[enLang], false, 2 )) == ZERO )
         this->cfgOpt.appLanguage = enLang ;
      else if ( (loco.compare( alStrings[esLang], false, 2 )) == ZERO )
         this->cfgOpt.appLanguage = esLang ;
      else if ( (loco.compare( alStrings[zhLang], false, 2 )) == ZERO )
         this->cfgOpt.appLanguage = zhLang ;
      else if ( (loco.compare( alStrings[viLang], false, 2 )) == ZERO )
         this->cfgOpt.appLanguage = viLang ;
      else
      {  //* Locale name does not indicate one of the supported *
         //* languages. Default to English language.            *
         this->cfgOpt.appLanguage = enLang ;
      }
   }

   //* Else, value was set from the configuration file *
   else
      ;

   //* Initialize the RTL languange flag. This determines *
   //* whether static text will be written as LTR or RTL. *
   this->cfgOpt.rtl = AppDirection[this->cfgOpt.appLanguage] ;

   if ( (loco.compare( alStrings[this->cfgOpt.appLanguage], false, 2 )) != ZERO )
      status = ERR ;    // language/locale mismatch

   return status ;

}  //* End Set_UI_Language() *

//*************************
//*  IsSupportedLocale   *
//*************************
//******************************************************************************
//* Verify that user-provided locale is on list of supported locales.          *
//*                                                                            *
//* Programmer's Note: User's input must be validated because if we pass an    *
//* unsupported locale name to the system, the application will crash and burn.*
//* This is actually a system error, but we don't want to take the blame.      *
//*                                                                            *
//* Input  : basePath: (by reference) receives the path/dirname                *
//*                    for creation of temporary files                         *
//*                                                                            *
//* Returns: 'true'  if locale name in on system list                          *
//*          'false' if locale name not found                                  *
//******************************************************************************

bool Taggit::IsSupportedLocale ( const char* localeName )
{
   const char* cmdTemplate = "locale -a 1>\"%S\" 2>/dev/null" ;
   gString gsPath,               // path of temp file
           lName( localeName ) ; // caller's locale name
   bool status = false ;         // return value

   //* Create a temp filename *
   if ( (this->CreateTempname ( gsPath )) )
   {
      //* Create the command *
      gString gsCmd( cmdTemplate, gsPath.gstr() ) ;

      //* Execute the system call, redirecting the output *
      //* of the 'locale' utility to the temp file.       *
      system ( gsCmd.ustr() ) ;

      //* Scan the temp file looking for a match to specified locale name *
      ifstream ifs ( gsPath.ustr(), ifstream::in ) ;
      if ( ifs.is_open() )             // if input file open
      {
         char lineData[gsMAXBYTES + 1] ;
         bool done = false ;
         while ( ! done )
         {
            ifs.getline ( lineData, gsMAXBYTES, NEWLINE ) ;
            if ( ifs.good() || ifs.gcount() > ZERO )
            {  //* Comparison IS case sensitive *
               if ( (lName.compare( lineData )) == ZERO )
               {
                  status = true ;
                  break ;
               }
            }
            else
               done = true ;
         }
         ifs.close() ;     // close the file
      }

      //* Delete the temp file *
      unlink ( gsPath.ustr() ) ;
   }
   return status ;

}  //* End IsSupportedLocale() *

//*************************
//*      IsColorTerm      *
//*************************
//******************************************************************************
//* Test the NCurses Engine's color capabilities.                              *
//*                                                                            *
//* Input  : goodAttr: color attribute for 'good' messages                     *
//*          badAttr : color attribute for 'bad' messages                      *
//*                                                                            *
//* Returns: 'true'  if terminal supports color output AND color engine        *
//*                  successfully initialized                                  *
//*          'false' if monochrome terminal                                    *
//*                  OR if color engine not initialized (not likely)           *
//******************************************************************************

bool Taggit::IsColorTerm ( attr_t goodAttr, attr_t badAttr )
{
   bool status = false ;            // return value

   //* Determine whether the hardware supports display of color text. *
   if ( nc.ColorText_Available () )
   {
      this->DiagMsg ( "Multi-color screen output supported.", goodAttr ) ;

      //* Color Engine successfully initialized? *
      if ( nc.ColorText_Initialized () )
      {
         termColorInfo tci ;
         nc.TerminalColorSupport ( tci ) ;
         gString gsOut( "Color Engine started. (%hd RGB Registers and "
                        "%hd fgnd/bkgnd Color Pairs)", 
                        &tci.rgb_regs, &tci.fgbg_pairs ) ;
         this->DiagMsg ( gsOut.ustr(), goodAttr ) ;
         status = true ;
      }
      else
         this->DiagMsg ( "Unable to start Color Engine.", badAttr ) ;
   }
   else
      this->DiagMsg ( "Terminal does not support color output. "
                      "Starting in monochrome mode.", badAttr ) ;

   return status ;

}  //* End IsColorTerm() *

//*************************
//*    CreateTemppath     *
//*************************
//******************************************************************************
//* Create a unique directory in the system's temp-file directory to contain   *
//* the application's temporary files. Resulting path is returned to caller.   *
//*                                                                            *
//* This method obtains the directory where temp file are to be created, then  *
//* calls 'mkdtemp' to create the unique directory name.                       *
//* All temporary files created by the application will live in this directory,*
//* so clean-up on exit will be easy.                                          *
//*                                                                            *
//* Input  : basePath: (by reference) receives the path/dirname                *
//*                    for creation of temporary files                         *
//*                                                                            *
//* Returns: 'true'  if path/dirname created successfully                      *
//*          'false' if library call failed                                    *
//******************************************************************************

bool Taggit::CreateTemppath ( gString& basePath )
{
   char tn[gsMAXBYTES] ;               // receives the particularized filespec
   bool  status = false ;

   //* Get path of temp directory*
   if ( (this->GetTempdirPath ( basePath )) != false )
   {
      basePath.append( "/TG_XXXXXX" ) ;
      basePath.copy( tn, gsMAXBYTES ) ;
      if ( (mkdtemp ( tn )) != NULL )
      {
         basePath = tn ;
         status = true ;
      }
      else
         basePath.clear() ;
   }
   return status ;

}  //* End CreateTemppath() *

//*************************
//*    CreateTempname     *
//*************************
//******************************************************************************
//* Create a unique path/filename for a temporary file within the previously   *
//* established application temporary directory. (See CreateTemppath())        *
//*                                                                            *
//* Input  : tmpPath: (by reference) receives the path/filename                *
//*                                                                            *
//* Returns: 'true'  if path/filename created successfully                     *
//*          'false' if library call failed                                    *
//******************************************************************************

bool Taggit::CreateTempname ( gString& tmpPath )
{
   char tn[gsMAXBYTES] ;               // receives the particularized filespec
   bool  status = false ;

   gString gstmp( "%s/TG_XXXXXX", this->cfgOpt.tfPath ) ;
   gstmp.copy( tn, gsMAXBYTES ) ;
   int descriptor ;
   if ( (descriptor = mkstemp ( tn )) != (-1) )
   {
      close ( descriptor ) ;     // close the file
      tmpPath = tn ;             // copy target path to caller's buffer
      status = true ;            // declare success
   }
   else
      tmpPath.clear() ;

   return status ;

}  //* End CreateTempname() *

//*************************
//*    DeleteTempname     *
//*************************
//******************************************************************************
//* Delete (unlink) the specified source file.                                 *
//* Although created to manage temporary files, this method can be used for    *
//* general file deletion.                                                     *
//*                                                                            *
//* Input  : srcPath : full path/filename specification of source              *
//*                                                                            *
//* Returns: OK if successful, else returns errno value                        *
//******************************************************************************

short Taggit::DeleteTempname ( const gString& srcPath )
{
   short    status = OK ;
   if ( (unlink ( srcPath.ustr() )) != OK )
      status = errno ;         // return 'errno' value
   return status ;

}  //* End DeleteTempname() *

//*************************
//*    DeleteTemppath     *
//*************************
//******************************************************************************
//* Delete the directory containing the application's temporary files.         *
//*                                                                            *
//* Called by the Taggit destructor. At this point, the directory              *
//* _should be_ empty, but in case it is not, we first delete the contents     *
//* of the directory, and then delete the directory itself.                    *
//*                                                                            *
//* Note that the temp file for exit-to-CWD is not stored in this directory    *
//* because the invocation shell script needs it _after_ exit.                 *
//*                                                                            *
//* Input  : none                                                              *
//*                                                                            *
//* Returns: nothing                                                           *
//*          (There is no point in complaining if we fail.)                    *
//******************************************************************************

void Taggit::DeleteTemppath ( void )
{
   //* First scan the target directory for its contents *
   //* and delete each remaining temp file (if any).    *
   DIR* dirPtr = NULL ;
   if ( (dirPtr = opendir ( this->cfgOpt.tfPath )) != NULL )
   {
      gString   tspec ;             // target filespec
      deStats*  destat ;            // directory-entry record
      bool      showFile ;          // 'true' if file is to be copied

      //* Read each item in the directory *
      while ( (destat = readdir64 ( dirPtr )) != NULL )
      {
         //* Do not include 'current dir' and 'parent dir' names.              *
         showFile = true ;
         if ( destat->d_name[ZERO] == PERIOD )
         {
            if (   destat->d_name[1] == NULLCHAR 
                || (destat->d_name[1] == PERIOD && destat->d_name[2] == NULLCHAR) )
               showFile = false ;
         }

         if ( showFile != false )
         {
            //* Build full path/filename for target, then delete the file.*
            tspec.compose( "%s/%s", this->cfgOpt.tfPath, destat->d_name ) ;
            unlink ( tspec.ustr() ) ;            
         }
      }
      closedir ( dirPtr ) ;      // Close the directory

      //* Delete the directory itself. *
      rmdir ( this->cfgOpt.tfPath ) ;
   }

}  //* End DeleteTemppath() *

//*************************
//*    GetTempdirPath     *
//*************************
//******************************************************************************
//* This is a stub function which provides access to the C++17 function:       *
//*                       temp_directory_path()                                *
//*                                                                            *
//* Because most programmer's will not reach C++17 for at least several        *
//* months, we isolate the call here and implement what we expect is the way   *
//* the library function will be implemented.                                  *
//*         (This WILL NOT work under Windoze--and we don't care.)             *
//*         (Screw Windoze, and the horse it rode in on.         )             *
//* -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- *
//* From <http://en.cppreference.com>                                          *
//* ---------------------------------                                          *
//* These functions are available in C++17 (released Jan. 2018):               *
//* const char* fs::temp_directory_path ( void ) ;                             *
//* const char* fs::temp_directory_path ( std::error_code& ec ) ;              *
//*                                                                            *
//* Returns A directory suitable for temporary files. The path is guaranteed   *
//* to exist and to be a directory. The overload that takes error_code&        *
//* argument returns an empty path on error.                                   *
//*                                                                            *
//* Exceptions: The overload that does not take a std::error_code& parameter   *
//* throws filesystem_error on underlying OS API errors, constructed with path *
//* to be returned as the first argument and the OS error code as the error    *
//* code argument. std::bad_alloc may be thrown if memory allocation fails.    *
//* The overload taking a std::error_code& parameter sets it to the OS API     *
//* error code if an OS API call fails, and executes ec.clear() if no errors   *
//* occur.                                                                     *
//*                                                                            *
//* On POSIX systems, the path may be the one specified in the environment     *
//* variables TMPDIR, TMP, TEMP, TEMPDIR, and, if none of them are specified,  *
//* the path "/tmp" is returned.                                               *
//*                                                                            *
//* Input  : tdPath : (by reference) receives the path string                  *
//*                                                                            *
//* Returns: 'true'  if successful, 'false' if system error                    *
//******************************************************************************

bool Taggit::GetTempdirPath ( gString& tdPath )
{
   //* Default path of temp directory on GNU/Linux systems. *
   //* (Used only if environment variable not set.)         *
   const char* const dfltPath = "/tmp" ;

   const char* envPath ;            // returned by getenv()
   tnFName     fStats ;             // target filestats
   bool        status = false ;     // return value

   if ( (envPath = std::getenv ( "TMPDIR" )) == NULL )
      if ( (envPath = std::getenv ( "TMP" )) == NULL )
         if ( (envPath = std::getenv ( "TEMP" )) == NULL )
            if ( (envPath = std::getenv ( "TEMPDIR" )) == NULL )
               envPath = dfltPath ;
   tdPath = envPath ;
   if ( (this->GetFileStats ( tdPath, fStats )) == OK )
   {
      if ( (fStats.fType == fmDIR_TYPE) && 
           ((access ( tdPath.ustr(), R_OK )) == ZERO) && 
           ((access ( tdPath.ustr(), W_OK )) == ZERO) )
         status = true ;
   }
   return status ;

}  //* End GetTempdirPath()

//************************
//*    GetLocalTime      *
//************************
//******************************************************************************
//* Get the system timecode and convert it to localTime format.                *
//*                                                                            *
//* Input  : ft  : (by reference, initial values ignored)                      *
//*                on return, contains decoded date/time                       *
//*                                                                            *
//* Returns: true if successful, false if system call fails                    *
//******************************************************************************
//* Programmer's Note: Currently we do not decode the timezone and DST fields. *
//*                                                                            *
//* -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  *
//* Note that we do a bit of defensive programming in anticipation of the      *
//* dreaded year 2038 overflow of the 32-bit time_t type.                      *
//******************************************************************************

bool Taggit::GetLocalTime ( localTime& ft )
{
bool        success = false ;                   // return value
  
   ft.reset() ;                                 // set to default values
   if ( (time ( &ft.sysepoch )) != (-1) )
   {
      if ( ft.sysepoch >= ZERO )                // reported time is AFTER the epoch
      {
         ft.epoch = (int64_t)ft.sysepoch ;      // promote to 64-bit
         //* Decode the system time *
         Tm    tm ;                             // Linux time structure
         if ( (localtime_r ( &ft.sysepoch, &tm )) != NULL )
         {
            //* Translate to localTime format *
            ft.julian   = tm.tm_yday ;          // Julian date
            ft.day      = tm.tm_wday ;          // 0 == Sunday ... 6 == Saturday
            ft.date     = tm.tm_mday ;          // today's date
            ft.month    = tm.tm_mon + 1 ;       // month
            ft.year     = tm.tm_year + 1900 ;   // year
            ft.hours    = tm.tm_hour ;          // hour
            ft.minutes  = tm.tm_min ;           // minutes
            ft.seconds  = tm.tm_sec ;           // seconds
            success = true ;
         }
      }
      else
      {  /* SYSTEM ERROR - time_t OVERFLOW */ }
   }
   return success ;

}  //* End GetLocalTime() *

//*************************
//*      gsRevChars       *
//*************************
//******************************************************************************
//* Reverse the order of wchar_t characters in a gString object.               *
//* Note: this will work for all single-integer characters; however, for       *
//*       languages with multi-integer characters (e.g. Arabic) some characters*
//*       may be reversed incorrectly.                                         *
//*                                                                            *
//* Input  : gs : (by reference) data to be reversed                           *
//*                                                                            *
//* Returns: number of characters in string (incl. NULLCHAR)                   *
//******************************************************************************

short Taggit::gsRevChars ( gString& gs )
{
   short wCnt = ZERO, t = ZERO ;
   wchar_t wbuff[gsMAXCHARS] ;
   const wchar_t* wPtr = gs.gstr( wCnt ) ;
   for ( short s = wCnt - 2 ; s >= ZERO ; --s )
      wbuff[t++] = wPtr[s] ;
   wbuff[t] = NULLCHAR ;
   gs = wbuff ;

   return gs.gschars() ;

}  //* End geRevChars() *

//****************************
//* LaunchDefaultApplication *
//****************************
//******************************************************************************
//* Launch the system default application for handling the specified file type.*
//*                                                                            *
//* Input  : targFile : filespec of file to be processed by the external       *
//*                     application                                            *
//*          redirect : (optional, 'true' by default)                          *
//*                     if 'true',  redirect stdout and stderr to temp files   *
//*                     if 'false', allow stdout and stderr to write to the    *
//*                                 terminal window                            *
//*                                                                            *
//* Returns: a) the child process ID                                           *
//*          b) -1 if child process not created                                *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) fork a new process: vfork()                                             *
//* 2) Redirection:                                                            *
//*    a) It is possible that the called application will write to the terminal*
//*       window, either during startup or with some later status message.     *
//*    b) If the called application is a console application, this is almost   *
//*       certain; while for external GUI applications, it is still likely to  *
//*       some extent.                                                         *
//*    c) If the parent application has relinquished control of stdout and     *
//*       stderr. i.e. the NCurses engine has been put into hibernation mode,  *
//*       then the called application can display messages without conflict.   *
//*    d) If, however, the parent application is still in active control of    *
//*       stdout and stderr, then resource conflicts are likely to occur.      *
//*       In this case, it is necessary to redirect the called application's   *
//*       output. Our solution is to redirect stdout and stderr to temporary   *
//*       files by default. These files will persist until the parent          *
//*       application closes. (To prevent this, call the method with           *
//*       'redirect' set to 'false'.) The Taggit-class destructor will delete  *
//*       these temporary files during application shutdown.                   *
//* 3) Use the new process to launch the application:                          *
//*    a) This method uses the 'xdg-open' system utility (shell script) which  *
//*       launches the application associated with the file type of 'targFile'.*
//*    b) execlp searches the path looking for 'xdg-open'                      *
//*       Because we are fairly safe in assuming that 'xdg-open' actually IS   *
//*       on the path, the first argument is the name of the script (no path). *
//*    c) The second argument is argv[0] i.e. the script name again.           *
//*    d) The third argument is argv[1] i.e. the filespec passed in by caller. *
//*    e) The fourth argument is argv[2] i.e. a null pointer                   *
//*    f) Note that xdg-open passes only the specified filespec to the target  *
//*       application, and cannot pass additional configuration arguments.     *
//*       To invoke the target application will additional setup/configuration *
//*       parameters, please see the LaunchExternalApplication() method, below.*
//*    g) The child process is not directly associated with the terminal       *
//*       window from which it was launched, so the window may be closed       *
//*       asynchronously.                                                      *
//*    h) The exec functions do not return. The child process terminates when  *
//*       the external application exits.                                      *
//* 4) primary process returns to caller                                       *
//*                                                                            *
//* Programmer's Note: This sequence assumes that multi-threading is not       *
//* active. For multi-threaded applications, use pthread_fork() instead.       *
//******************************************************************************

int Taggit::LaunchDefaultApplication ( const char* targFile, bool redirect )
{
   gString stdoutTemp, stderrTemp ;       // stdout target file
   this->CreateTempname ( stdoutTemp ) ;  // stderr target file
   this->CreateTempname ( stderrTemp ) ;  // create the temp filenames
   int soDesc = ZERO,                     // stdout target descriptor
       seDesc ;                           // stderr target descriptor

   pid_t fpid = vfork () ;                // create the child process

   if ( fpid == ZERO)
   {  //* The child process (if created) executes here.                  *
      //* Create temp files as targets for redirecting stdout and stderr.*
      if ( redirect )
      {
         soDesc = open ( stdoutTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         seDesc = open ( stderrTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         dup2 ( soDesc, STDOUT_FILENO) ;
         dup2 ( seDesc, STDERR_FILENO) ;
         close ( soDesc ) ;
         close ( seDesc ) ;
      }

      execlp ( "xdg-open", "xdg-open", targFile, NULL ) ;

      //* In case the exec call fails: child process MUST NOT return.*
      _exit ( 0 ) ;
   }
   else
   {  //* The parent process continues execution here *
      if ( fpid > ZERO )      // child successfully launched
      {
         /* Currently, nothing to be done. */
      }
   }
   return fpid ;

}  //* End LaunchDefaultApplication() *

//*****************************
//* LaunchExternalApplication *
//*****************************
//******************************************************************************
//* Launch a specific external application along with desired arguments.       *
//* See notes below for a detailed discussion of the arguments.                *
//*                                                                            *
//* Input  : appSpec  : filespec (or filename) of application                  *
//*        : argList  : arguments sent to target application as argv[] array   *
//*        : redirect : (optional, 'true' by default)                          *
//*                     if 'true',  redirect stdout and stderr to temp files   *
//*                     if 'false', allow stdout and stderr to write to the    *
//*                                 terminal window                            *
//*                                                                            *
//* Returns: a) the child process ID                                           *
//*          b) -1 if child process not created                                *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) This method requires some intelligence on the part of the caller.       *
//*    Unlike the LaunchDefaultApplication() method above, this method must    *
//*    be able to parse the individual arguments into separate substrings      *
//*    with a pointer to each. This is what the called application will see as *
//*    the "char* argv[]" array.                                               *
//* 2) The 'appSpec' parameter will contain the target application's name,     *
//*    relative path or absolute path.                                         *
//*    a) If it is known that the application in on the path, this may be the  *
//*       name of the executable file only.                                    *
//*    b) If caller is not sure whether the executable is on the path, then    *
//*       'appSpec' parameter should be the absolute or relative filespec.     *
//*    c) Path search:                                                         *
//*        i) If 'appSpec' does not contain slash ( '/' ) characters,          *
//*           then 'execvp' will search the $PATH environment variable.        *
//*       ii) If 'appSpec' DOES contain slash ( '/' ) characters,              *
//*           then 'execvp' will assume that you have provided a valid         *
//*           filespec, and will not search the $PATH.                         *
//*    d) By convention, argv[0] is the application name. This method captures *
//*       the tail of 'appSpec', stripping any path elements and inserts the   *
//*       application name as argv[0].                                         *
//* 3) The 'argList' parameter contains a list of arguments to be passed to    *
//*    target application.                                                     *
//*    a) Each argument substring, except the last, must be terminated by a    *
//*       newline '\n' character to allow reliable parsing.                    *
//*       Example: "arg01\narg02\narg03\narg04"                                *
//*    b) Any arguments which contain whitespace must be properly quoted:      *
//*       Example:  "arg01\n'Waiting for the Rain.ogs'\narg03"                 *
//*       This method WILL NOT automatically delimit the substrings.           *
//*    c) Any characters which might be incorrectly interpreted as regexp      *
//*       expressions must be properly escaped to prevent the shell from       *
//*       trying to expand them. Examples would be: \[  \]  \*  \.             *
//*       Arguments which contain _intentional_ regexp need not be escaped.    *
//*       Note however, that regexp wildcard characters ABSOLUTELY WILL NOT BE *
//*       EXPANDED. For instance: grep -n 'Taggit::' *.cpp                     *
//*       will not work because the shell will not expand the '*' character.   *                                                                        *
//* 4) One additional argument will be automatically appended to the arg list: *
//*    The NULL pointer indicates the end of the argv[] array.                 *
//* 5) Redirection of 'stdout' and 'stderr':                                   *
//*    Please see discussion of redirection in the LaunchDefaultApplication()  *
//*    method, above.                                                          *
//******************************************************************************

int Taggit::LaunchExternalApplication ( const char* appSpec, 
                                        const gString& argList, bool redirect )
{
   const short MAX_ARGS = 48 ;         // maximum number of argument tokens (arbitrary)
   const char* argv[MAX_ARGS + 1] ;    // argv[] array of arg pointers
   char argvStr[gsMAXBYTES] ;          // temp buffer for arguments
   char appName[gsMAXBYTES] ;          // application filename (no path)
   gString gs ;                        // text formatting
   int soDesc, seDesc ;                // descriptors for temp files
   short wCnt = argList.gschars(),     // number of characters in source
         windx = ZERO,                 // index into wbuff
         nindx = ZERO,                 // index of newline character
         subcnt = ZERO,                // number of characters in substring
         avindx = ZERO,                // index of argvStr array
         avCount = ZERO ;              // number of arguments i.e. 'argc'

   //* Create temp filnames for redirection of stdout and stderr.   *
   //* Unfortunately, the child process will not know how to delete *
   //* the temp files, so we must rely on the Taggit destructor to  *
   //* delete the temp files (which it does by default).            *
   gString stdoutTemp, stderrTemp ;
   if ( redirect )
   {
      this->CreateTempname ( stdoutTemp ) ;
      this->CreateTempname ( stderrTemp ) ;
   }

   //* Isolate the target application's name *
   gs = appSpec ;
   short sl = gs.findlast( L'/' ) ;
   if ( sl >= ZERO )    // if a path specified and not just a filename
      gs.shiftChars( -(sl + 1) ) ;
   gs.copy( appName, gsMAXBYTES ) ;
   argv[avCount++] = appName ;

   //* Convert caller's arguments to an array of char pointers.*
   while ( (nindx = argList.find( L'\n', windx )) >= ZERO )
   {
      subcnt = nindx - windx ;
      argList.substr( gs, windx, subcnt ) ;
      windx = nindx + 1 ;
      argv[avCount++] = &argvStr[avindx] ;
      avindx += gs.copy( &argvStr[avindx], gsMAXBYTES ) ;
   }
   if ( windx < (wCnt - 1) )
   {
      subcnt = (wCnt - windx) - 1 ;
      argList.substr( gs, windx, subcnt ) ;
      argv[avCount++] = &argvStr[avindx] ;
      avindx += gs.copy( &argvStr[avindx], gsMAXBYTES ) ;
   }
   argv[avCount] = NULL ;

   pid_t fpid = vfork () ;
   if ( fpid == ZERO)
   {  //* The child process (if created) executes here. *
      if ( redirect )
      {
         //*  1) Open (create) each temp file.             *
         //*  2) Reassign (redirect) stdout and stderr     *
         //*     to the new descriptors.                   *
         //*  3) Close the temp files.                     *
         soDesc = open ( stdoutTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         seDesc = open ( stderrTemp.ustr(), O_WRONLY | O_CREAT | O_TRUNC, 
                           S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH ) ;
         dup2 ( soDesc, STDOUT_FILENO) ;
         dup2 ( seDesc, STDERR_FILENO) ;
         close ( soDesc ) ;
         close ( seDesc ) ;
      }

      execvp ( appSpec, (char* const*)argv ) ;

      //* In case the exec call fails: child process MUST NOT return.*
      _exit ( 0 ) ;
   }
   else
   {  //* The parent process continues execution here *
      if ( fpid > ZERO )      // child successfully launched
      {
         /* Currently, nothing to be done. */
      }
   }
   return fpid ;

}  //* End LaunchExternalApplication() *

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

