//******************************************************************************
//* File       : TagMp3.cpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 16-Dec-2019                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module supports parsing, display and modification of     *
//*              MP3 (ID3v2.3.0) audio files. Selected id2v3.4.0 extensions    *
//*              are also supported.                                           *
//*                                                                            *
//******************************************************************************
//* Notes:                                                                     *
//*                                                                            *
//* -- Check the list of supported Text-frame IDs in ID3v2.4.                  *
//*    Add these definitions (if any) so we don't seem too far behind the times*
//*                                                                            *
//* -- Multiple tag sequences are possible.                                    *
//*    -- The specification is unclear on this. Older versions say that if a   *
//*       new tag is found, then the data from the old tag should be discarded.*
//*    -- The id3v2.4 specification says that prepended and/or appended tags   *
//*       are acceptable.                                                      *
//*    -- Tags located at the end of the file instead of or in addition to the *
//*       prepended leading tag(s) are signalled by a "SCAN" frame (whatever   *
//*       that is), and a id3v2 footer. This footer carries the same info as   *
//*       tag header, except that the "ID3" becomes "3DI". We should be able   *
//*       to find these tags and report them.                                  *
//*    -- Note that id3v1 tag data was at the last 128 bytes of the file.      *
//*       but if we see that, we silently discard it.                          *
//*                                                                            *
//* -- Multiple strings in a field:                                            *
//*    -- id3v2.3 states that if a null character is found in the text frame   *
//*       byte stream that the remaining data in the frame should be discarded.*
//*    -- id3v2.4 states that multiple strings may be encoded into a single    *
//*       text frame and should be separated by a null character.              *
//*       the format of a null character is encoding dependent:                *
//*       UTF-16: 00 00h  UTF-8: 00h                                           *
//*                                                                            *
//*       4.2.   Text information frames                                       *
//*       "The text information frames are often the most important frames,    *
//*        containing information like artist, album and more. There may only  *
//*        be one text information frame of its kind in an tag. All text       *
//*        information frames supports multiple strings, stored as a null      *
//*        separated list, where null is represented by the termination code   *
//*        for the charater encoding."                                         *
//*    -- See the id3v2_framehdr::txtConv() method and it children for more    *
//*       information.                                                         *
//*                                                                            *
//* -- POPM and PCNT frames:                                                   *
//*    -- The standard is almost _too_flexible on these. Integer values can    *
//*       expand from the basic 32-bit width in 8-bit increments. We do not    *
//*       support that particular foolishness. We limit the counters to        *
//*       32 bits for both read and write of the frame.                        *
//*    -- The so-called email address field in the POPM frame seems to be      *
//*       poorly defined. We can't imagine any user dumb enough to embed their *
//*       email address into an audio file. However, the Windoze media player  *
//*       inserts its own name and version number into this field. This is not *
//*       such a terrible idea because it is the specific player that will set *
//*       and display the "star rating."                                       *
//*    -- See mptDecodePopFrame() and mptEncodePopFrame() as well as the       *
//*       popMeter class definition for additional information.                *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//******************************************************************************

//****************
//* Header Files *
//****************
#include "Taggit.hpp"         // Taggit-class definitions and data
                              // plus general definitions and NcDialogAPI definition

//**************
//* Local data *
//**************

//* ExtractMetadata_MP3: Set to non-zero for debugging only.*
#define DEBUG_EMDATA (0)

//* WriteMetadata_MP3: Set to non-zero for debugging only.*
#define DEBUG_WMDATA (0)

#if DEBUG_EMDATA != 0 || DEBUG_WMDATA != 0
//* Human-friendly equivalents to the TextFrameID list in the header file.*
const char* const TextFrameDesc[TEXT_FRAMES] = 
{
"Title ------------------------ ",  // TIT2
"Lead performer(s) ------------ ",  // TPE1
"Album Title ------------------ ",  // TALB
"Track number/Position -------- ",  // TRCK
"Year ------------------------- ",  // TYER
"Genre ------------------------ ",  // TCON
"Guest artist/group ----------- ",  // TPE2
"Publisher -------------------- ",  // TPUB
"Composer --------------------- ",  // TCOM
"Lyricist/Text writer --------- ",  // TEXT
"Remixed by ------------------- ",  // TPE4
"Copyright © ------------------ ",  // TCOP
"File owner ------------------- ",  // TOWN
"User Defined ----------------- ",  // TXXX
"Date ------------------------- ",  // TDAT
"Time ------------------------- ",  // TIME
"Recording dates -------------- ",  // TRDA
"Length ----------------------- ",  // TLEN
"Size ------------------------- ",  // TSIZ
"BPM (beats per minute) ------- ",  // TBPM
"Content Group ---------------- ",  // TIT1
"Subtitle --------------------- ",  // TIT3
"Conductor/performer ---------- ",  // TPE3
"Part of a set ---------------- ",  // TPOS
"Initial Key ------------------ ",  // TKEY
"Language --------------------- ",  // TLAN
"Original artist -------------- ",  // TOPE
"Original Title --------------- ",  // TOAL
"Original Filename ------------ ",  // TOFN
"Original lyricist ------------ ",  // TOLY
"Original Release ------------- ",  // TORY
"Internet radio station name -- ",  // TRSN
"Internet radio station owner - ",  // TRSO
"ISRC code -------------------- ",  // TSRC
"Software/Hardware ------------ ",  // TSEE
"File type -------------------- ",  // TFLT
"Playlist delay --------------- ",  // TDLY
"Encoded by ------------------- ",  // TENC
"Media type ------------------- ",  // TMED
} ;
#endif   // DEBUG_EMDATA != 0 || DEBUG_WMDATA != 0

//* Size of the obsolete id3v1 tag data which was the last 128 bytes *
//* of the audio file. If found, we discard these obsolete data.     *
static const UINT64 ID3V1_TAG_BYTES = 128 ;
static const char* ID3V1_TAG = "TAG" ;    // id3v1 tag ID


//*****************************
//* Local, non-member methods *
//*****************************
static bool  mptDecodeTagHeader ( ifstream& ifs, id3v2_taghdr& thdr, 
                                  id3v2_audioframe& afhdr ) ;
static short mptEncodeTagHeader ( const id3v2_taghdr& thdr, char* obuff ) ;
static short mptEncodeTagFooter ( const id3v2_tagfooter& tftr, char* obuff ) ;
static short mptDecodeExtendedHeader ( ifstream& ifs, id3v2_taghdr& thdr ) ;
static short mptEncodeExtendedHeader ( const id3v2_taghdr& thdr, UCHAR* obuff ) ;
static short mptDecodeFrameHeader ( ifstream& ifs, id3v2_framehdr& fhdr, 
                                    id3v2_tagfooter& tftr ) ;
static short mptEncodeFrameHeader ( id3v2_framehdr& fhdr, char* obuff ) ;
static UINT  mptDecodeTextFrame ( ifstream& ifs, id3v2_framehdr& fhdr, 
                                  gString& gsOut, ofstream& dbg ) ;
static UINT  mptDecodeImageFrame ( ifstream& ifs, const id3v2_framehdr& fhdr, 
                                   id3v2_image& imgData, AppLang lang ) ;
static UINT  mptDecodePopFrame ( ifstream& ifs, const id3v2_framehdr& fhdr, 
                                 popMeter& popData ) ;


//*************************
//*  ExtractMetadata_MP3  *
//*************************
//******************************************************************************
//* Read the media file and extract the metadata tag fields.                   *
//* Note: Certain data are not displayed:                                      *
//*       a) images (although the image description will be captured)          *
//*       b) compressed data                                                   *
//*       c) encrypted data                                                    *
//*       d) non-text frames are displayed only in verbose mode                *
//*                                                                            *
//* Input  : si     : index into list of source data                           *
//*                                                                            *
//* Returns: 'true' if valid MP3 header,                                       *
//*          'false' invalid header record, file not found, access error, etc. *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) Notes on embedded images.                                               *
//*    a) Filename extensions for temporary image files are named according to *
//*       the MIME type specified by the frame. We do this by making a somewhat*
//*       dangerous assumption: that the application which embedded the image  *
//*       actually followed the specification. The specification states that   *
//*       the image format should be either jpeg or png. "if interoperability  *
//*       is desired." This is a pretty strong warning, but non-standard       *
//*       encodings or incorrect MIME type indicator may cause us to name the  *
//*       file with an incorrect filename extension, in which case, the        *
//*       external image viewer may not be able to display the image.          *
//*                                                                            *
//* 2) Note on "Tag Size":                                                     *
//*    From the id3.v2.4 specification:                                        *
//*    "The ID3v2 tag size is the sum of the byte length of the extended       *
//*     header, the padding and the frames after unsynchronisation. If a       *
//*     footer is present this equals to ('total size' - 20) bytes, otherwise  *
//*     ('total size' - 10) bytes."                                            *
//*    a) What this means is that tag_size does not include the size of the    *
//*       tag header, nor does it include the size of the tag footer (if       *
//*       present.)                                                            *
//*    b) tag_size includes:                                                   *
//*         i) size of extended header (if present), plus                      *
//*        ii) size of the frame data, plus                                    *
//*       iii) size of the padding (if any)                                    *
//******************************************************************************

bool Taggit::ExtractMetadata_MP3 ( short si )
{
   //* Debugging file: Opened for debugging output only. *
   ofstream dbg ;

   gString gsOut, gstmp ;        // text formatting
   bool status = false ;         // return value

   #if DEBUG_EMDATA != 0
   const char* tbrTemplate = "** Tag Bytes Remaining: %S\n" ;
   const char* padTemplate = "Uninitialized space (padding): %S\n" ;
   bool verbose = false ;
   gString dbgFile( this->tData.sf[si].sfPath ) ;
   short indx = (dbgFile.findlast( L'/' )) + 1 ;
   dbgFile.limitChars( indx ) ;
   dbgFile.append( "emd.txt" ) ;
   dbg.open( dbgFile.ustr(), 
             ofstream::out | (si == ZERO ? ofstream::trunc : ofstream::app) ) ;
   if ( dbg.is_open() )
   {
      if ( si == ZERO )
         dbg << "Debug ExtractMetadata_MP3()\n"
                "---------------------------\n" ;
      dbg << "si(" << si << ") '" 
          << this->tData.sf[si].sfName << "'\n" << endl ;
   }
   #endif   // DEBUG_EMDATA

   //* Open the source file *
   ifstream ifs( this->tData.sf[si].sfPath, ifstream::in ) ;
   if ( ifs.is_open() )
   {
      char ibuff[gsMAXBYTES + 4] ;  // input buffer
      UINT Tag_eod = ZERO ;         // unread bytes in tag
      id3v2_taghdr thdr ;           // receives the decoded tag-frame header
      //* Initialized if no tag header found BUT possible audio frame *
      id3v2_audioframe afhdr ;

      bool valid_hdr = mptDecodeTagHeader ( ifs, thdr, afhdr ) ;
      if ( valid_hdr )
      {
         status = true ;
         Tag_eod = thdr.tag_size ;     // tag data byte count

         #if DEBUG_EMDATA != 0
         gstmp.formatInt( thdr.tag_size, 11, true ) ;
         bool funsynch = thdr.flags.unsynch(),
              fexthdr  = thdr.flags.exthdr(),
              fexper   = thdr.flags.exper(),
              ffooter  = thdr.flags.footer() ;
         gsOut.compose( "ID3v2 Tag Header:\n"
                        " File ID: %s\n"
                        " Version: 2.%hhX.%hhX\n"
                        " Flags  : %hhd %hhd %hhd %hhd\n"
                        " TagSize: %S\n"
                        " Valid  : true\n",
                        thdr.file_id, &thdr.major, &thdr.rev, 
                        &funsynch, &fexthdr, &fexper, &ffooter, 
                        gstmp.gstr() ) ;
         dbg << gsOut.ustr() << endl ;
         #endif   // DEBUG_EMDATA

         //* If an "extended header" is defined *
         if ( thdr.flags.exthdr() )
         {
            Tag_eod -= mptDecodeExtendedHeader ( ifs, thdr ) ;

            #if DEBUG_EMDATA != 0
            gstmp.formatInt( thdr.padBytes, 11, true ) ;
            gsOut.compose( L"Extended Header:\n"
                            "  exBytes: %d\n"
                            "  exFlags: %-#hb\n"
                            "  crcFlag: %hhd\n"
                            " padBytes: %S\n"
                            "   tagCRC: %08X\n",
                            &thdr.exBytes, &thdr.exFlags, &thdr.crcFlag, 
                            gstmp.gstr(), &thdr.tagCRC ) ;
            dbg << gsOut.ustr() << endl ;
            #endif   // DEBUG_EMDATA
         }

         #if DEBUG_EMDATA != 0
         gstmp.formatInt( Tag_eod, 11, true ) ;
         gsOut.compose( tbrTemplate, gstmp.gstr() ) ;
         dbg << gsOut.ustr() << endl ;
         #endif   // DEBUG_EMDATA

         //* Read each frame and report its contents *
         id3v2_framehdr fhdr ;
         id3v2_tagfooter tftr ;
         do
         {
            //* Read the frame header to determine the type of frame.*
            Tag_eod -= mptDecodeFrameHeader ( ifs, fhdr, tftr ) ;

            #if DEBUG_EMDATA != 0
            if ( verbose &&
                  ((*fhdr.frame_id >= 'A' && *fhdr.frame_id <= 'Z') ||
                   (*fhdr.frame_id >= '0' && *fhdr.frame_id <= '9')) )
            {
               gstmp.formatInt( fhdr.frame_size, 11, true ) ;
               gsOut.compose( L"Frame %s\n"
                               " size     : %S\n"
                               " flags    : %-#hhb %-#hhb\n"
                               " tag_pres : %hhd\n"
                               " file_pres: %hhd\n"
                               " readonly : %hhd\n"
                               " compress : %hhd\n"
                               " encrypt  : %hhd\n"
                               " grouped  : %hhd\n",
                               fhdr.frame_id, gstmp.gstr(), 
                               &fhdr.status_flags, &fhdr.encode_flags, 
                               &fhdr.flag_tag_pres, &fhdr.flag_file_pres, 
                               &fhdr.flag_readonly, &fhdr.flag_compress, 
                               &fhdr.flag_encrypt, &fhdr.flag_grouped ) ;

               //* Optional data appended to frame header.                     *
               //* Note that if any of these optional data are present, then   *
               //* 'frame_size' will have been reduced by an equivalent amount.*
               if ( fhdr.decomp > ZERO )     // decompressed frame-data size
               {
                  gstmp.formatInt( fhdr.decomp, 11, true ) ;
                  gsOut.append( " expanded : %S\n", gstmp.gstr() ) ;
               }
               if ( fhdr.crypto != ZERO )
                  gsOut.append( " crypto   : %hhX\n", &fhdr.crypto ) ;
               if ( fhdr.group_id != ZERO )
                  gsOut.append( " group id : %hhX\n", &fhdr.group_id ) ;

               dbg << gsOut.ustr() ;
            }
            #endif   // DEBUG_EMDATA

            if ( *fhdr.frame_id == 'T' )        // text frames
            {
               Tag_eod -= mptDecodeTextFrame ( ifs, fhdr, gsOut, dbg ) ;

               //* Determine which field the tag belongs to.  *
               //* Walk through the list of field identifiers *
               //* until a match is found, then write the text*
               //* to the corresponding tag field.            *
               bool goodID = false ;
               gstmp = fhdr.frame_id ;
               for ( short i = ZERO ; i < TEXT_FRAMES ; ++i )
               {
                  if ( (gstmp.compare( TextFrameID[i] )) == ZERO )
                  {
                     gsOut.copy( this->tData.sf[si].sfTag.field[i], gsMAXCHARS ) ;

                     #if DEBUG_EMDATA != 0
                     dbg << TextFrameDesc[i] << gsOut.ustr() << '\n' ;
                     gstmp.formatInt( Tag_eod, 11, true ) ;
                     gsOut.compose( tbrTemplate, gstmp.gstr() ) ;
                     dbg << gsOut.ustr() << endl ;
                     #endif   // DEBUG_EMDATA

                     goodID = true ;
                     break ;
                  }
               }
               if ( ! goodID )
               {  //* Text-frame ID is not recognized.             *
                  //* Currently, we simply ignore the non-standard *
                  //* frame because we have no place to display it.*
                  #if DEBUG_EMDATA != 0
                  gsOut.compose( "Warning: Non-standard text-frame ID: '%S'\n"
                                 "         (frame discarded)", gstmp.gstr() ) ;
                  dbg << gsOut.ustr() << endl ;
                  #endif   // DEBUG_EMDATA
               }
            }
            //* Non-text frames *
            else if (   (*fhdr.frame_id >= 'A' && *fhdr.frame_id <= 'Z')
                     || (*fhdr.frame_id >= '0' && *fhdr.frame_id <= '9') )
            {
               //* If an embedded image, extract the setup info *
               //* and save the image to a temporary file.      *
               gstmp = fhdr.frame_id ;
               if ( (gstmp.compare( IMAGE_TAG )) == ZERO )
               {
                  //* Point to the first unused node in the image list. *
                  //* If no unused nodes, add a node to the list.       *
                  EmbeddedImage* eiPtr = this->tData.sf[si].sfTag.addPic () ;
                  id3v2_image*   epPtr = &eiPtr->mp3img ;
                  eiPtr->inUse = true ;

                  //* Create a temporary file to hold the image data. *
                  this->CreateTempname ( gstmp ) ;
                  gstmp.copy( epPtr->picPath, gsMAXBYTES ) ;

                  //**************************
                  //* Extract the image data *
                  //**************************
                  UINT br = mptDecodeImageFrame ( ifs, fhdr, *epPtr, 
                                                  this->cfgOpt.appLanguage ) ;
                  Tag_eod -= br ;

                  //* Rename the temporary file so it can be read by an *
                  //* external image viewer. See note in method header. *
// UNDER CONSTRUCTION - IF MIME TYPE IS A URL REFERENCE, THEN THIS COULD BE INCORRECT.
                  gstmp = epPtr->mimType ;
                  gsOut = epPtr->picPath ;
                  bool ispng = ((gstmp.find( "png" )) >= ZERO) ;
                  gstmp.compose( "%S.%s", gsOut.gstr(), (ispng ? "png" : "jpg") ) ;
                  this->RenameFile ( gsOut, gstmp ) ;
                  gstmp.copy( epPtr->picPath, gsMAXBYTES ) ;

                  #if DEBUG_EMDATA != 0
                  //* Report: Image found, image size, MIME type, *
                  //* text description and target image file.     *
                  if ( dbg.is_open() )
                  {
                     gstmp.formatInt( epPtr->picSize, 11, true ) ;
                     gString gsOut( "** %s FRAME (embedded image)\n"
                                    "Text encoding: %02hhXh\n"
                                    "Picture type : %02hhXh - %s\n"
                                    "MIME type    : %s\n"
                                    "Description  : '%s'\n"
                                    "Image size   : %S\n"
                                    "Image file   : '%s'\n",
                        IMAGE_TAG, &epPtr->encoding, &epPtr->picType, 
                        epPtr->picExpl, epPtr->mimType, epPtr->txtDesc, 
                        gstmp.gstr(), epPtr->picPath ) ;
                     dbg << gsOut.ustr() << endl ;
                  }
                  #endif   // DEBUG_EMDATA
               }

               //* If a popularimeter OR play-counter frames *
               else if ( ((gstmp.compare( POP_TAG )) == ZERO) ||
                         ((gstmp.compare( CNT_TAG )) == ZERO) )
               {
                  UINT br = mptDecodePopFrame ( ifs, fhdr, 
                                                this->tData.sf[si].sfTag.pop ) ;
                  Tag_eod -= br ;

                  #if DEBUG_EMDATA != 0
                  if ( dbg.is_open() )
                  {
                     gstmp.formatInt( this->tData.sf[si].sfTag.pop.popCount, 11, true ) ;
                     gString gsplay( this->tData.sf[si].sfTag.pop.playCount, 11, true ) ;
                     gString gsOut( "** %s FRAME(%d bytes):\n"
                                    "popEmail  : %s\n"
                                    "popStar   : %02hhu\n"
                                    "popCount  : %S\n"
                                    "playCount : %S\n",
                        fhdr.frame_id, &fhdr.frame_size, 
                        this->tData.sf[si].sfTag.pop.popEmail,
                        &this->tData.sf[si].sfTag.pop.popStar,
                        gstmp.gstr(), gsplay.gstr() ) ;
                     dbg << gsOut.ustr() << endl ;
                  }
                  #endif   // DEBUG_EMDATA
               }

               //* Other non-text frames are not decoded *
               else
               {
                  int loop = fhdr.frame_size / gsMAXBYTES, 
                      remainder = fhdr.frame_size % gsMAXBYTES ;

                  //* Read the raw data *
                  for ( int i = ZERO ; i < loop ; ++i )
                  {
                     ifs.read( ibuff, gsMAXBYTES ) ;
                     Tag_eod -= gsMAXBYTES ;
                  }
                  if ( remainder > ZERO )
                  {
                     ifs.read( ibuff, remainder ) ;
                     Tag_eod -= remainder ;
                  }

                  #if DEBUG_EMDATA != 0
                  if ( verbose )
                  {
                     gstmp.formatInt( Tag_eod, 11, true ) ;
                     gsOut.compose( tbrTemplate, gstmp.gstr() ) ;
                     dbg << " (non-text frame: '"
                         << fhdr.frame_id
                         << "' not decoded)\n"
                         << gsOut.ustr() << endl ;
                  }
                  #endif   // DEBUG_EMDATA
               }
            }
            else
            {
               #if 0    // DISABLE
               //* If the frame-header object was not initialized, *
               //* we may have a frame footer. Currently, there is *
               //* nothing to be done about it, especially since   *
               //* a footer would be out-of-position here because  *
               //* its size is not included in the tag size.       *
               if ( *fhdr.frame_id == NULLCHAR )
               {
                  if ( (tftr.foot_id[0] == '3') && 
                       (tftr.foot_id[1] == 'D') && 
                       (tftr.foot_id[2] == 'I') )
                  {
                     
                  }
               }
               #endif   // DISABLE

               // Programmer's Note: In real life, padding is routinely added
               // to the tag without bothering to mention it (no extended header).
               // Although this is technically an error, there's not much we 
               // can do about it at this level.
               // Therefore, an invalid frame ID means we're done.
               #if DEBUG_EMDATA != 0
               int padCount = Tag_eod + framehdrCNT ;
               gstmp.formatInt( padCount, 11, true ) ;
               gsOut.compose( padTemplate, gstmp.gstr() ) ;
               dbg << gsOut.ustr() ;
               #endif   // DEBUG_EMDATA

               Tag_eod = ZERO ;     // discard remainder of tag
            }
         }
         while ( Tag_eod > ZERO ) ;

         //* If a tag footer is defined, read and discard it.*
         // Programmer's Note: Because we have never found an audio file which 
         // actually has a footer, this has only been tested on simulated data.
         // The specification says that there will be no padding between the 
         // frame data and the footer. This means that Tag_eod SHOULD hit zero 
         // just before the footer. We need to find one to test our theory...
         if ( thdr.flags.footer() )
         {
            mptDecodeFrameHeader ( ifs, fhdr, tftr ) ;

            #if DEBUG_EMDATA != 0
            if ( dbg.is_open() )
            {
               gstmp.formatInt( tftr.tag_size, 11, true ) ;
               UCHAR flags = tftr.flags.getflags() ;
               gsOut.compose( L"ID3c2.4 Tag Footer\n"
                               " Footer ID: %s\n"
                               " Version: 2.%hhX.%hhX\n"
                               " flags    : %-#hhb\n"
                               " TagSize  : %S\n",
                               tftr.foot_id, &tftr.major, &tftr.rev,
                               &flags, gstmp.gstr() ) ;
               dbg << gsOut.ustr() << endl ;
            }
            #endif   // DEBUG_EMDATA
         }
      }

      //* Tag Header was missing or corrupted, *
      //* test for an audio frame header.      *
      else
      {
         #if DEBUG_EMDATA != 0
         if ( dbg.is_open() )
         {
            if ( afhdr.valid )
            {
               gsOut.compose( "MPEG Audio Frame Header\n"
                  " synch     : %03hXh\n"
                  " version   : %02hXh\n"
                  " layer     : %02hXh\n"
                  " bitrate   : %02hXh\n"
                  " samprate  : %02hXh\n"
                  " channel   : %02hXh\n"
                  " jsmode    : %02hXh\n"
                  " emphasis  : %02hXh\n"
                  " crc       : %hhd\n"
                  " padding   : %hhd\n"
                  " infobit   : %hhd\n"
                  " copyright : %hhd\n"
                  " original  : %hhd\n"
                  " frame_size: %u\n"
                  " valid     : %hhd\n",
                  &afhdr.synch, &afhdr.version, &afhdr.layer,
                  &afhdr.bitrate, &afhdr.samprate, &afhdr.channel, &afhdr.jsmode, 
                  &afhdr.emphasis, &afhdr.crc, &afhdr.padding, &afhdr.infobit,
                  &afhdr.copyright, &afhdr.original, &afhdr.frame_size,
                  &afhdr.valid ) ;
               dbg << gsOut.ustr() << endl ;
            }
            else
               dbg << "ID3v2 Tag Header: missing or invalid\n" << endl ;
         }
         #endif   // DEBUG_EMDATA

         //* The mptDecodeTagHeader() method has read the first few *
         //* bytes of the source file and has determined whether    *
         //* they contain a valid MPEG Audio Frame Header.          *
         status = afhdr.valid ;
      }     // invalid or missing tag header

      ifs.close() ;           // close the source file
   }

   #if DEBUG_EMDATA != 0
   dbg << endl ;
   dbg.close() ;              // close the debugging file
   #endif   // DEBUG_EMDATA

   return status ;

}  //* End ExtractMetadata_MP3() *

//*************************
//*   WriteMetadata_MP3   *
//*************************
//******************************************************************************
//* Write edited metadata to target file, or alternatively write only the      *
//* headers and audio data.                                                    *
//*                                                                            *
//* Input  : si       : index into list of source data                         *
//*          filter   : (member of enum WriteFilter)                           *
//*                     determines which metadata will be written to target.   *
//*          preserve : (optional, 'false' by default)                         *
//*                     If specified, retain the backup file after the target  *
//*                     has been successfully written.                         *
//*                                                                            *
//* Returns: 'true' if file successfully written                               *
//*          'false' if: a) invalid source header                              *
//*                      b) unable to encode the data                          *
//*                      c) file not found, access error, out-of-storage-space,*
//*                         etc.                                               *
//******************************************************************************
//* Notes:            
//* 1) Rename source file as a backup file (unless alternate target filename). *
//*    a) If a file of that name already exists, it will be deleted before     *
//*       the rename.                                                          *
//* 2) Set the target filename.                                                *
//*    a) By default, this is the original source filename.                    *
//*    b) Optionally, caller will have specified an alternate target filename. *
//* 3) Create a temporary file which will contain the tag data.                *
//*    a) extended header (if defined)                                         *
//*    b) existing non-text frames (excluding image frames)                    *
//*    c) edited text frames                                                   *
//*    d) edited image frames                                                  *
//* 4) Open the source file.                                                   *
//*    a) Read and decode the tag header, extracting the size of existing tag. *
//*    b) Read and decode the extended header (if defined), size of padding    *
//*       area and CRC valued if defined. (The source extended-header is       *
//*       discarded.)                                                          *
//*    c) Important Note: The extended header (if defined for _output_) is part*
//*       of the tag data and technically should be written at the top of the  *
//*       temp file; _however_ we don't yet know if it is defined for output,  *
//*       nor what its contents would be.                                      *
//*    d) Read the source file's tag data.                                     *
//*         i) Discard all source file text-frame data.                        *
//*        ii) Discard all source file image-frame data.                       *
//*       iii) Discard all source file popularimeter and play-counter frames.  *
//*        iv) Write all other non-text frames to the temp file without        *
//*            modification.                                                   *
//*         v) Discard any source file padding data.                           *
//*        vi) Discard tag footer (if defined).                                *
//* 5) Encode the edited text-frame data and write them to the temp file.      *
//* 6) Encode any Popularimeter or Play Count frame data and write them        *
//*    to the temp file.                                                       *
//* 7) Encode any image frame data and write them to the temp file.            *
//* 8) Get the number of bytes written to the temp file. This, plus the size   *
//*    of the target extended header is the 'tag_size' incorporated into the   *
//*    target file's tag header.                                               *
//* 9) Close the temp file.                                                    *
//*10) Calculate the size of the target Tag.                                   *
//*    a) size of the temp file which includes:                                *
//*         i) non-text/non-image frames                                       *
//*        ii) user-selected text frames                                       *
//*       iii) collected image frames                                          *
//*        iv) popularimeter and play-count frames                             *
//*         v) padding                                                         *
//*    b) extended header                                                      *
//*    d) tag footer (if used)                                                 *
//*11) Open the target file.                                                   *
//*    a) Encode and write the target tag header.                              *
//*    b) Create the Extended Header. (This is needed ONLY if there is padding *
//*       at the end of the Tag.) Note that padding is forbidden if there      *
//*       is a Tag Footer.                                                     *
//*    c) Write the Extended Header to target.                                 *
//*12) Reopen the temp file for reading.                                       *
//*13) Copy contents of temp file i.e. the tag data, to target file.           *
//*    (If specified, continue CRC calculation.)                               *
//*    (Note that a tag must contain at least one frame.)                      *
//*14) Write the remainder of the source file to target.                       *
//*    a) Exclude any redundant tag data at the end of the source file.        *
//*15) Close source, target and temporary files.                               *
//*16) Delete the temporary file.                                              *
//*17) Unless caller has specified otherwise, delete the backup file.          *
//*                                                                            *
//*                                                                            *
//******************************************************************************

bool Taggit::WriteMetadata_MP3 ( short si, WriteFilter filter, bool preserve )
{
   #define DUMP_IMAGE_LIST (0)   // FOR DEBUGGING ONLY
   #define DUMP_ID3V1_TEST (0)

   //* Enable/Disable inclusion of the extended header in the target file.   *
   //* Many (if not most) id3v2.3 media players cannot decode the Extended   *
   //* Header, therefore we have placed the Extended Header code _including_ *
   //* the padding block under the control of these flags. We will revisit   *
   //* this when we obtain a few id3v2.4-compliant media players for testing.*
   #define ENABLE_EXTHDR   (0)
   #define ENABLE_PADDING  (1)


   gString gsOut, gstmp ;        // text formatting
   ofstream dbg ;                // handle for debugging output
   bool status = true ;          // Return value

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   const char* fraTemplate = "'%s' (%d bytes) tbr:%S" ;
   bool verbose = false ;  // verbose debugging output
   //* For debugging, do not overwrite the source file *
   static short anSequence = ZERO ; // for dynamic creation of target filenames
   char altName[MAX_FNAME] = "" ;
   #if 1    // ENABLED BY DEFAULT
   gString gsdbg( "wmd%02hd.%02hd.mp3", &si, &anSequence ) ;
   gsdbg.copy( altName, MAX_FNAME ) ;
   ++anSequence ;
   #endif   // ENABLED BY DEFAULT

   gString dbgFile( this->tData.sf[si].sfPath ) ;
   short indx = (dbgFile.findlast( L'/' )) + 1 ;
   dbgFile.limitChars( indx ) ;
   dbgFile.append( "wmd.txt" ) ;
   dbg.open( dbgFile.ustr(), 
             ofstream::out | (si == ZERO ? ofstream::trunc : ofstream::app) ) ;
   if ( dbg.is_open() )
   {
      if ( si == ZERO )
         dbg << "Debug WriteMetadata_MP3()\n"
                "-------------------------\n" ;
      dbg << "si(" << si << ")  filter(" 
          << (filter == wfALL_DATA ? "wfALL_DATA" : 
              filter == wfALL_ACTDATA ? "wfALL_ACTDATA" :
              filter == wfALL_ACTIVE ? "wfALL_ACTIVE" : "wfAUDIO_ONLY")
          << ")  preserve(" << preserve << ")\n"
          << " sfName: '" << this->tData.sf[si].sfName << "'\n" ;
      if ( *altName != NULLCHAR )
          dbg << "altName: '" << altName << "'\n" ;
      dbg << endl ;

      #if DEBUG_WMDATA != 0 && DUMP_IMAGE_LIST != 0   // FOR DEBUGGING ONLY
      dbg << "\nImage Dump\n----------" << endl ;
      if ( *this->cfgOpt.extImage.picPath != NULLCHAR )
      {
         gsdbg.compose( "External Image:\n"
                        "  picPath: '%s'\n"
                        "  txtDesc: '%s'\n"
                        "  mimType: '%s' picType: %02hhX picExpl: '%s'\n"
                        "  picSize: %d",
                        this->cfgOpt.extImage.picPath, this->cfgOpt.extImage.txtDesc,
                        this->cfgOpt.extImage.mimType, 
                        &this->cfgOpt.extImage.picType, this->cfgOpt.extImage.picExpl,
                        &this->cfgOpt.extImage.picSize ) ;
         dbg << gsdbg.ustr() << endl ;
      }
      else
         dbg << "External image not specified" << endl ;

      EmbeddedImage* imgPtr = &this->tData.sf[si].sfTag.ePic ;
      while ( imgPtr != NULL && imgPtr->inUse )
      {
         gsdbg.compose( "Embedded Image:\n"
                        "  picPath: '%s'\n"
                        "  txtDesc: '%s'\n"
                        "  mimType: '%s' picType: %02hhX picExpl: '%s'\n"
                        "  picSize: %d",
                        imgPtr->mp3img.picPath, imgPtr->mp3img.txtDesc,
                        imgPtr->mp3img.mimType, 
                        &imgPtr->mp3img.picType, imgPtr->mp3img.picExpl,
                        &imgPtr->mp3img.picSize ) ;
         dbg << gsdbg.ustr() << endl ;
         imgPtr = imgPtr->next ;
      }
      dbg << endl ;
      #endif   // DEBUG_WMDATA && DUMP_IMAGE_LIST
   }
   #endif                  // DEBUG_WMDATA

   gString srcPath( this->tData.sf[si].sfPath ),      // source filespec
           bacPath( "%S~", srcPath.gstr() ),          // backup filespec
           tmpPath,                                   // temporary filespec
           trgPath = srcPath ;                        // target filespec

   //* If user has specified a different target filename,      *
   //* adjust the target filespec.                             *
   //*                                                         *
   if ( this->tData.sf[si].sfMod != false )
   {
      short fnameIndex = (trgPath.findlast( L'/' )) + 1 ;
      trgPath.limitChars( fnameIndex ) ;
      trgPath.append( this->tData.sf[si].sfName ) ;
   }

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   //* For debugging, create the temp file in the target directory.*
   tmpPath = trgPath ;
   indx = ((tmpPath.findlast( L'/')) + 1) ;
   tmpPath.limitChars( indx ) ;
   tmpPath.append( "wmdtmp.mp3" ) ;

   //* If alternate target filename *
   if ( *altName != NULLCHAR )
   {
      trgPath.limitChars( indx ) ;
      trgPath.append( altName ) ;

      if ( dbg.is_open() )
         dbg << "         Source backup not needed." << endl ;
   }
   else
   #endif   // DEBUG_WMDATA
   {
      //* Create the temporary file path/filename *
      this->CreateTempname ( tmpPath ) ;

      //* Rename the source file as a backup  *
      //* If backup already exists, delete it *
      short bac = OK ;
      fmFType fType ;
      if ( (this->TargetExists ( bacPath, fType )) != false )
      {
         bac = this->DeleteFile ( bacPath ) ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {  if ( bac == OK )
               dbg << "         Existing backup deleted." << endl ;
            else
               dbg << "         Existing backup delete error!" << endl ;
         }
         #endif                  // DEBUG_WMDATA
      }

      //* Rename the source file *
      if ( bac == OK )
      {
         bac = this->RenameFile ( srcPath, bacPath ) ;
         srcPath = bacPath ;  // original source no longer exists
      }
      if ( bac != OK )        // access error, cannot continue
      {
         status = false ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            dbg << "Rename to backup failed. Abort.\n" << endl ;
         }
         #endif                  // DEBUG_WMDATA
      }
   }

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   if ( dbg.is_open() )
   {
      dbg << "srcPath: '" << srcPath.ustr() << "'\n"
          << "bacPath: '" << bacPath.ustr() << "'\n"
          << "tmpPath: '" << tmpPath.ustr() << "'\n"
          << "trgPath: '" << trgPath.ustr() << "'\n" << endl ;
   }
   #endif                  // DEBUG_WMDATA

   //***************************************************
   //* If setup successful, open source and temp files *
   //***************************************************
   if ( status != false )
   {
      ifstream ifs( srcPath.ustr(), ifstream::in ) ;
      ofstream tofs( tmpPath.ustr(), ofstream::trunc ) ;
      if ( (ifs.is_open() != false) && (tofs.is_open() != false) )
      {
         char ibuff[gsMAXBYTES + 4] ;  // input buffer
         char obuff[gsMAXBYTES + 4] ;  // output buffer
         id3v2_taghdr thdr ;           // tag header data begins the MP3 file
         id3v2_tagfooter tftr ;        // tag footer (if present)
         id3v2_framehdr fhdr ;         // for decoding tag frames
         UINT Tag_eod = ZERO,          // unread bytes in tag
              ntfBytes = ZERO ;        // bytes of non-text-frame data
         int  tmpBytes = ZERO ;        // bytes written to temp file (ideal)

         //* Stat the source file for file size *
         tnFName sfStats ;
         this->GetFileStats ( srcPath, sfStats ) ;

         //* Read the Tag Header *
         id3v2_audioframe afhdr ; // initialized if no tag header found BUT possible audio frame
         bool valid_hdr = mptDecodeTagHeader ( ifs, thdr, afhdr ) ;

         //* If tag header verified, initialize the input byte counter *
         if ( valid_hdr )
            Tag_eod = thdr.tag_size ;
         //* Invalid tag header *
         else
         {
            Tag_eod = ZERO ;

            //* If we read the beginning of the file and found *
            //* an MPEG Audio Frame header instead of a tag    *
            //* header, we must rewind the source file so that *
            //* all the audio data will be written to target.  *
            if ( afhdr.valid )
            {
               ifs.seekg( ZERO ) ;
            }
         }

         #if DEBUG_WMDATA != 0
         if ( dbg.is_open() )
         {
            gstmp.formatInt( thdr.tag_size, 11, true ) ;
            bool funsynch = thdr.flags.unsynch(),
                 fexthdr  = thdr.flags.exthdr(),
                 fexper   = thdr.flags.exper(),
                 ffooter  = thdr.flags.footer() ;
            gsdbg.compose( "Tag Header:\n"
                           " File ID: %s\n"
                           " Version: 2.%hhX.%hhX\n"
                           " Flags  : %hhd %hhd %hhd %hhd\n"
                           " TagSize: %S\n"
                           " Valid  : %s\n",
                           thdr.file_id, &thdr.major, &thdr.rev, 
                           &funsynch, &fexthdr, &fexper, &ffooter, 
                           gstmp.gstr(), (char*)(valid_hdr ? "true" : "false") ) ;
            dbg << gsdbg.ustr() << endl ;

            if ( afhdr.valid )
            {
               dbg << "Tag Header not found, but valid\n"
                      "MPEG Audio Frame Header identified.\n" << endl ;
            }
         }
         #endif   // DEBUG_WMDATA

         //* If Tag Header validated, continue decoding the source file *
         if ( valid_hdr )
         {
            //* If an "extended header" is defined *
            if ( thdr.flags.exthdr() )
            {
               Tag_eod -= mptDecodeExtendedHeader ( ifs, thdr ) ;
   
               #if DEBUG_WMDATA != 0
               if ( dbg.is_open() )
               {
                  gstmp.formatInt( thdr.padBytes, 11, true ) ;
                  gsOut.compose( L"Extended Header:\n"
                                  "  exBytes: %d\n"
                                  "  exFlags: %-#hb\n"
                                  "  crcFlag: %hhd\n"
                                  " padBytes: %S\n"
                                  "   tagCRC: %08X\n",
                                  &thdr.exBytes, &thdr.exFlags, &thdr.crcFlag, 
                                  gstmp.gstr(), &thdr.tagCRC ) ;
                  dbg << gsOut.ustr() << endl ;
               }
               #endif   // DEBUG_WMDATA
            }

            #if DEBUG_WMDATA != 0
            if ( dbg.is_open() )
            {
               gstmp.formatInt( Tag_eod, 11, true ) ;
               gsOut.compose( fraTemplate, "TAGHEADER", &thdr.tag_size, gstmp.gstr() ) ;
               dbg << gsOut.ustr() << endl ;
            }
            #endif   // DEBUG_WMDATA
            
            //* Read all frames in the tag. This pass calculates the space *
            //* adjustment needed for writing the modified tag.            *
            //* -- Non-text frames are saved to a temporary file.          *
            //* -- Text frames are discarded.                              *
            do
            {
               Tag_eod -= mptDecodeFrameHeader ( ifs, fhdr, tftr ) ;

               #if DEBUG_WMDATA != 0
               if ( (dbg.is_open()) && verbose &&
                     ((*fhdr.frame_id >= 'A' && *fhdr.frame_id <= 'Z') ||
                      (*fhdr.frame_id >= '0' && *fhdr.frame_id <= '9')) )
               {
                  gstmp.formatInt( fhdr.frame_size, 11, true ) ;
                  gsOut.compose( L"Frame %s\n"
                                  " size     : %S\n"
                                  " flags    : %-#hb %-#hhb\n"
                                  " tag_pres : %hhd\n"
                                  " file_pres: %hhd\n"
                                  " readonly : %hhd\n"
                                  " compress : %hhd\n"
                                  " encrypt  : %hhd\n"
                                  " grouped  : %hhd\n",
                                  fhdr.frame_id, gstmp.gstr(),
                                  &fhdr.status_flags, &fhdr.encode_flags, 
                                  &fhdr.flag_tag_pres, &fhdr.flag_file_pres, 
                                  &fhdr.flag_readonly, &fhdr.flag_compress, 
                                  &fhdr.flag_encrypt, &fhdr.flag_grouped ) ;

                  //* Optional data appended to frame header.                     *
                  //* Note that if any of these optional data are present, then   *
                  //* 'frame_size' will have been reduced by an equivalent amount.*
                  if ( fhdr.decomp > ZERO )     // decompressed frame-data size
                  {
                     gstmp.formatInt( fhdr.decomp, 11, true ) ;
                     gsOut.append( " expanded : %S\n", gstmp.gstr() ) ;
                  }
                  if ( fhdr.crypto != ZERO )
                     gsOut.append( " crypto   : %hhX\n", &fhdr.crypto ) ;
                  if ( fhdr.group_id != ZERO )
                     gsOut.append( " group id : %hhX\n", &fhdr.group_id ) ;

                  dbg << gsOut.ustr() ;
               }
               #endif   // DEBUG_WMDATA

               //* Discard the existing text frames *
               if ( *fhdr.frame_id == 'T' )        // text frames
               {
                  UINT fsize = mptDecodeTextFrame ( ifs, fhdr, gsOut, dbg ) ;
                  Tag_eod -= fsize ;
   
                  #if DEBUG_WMDATA != 0
                  if ( dbg.is_open() )
                  {
                     gstmp.formatInt( Tag_eod, 11, true ) ;
                     gsOut.compose( fraTemplate, fhdr.frame_id, &fhdr.frame_size, gstmp.gstr() ) ;
                     dbg << gsOut.ustr() << endl ;
                  }
                  #endif   // DEBUG_WMDATA
               }

               //* Discard the existing image frames *
               else if ( (fhdr.frame_id[0] == 'A') && (fhdr.frame_id[1] == 'P') &&
                         (fhdr.frame_id[2] == 'I') && (fhdr.frame_id[3] == 'C') )
               {
                  id3v2_image img ;     // (with no target file specified)
                  UINT fsize = mptDecodeImageFrame ( ifs, fhdr, img, 
                                                     this->cfgOpt.appLanguage ) ;
                  Tag_eod -= fsize ;
                  
                  #if DEBUG_WMDATA != 0
                  if ( dbg.is_open() )
                  {
                     gstmp.formatInt( Tag_eod, 11, true ) ;
                     gsOut.compose( fraTemplate, fhdr.frame_id, &fhdr.frame_size, gstmp.gstr() ) ;
                     dbg << gsOut.ustr() << endl ;
                  }
                  #endif   // DEBUG_WMDATA
               }

               //* Discard Popularimeter and Play Count frames *
               else if ( (fhdr.frame_id[0] == 'P') &&
                         (((fhdr.frame_id[1] == 'O') && (fhdr.frame_id[2] == 'P') && 
                          (fhdr.frame_id[3] == 'M')) ||
                          ((fhdr.frame_id[1] == 'C') && (fhdr.frame_id[2] == 'N') && 
                           (fhdr.frame_id[3] == 'T'))) )
               {
                  popMeter pm ;     // (with no target file specified)
                  UINT fsize = mptDecodePopFrame ( ifs, fhdr, pm ) ;
                  Tag_eod -= fsize ;
   
                  #if DEBUG_WMDATA != 0
                  if ( dbg.is_open() )
                  {
                     gstmp.formatInt( Tag_eod, 11, true ) ;
                     gsOut.compose( fraTemplate, fhdr.frame_id, &fhdr.frame_size, gstmp.gstr() ) ;
                     dbg << gsOut.ustr() << endl ;
                  }
                  #endif   // DEBUG_WMDATA
               }

               //* Copy other non-text frames *
               else if (   (*fhdr.frame_id >= 'A' && *fhdr.frame_id <= 'Z')
                        || (*fhdr.frame_id >= '0' && *fhdr.frame_id <= '9') )
               {
                  //* Encode and write the frame header to the temp file *
                  short tb = mptEncodeFrameHeader ( fhdr, obuff ) ;
                  tmpBytes += tb ;
                  tofs.write( obuff, tb ) ;

                  //* Read the raw tag data, and write it to the temp file *
                  int loop = fhdr.frame_size / gsMAXBYTES, 
                      remainder = fhdr.frame_size % gsMAXBYTES,
                      gc ;
                  for ( int i = ZERO ; i < loop ; ++i )
                  {
                     ifs.read( ibuff, gsMAXBYTES ) ;
                     Tag_eod -= gc = ifs.gcount() ;
                     ntfBytes += gc ;
                     tofs.write( ibuff, gc ) ;
                     tmpBytes += gc ;
                  }
                  if ( remainder > ZERO )
                  {
                     ifs.read( ibuff, remainder ) ;
                     Tag_eod -= gc = ifs.gcount() ;
                     ntfBytes += gc ;
                     tofs.write( ibuff, gc ) ;
                     tmpBytes += gc ;
                  }

                  #if DEBUG_WMDATA != 0
                  if ( dbg.is_open() )
                  {
                     gstmp.formatInt( Tag_eod, 11, true ) ;
                     gsOut.compose( fraTemplate, fhdr.frame_id, 
                                    &fhdr.frame_size, gstmp.gstr() ) ;
                     dbg << gsOut.ustr() << endl ;
                  }
                  #endif   // DEBUG_WMDATA
               }
               //* Invalid frame ID *
               else
               {
                  #if DEBUG_WMDATA != 0
                  UINT padBytes = ZERO ;        // bytes of old padding data
                  #endif   // DEBUG_WMDATA

                  //* Check for a captured Tag Footer record.       *
                  //* Tag Footer is appended to the tag proper, and *
                  //* if present, it signals the end of the tag.    *
                  if ( (tftr.foot_id[0] == '3') &&
                       (tftr.foot_id[1] == 'D') &&
                       (tftr.foot_id[2] == 'I') )
                  {
                     #if DEBUG_WMDATA != 0
                     padBytes = ZERO ; // if footer is present, padding not allowed
                     if ( dbg.is_open() )
                     {
                        gsOut.compose( "\nTag Footer identified\n"
                                         "%s v:%hhd.%hhd  flags:%02hhX\n"
                                         "tag_size:%d\n",
                                       tftr.foot_id, &tftr.major, &tftr.rev, 
                                       &tftr.flags, &tftr.tag_size ) ;
                        dbg << gsOut.ustr() << endl ;
                     }
                     #endif   // DEBUG_WMDATA
                  }

                  //* If it is an invalid ID, it is probably also an invalid frame*
                  //* size, so we don't know how much garbage data to discard to  *
                  //* reach the next frame (if any). Therefore, we must assume    *
                  //* that either:                                                *
                  //* 1) We have reached the end of the tag, (unlikely because we *
                  //*    have not read all data specified by the tag header), OR  *
                  //* 2) We have just read padding data. Save the pad data size,  *
                  //*    then read and discard the padding.                       *
                  else
                  {
                     #if DEBUG_WMDATA != 0
                     padBytes = Tag_eod + framehdrCNT ;
                     #endif   // DEBUG_WMDATA

                     int loop = Tag_eod / gsMAXBYTES, 
                         remainder = Tag_eod % gsMAXBYTES ;
                     for ( int i = ZERO ; i < loop ; ++i )
                     {
                        ifs.read( ibuff, gsMAXBYTES ) ;
                        Tag_eod -= ifs.gcount() ;
                     }
                     if ( remainder > ZERO )
                     {
                        ifs.read( ibuff, remainder ) ;
                        Tag_eod -= ifs.gcount() ;
                     }
                  }
                  Tag_eod = ZERO ;     // be safe

                  #if DEBUG_WMDATA != 0
                  if ( dbg.is_open() )
                  {
                     gstmp.formatInt( padBytes, 11, true ) ;
                     gsOut.compose( fraTemplate, "ENDFRAMES", &padBytes, gstmp.gstr() ) ; 
                     dbg << gsOut.ustr() << endl ;
                  }
                  #endif   // DEBUG_WMDATA
               }
            }
            while ( Tag_eod > ZERO ) ;
         }     // valid_hdr

         //***********************************************************
         //* Initial Tag has now been extracted from the source file.*
         //* Note that it is _possible_ that additional tags may     *
         //* follow, but currently, we ignore that possibility.      *
         //*                                                         *
         //* Important Note: Regardless of whether user is saving the*
         //* edited tag data with the audio or only the audio data,  *
         //* we will always write a Tag Header, Extended Header and  *
         //* padding to the target. This not only makes it easier to *
         //* add tag data later, it also makes it easier to identify *
         //* the file format.                                        *
         //* If writing "audio only," we first write a placeholder   *
         //* tag header, an extended header and fill out to one KiB  *
         //* with padding, with the audio data following.            *
         //***********************************************************
         if ( filter != wfAUDIO_ONLY )
         {
            //* Next, encode the text tags, and write the *
            //* encoded data to the temp file.            *
            tmpBytes += this->mptEncodeTextFrames ( si, filter, tofs, dbg ) ;

            //* Encode Popularimeter and Play Count frames (if any) *
            tmpBytes += this->mptEncodePopFrame ( si, tofs, dbg ) ;

            //* Encode all attached image frames and write the encoded data   *
            //* to the temp file. This includes the images read from the      *
            //* source file unless user has specified to discard them, AND an *
            //* external image file if specified.                             *
            tmpBytes += this->mptEncodeImageFrames ( si, tofs, dbg ) ;

         }     // ! wfAUDIO_ONLY

         //* Reinitialize the tag header object.             *
         //* Because we will always add padding, we need     *
         //* an extended header, so set the 'exthdr' flag.   *
         //* 'major' and 'rev' are initialized for id3v2.3.0.*
         // Programmer's Note: Ideally, we should take the 'major' and 'rev' values
         // from the source file's header, but this needs further consideration.
         //char oldMajor = thdr.major,
         //     oldRev   = thdr.rev ;
         thdr.reset() ;
         thdr.setHeader() ;            // initialize ID and version numbers
         thdr.flags.exthdr( true ) ;   // set extended-header flag

         //* Add an appropriate amount of padding to the temp file.         *
         //* Note that if a tag footer is specified, padding is not allowed.*
         if ( thdr.flags.exthdr() )
         {
            //* According to the specification, the "extended header"     *
            //* is both "optional" AND must be defined if padding is to   *
            //* be added, even though most tag editors (incorrectly)      *
            //* skip this step.                                           *
            //* Calculate the size of padding space. Padding is used by   *
            //* media players to make small modifications to the existing *
            //* text frames without the necessity of re-sizing the overall*
            //* media file. Logically, this would require a few hundred   *
            //* bytes at most; however, analysis shows that padding is    *
            //* generally added to a logical boundary such as the next    *
            //* KiB or is a fixed amount such as 1 KiB (or both).         *
            if ( ! thdr.crcFlag )
               thdr.exBytes = exthdrCNT ;
            else  // CRC values increases size of ext header
            {     // (currently, we do not use a CRC)
               thdr.exBytes = (exthdrCNT + 4) ;
               thdr.exFlags = (short)0x80000 ;
            }
            #if ENABLE_PADDING != ZERO    // (see note above)
            thdr.padBytes = (1024 - (tmpBytes % 1024)) + 1024 ;
            for ( int i = ZERO ; i < thdr.padBytes ; ++i )
               tofs.put( 0x00 ) ;
            tmpBytes += thdr.padBytes ;
            #endif   // ENABLE_PADDING
            #if ENABLE_EXTHDR == 0        // if exthdr disabled
            thdr.flags.exthdr( false ) ;  // reset extended-header flag
            thdr.crcFlag = false ;        // reset CRC flag and adjust hdr size
            thdr.exBytes = exthdrCNT ;
            #endif   // ENABLE_EXTHDR
         }

         //* Get the number of bytes written to the temp file,*
         //* then close the temporary file.                   *
         //* Note that the target file tag size is the number *
         //* of bytes in the temp file PLUS the size of the   *
         //* extended header.                                 *
         thdr.tag_size = thdr.exBytes + tofs.tellp() ;
         tofs.close() ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            int totBytes = tmpBytes + thdr.exBytes ;
            gsdbg.compose( "Temp File Write: (tmpBytes+exBytes:%d) tag_size:%d %s", 
                           &totBytes, &thdr.tag_size, 
                           (const char*)(totBytes == thdr.tag_size ? "OK" : "ERROR") ) ;
            dbg << gsdbg.ustr() << endl ;

            //* New Tag Header *
            gstmp.formatInt( thdr.tag_size, 11, true ) ;
            bool funsynch = thdr.flags.unsynch(),
                 fexthdr  = thdr.flags.exthdr(),
                 fexper   = thdr.flags.exper(),
                 ffooter  = thdr.flags.footer() ;
            gsdbg.compose( "Tag Header:\n"
                           " File ID: %s\n"
                           " Version: 2.%hhX.%hhX\n"
                           " Flags  : %hhd %hhd %hhd %hhd\n"
                           " TagSize: %S",
                           thdr.file_id, &thdr.major, &thdr.rev, 
                           &funsynch, &fexthdr, &fexper, &ffooter, 
                           gstmp.gstr() ) ;
            dbg << gsdbg.ustr() << endl ;

            gsdbg.clear() ;                  // indicates neither exthdr nor footer

            if ( thdr.flags.exthdr() )       // if extended header
            {
               gstmp.formatInt( thdr.padBytes, 11, true ) ;
               gsdbg.compose( "Extended Header: %s\n"
                              " HdrSize: %d\n"
                              " Flags  : %04hXh\n"
                              " Padding: %S",
                              (ENABLE_EXTHDR == ZERO ? "(disabled)" : " "),
                              &thdr.exBytes, &thdr.exFlags, gstmp.gstr() ) ;
               if ( thdr.crcFlag )
                  gsdbg.append( "\n CRC    : %08Xh", &thdr.tagCRC ) ;
            }
            else if ( thdr.flags.footer() )  // if tag footer
            {
               gstmp.formatInt( tftr.tag_size, 11, true ) ;
               bool funsynch = tftr.flags.unsynch(),
                    fexthdr  = tftr.flags.exthdr(),
                    fexper   = tftr.flags.exper(),
                    ffooter  = tftr.flags.footer() ;
               gsdbg.compose( "Tag Footer:\n"
                              " File ID: %s\n"
                              " Version: 2.%hhX.%hhX\n"
                              " Flags  : %hhd %hhd %hhd %hhd\n"
                              " TagSize: %S",
                              tftr.foot_id, &tftr.major, &tftr.rev, 
                              &funsynch, &fexthdr, &fexper, &ffooter, 
                              gstmp.gstr() ) ;
            }
            dbg << gsdbg.ustr() << '\n' << endl ;
         }
         #endif                  // DEBUG_WMDATA

         //* Open the target file *
         ofstream ofs( trgPath.ustr(), ofstream::trunc ) ;

         //* Encode the tag header and write it to target file.*
         //*    ('footer' bit OR 'exthdr' bit set above)       *
         mptEncodeTagHeader ( thdr, obuff ) ;
         ofs.write( obuff, taghdrCNT ) ;

         //* The Extended Header is considered to be part of the tag     *
         //* data, BUT it is not included in the temporary file, so      *
         //* we encode and write it here. (Note that if CRC is used, the *
         //* extended header must be included in the CRC calculation.    *
         if ( thdr.flags.exthdr() )
         {
            #if ENABLE_EXTHDR != 0        // (see note above)
            int ehbytes = mptEncodeExtendedHeader ( thdr, (UCHAR*)obuff ) ;
            ofs.write( obuff, ehbytes ) ;
            #else    // ! ENABLE_EXTHDR (silences the compiler warning)
            mptEncodeExtendedHeader ( thdr, (UCHAR*)obuff ) ;
            thdr.flags.exthdr( false ) ;
            #endif   // ENABLE_EXTHDR
         }

         //* Append the data in the temporary file to the target file.*
         int gcnt ;
         ifstream tifs( tmpPath.ustr(), ifstream::in ) ;
         if ( tifs.is_open() )
         {
            while ( true )
            {
               tifs.read( ibuff, gsMAXBYTES ) ;
               if ( (gcnt = tifs.gcount()) > ZERO )
                  ofs.write( ibuff, gcnt ) ;
               else
                  break ;
            }
            tifs.close() ;                   // close the temporary file
         }
         this->DeleteTempname ( tmpPath ) ;  // delete the temporary file

         //* If a tag footer has been specified, create it now,     *
         //* encode it and write it to the target file.             *
         //* Programmer's Note: Currently, we never create a footer,*
         //* but we have tested it successfully.                    *
         if ( !(thdr.flags.exthdr()) && (thdr.flags.footer()) )
         {
            tftr.reset() ;
            tftr.setFootID() ;
            tftr.major    = thdr.major ;
            tftr.rev      = thdr.rev ;
            tftr.flags    = thdr.flags.getflags() ;
            tftr.tag_size = thdr.tag_size ;
            //* Encode the footer and write it to temp file.*
            mptEncodeTagFooter ( tftr, obuff ) ;
            ofs.write( obuff, taghdrCNT ) ;
         }

         //***********************************************************
         //* Copy remainder of source file (this is the audio data)  *
         //* to the target file.                                     *
         //* Note: If the ID3V1 tag data are present and end of file,*
         //*       do not copy that section to the target file.      *
         //***********************************************************
         UINT64 bytesThreshold = gsMAXBYTES + ID3V1_TAG_BYTES,
                bytesRemaining = sfStats.fBytes - ifs.tellg() ;
         while ( bytesRemaining > bytesThreshold )
         {
            ifs.read( ibuff, gsMAXBYTES ) ;
            if ( (gcnt = ifs.gcount()) > ZERO )
               ofs.write( ibuff, gcnt ) ;
            bytesRemaining -= gcnt ;
            if ( gcnt == ZERO )
               break ;
         }
         if ( bytesRemaining > ZERO && gcnt > ZERO )
         {
            ifs.read( ibuff, (bytesRemaining - ID3V1_TAG_BYTES) ) ;
            if ( (gcnt = ifs.gcount()) > ZERO )
               ofs.write( ibuff, gcnt ) ;
            ifs.read( ibuff, gsMAXBYTES ) ;

            #if DEBUG_WMDATA != 0 && DUMP_ID3V1_TEST != 0  // FOR DEBUGGING ONLY
            if ( dbg.is_open() )
            {
               dbg << "ID3V1 Tag Data test 54 41 47 ..." << endl ;
               for ( UINT i = ZERO ; i < ID3V1_TAG_BYTES ; ++i )
               {
                  gsOut.compose( "%02hhX ", &ibuff[i] ) ;
                  if ( (i == 31) || (i == 63) || (i == 95) )
                     gsOut.append( L'\n' ) ;
                  dbg << gsOut.ustr() ;
               }
               dbg << '\n' << endl ;
            }
            #endif                  // DEBUG_WMDATA && DUMP_ID3V1_TEST
            if ( (ibuff[0] != ID3V1_TAG[0]) || (ibuff[1] != ID3V1_TAG[1]) || 
                 (ibuff[2] != ID3V1_TAG[2]) )
               ofs.write( ibuff, gcnt ) ;
         }

         if ( ofs.is_open() )    // close target file
            ofs.close() ;

         //* When save is complete, then the target becomes the source for *
         //* any subsequent updates.                                       *
         //* If target filespec != source filespec, update source filespec.*
         if ( this->tData.sf[si].sfMod != false )
            trgPath.copy( this->tData.sf[si].sfPath, gsMAXBYTES ) ;

         //* Reset the edits-pending flags and update the display *
         this->tData.sf[si].sfTag.tfMod = 
         this->tData.sf[si].sfMod       = false ;
      }
      else                    // access error - Abort!
      {
         status = false ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            dbg << "ifs and/or ofs failed to open. Abort.\n" << endl ;
         }
         #endif                  // DEBUG_WMDATA
      }

      if ( ifs.is_open() )    // close source file
         ifs.close() ;

      //* Delete the backup file. *
      // Programmer's Note: Under some circumstances (see above) the backup 
      // file will not have been created, but it does no harm to delete a 
      // file which does not exits.
      if ( ! preserve
         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         && (*altName == NULLCHAR)
         #endif   // DEBUG_WMDATA
         )
      {
         #if DEBUG_WMDATA == 0   // Production

         this->DeleteFile ( bacPath ) ;

         #else    // DEBUG_WMDATA
         if ( (this->DeleteFile ( bacPath )) == OK )
            dbg << "         Backup file deleted" << endl ;
         else
            dbg << "         Error deleting backup file." << endl ;
         #endif   // DEBUG_WMDATA
      }
      #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
      else if ( *altName != NULLCHAR )
      {  //* If our dummy filename was used for output, update *
         //* the source path and filename to point to it.      *
         gsdbg = this->tData.sf[si].sfPath ;
         indx = gsdbg.findlast( L'/' ) ;
         gsdbg.limitChars( indx + 1 ) ;
         gsdbg.append ( altName ) ;
         gsdbg.copy( this->tData.sf[si].sfPath, gsMAXCHARS ) ;
         gsdbg = altName ;
         gsdbg.copy( this->tData.sf[si].sfName, MAX_FNAME ) ;
      }
      #endif   // DEBUG_WMDATA
   }

   #if DEBUG_WMDATA != 0
   dbg << endl ;
   dbg.close() ;              // close the debugging file
   #endif   // DEBUG_WMDATA

   return status ;

   #undef DUMP_IMAGE_LIST
}  //* End WritetMetadata_MP3() *

//*************************
//*  mptEncodeTextFrames  *
//*************************
//******************************************************************************
//* Encode the set of text frames for the specified file.                      *
//*                                                                            *
//* Input  : si    : index into list of source data                            *
//*          filter: (member of enum WriteFilter)                              *
//*                  determines which metadata will be written to target.      *
//*          tofs  : handle for open output stream (temp file)                 *
//*          dbg   : handle for debugging output stream                        *
//*                  (file is opened by caller if debugging output is required)*
//*                                                                            *
//* Returns: number of bytes written to target file                            *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* There are options when encoding a text frame (see enum TextEncode).        *
//* id3v2.3:                                                                   *
//* ------                                                                     *
//*  ASCII    : This is by far the most common encoding for MP3 file tags;     *
//*             however, it leaves out most of the world's population.         *
//*  UTF-16   : This encoding uses the Byte-Order-Mark (BOM), which is a       *
//*             two-byte code (0xFEFF) indicating whether the data were encoded*
//*             as big-endian or little-endian.                                *
//*                                                                            *
//* id3v2.4:                                                                   *
//* ------                                                                     *
//*  UTF-8    : This is the Linux standard (and the logical choice), but is    *
//*             not available in earlier versions of the id3 standard.         *
//*  UTF-16BE : This is a completely ridiculous extension to the utterly       *
//*             ridiculous UTF-16 encoding of earlier versions.                *
//*                                                                            *
//* Logic:                                                                     *
//* ------                                                                     *
//* Considerations:                                                            *
//*  1) id3 version with which the source file was originally created.         *
//*  2) Type of source data, with international characters, or in pure ASCII.  *
//*  3) User's preference ('enc' parameter).                                   *
//*                                                                            *
//* 1) If encoding to UTF-8, just copy source to target.                       *
//* 2) If encoding to ASCII, verify that source is ASCII and copy to target.   *
//* 3) If encoding to UTF-16 (with BOM, always big-endian in production mode)  *
//* 4) If encoding to UTF-16BE (big-endian, no BOM)                            *
//*                                                                            *
//* There are many sub-options for text-frame encoding, but we do not support  *
//* them:                                                                      *
//*  1) We do not support data compression.                                    *
//*  2) We do not support data encryption.                                     *
//*  3) We do not support little-endian 16-bit encoding                        *
//*     (but see 'utf16leEncode' method used for debugging only).              *
//*  4) We do not support the obsolete UCS-2 16-bit fixed-width encoding.      *
//*                                                                            *
//* The code supports UTF-16BE and UTF-8 encoding which are available in       *
//* the id3.4 specification. Note, however, that these encodings may not be    *
//* recognized by current media playback hardware and software.                *
//* For this reason, if the desired encoding is not specified in the           *
//* configuration file, then if source data are not ASCII we default to        *
//* UTF-16, big-endian encoding _with_ BOM.                                    *
//*                                                                            *
//*                                                                            *
//* Basic Multilingual Plane (BMP):                                            *
//* -------------------------------                                            *
//* Values in the Basic Multilingual Plane (BMP) i.e. UCS-2, are:              *
//*    0x0000 - 0xFFFF                                                         *
//* For 4-byte code points outside the BOM, UTF-16, UTF-16BE and UTF-16LE      *
//* encoding will require more work.                                           *
//*  a) Subtract 0x010000 to get a 20-bit value between 0x000000 and 0xFFFFFF  *
//*  b) Top 10 bits are added to 0xD800                                        *
//*  c) Bottom 10 bits are added to 0xDC00                                     *
//* Template:  10000h + (H − D800h) × 400h + (L − DC00h)                       *
//*                                                                            *
//* Additional info: convert UTF-16 to UTF-32                                  *
//* From: http://unicode.org/faq/utf_bom.html#utf16-3                          *
//*                                                                            *
//* Constants:                                                                 *
//*  const UTF32 LEAD_OFFSET = 0xD800 - (0x10000 >> 10);                       *
//*  const UTF32 SURROGATE_OFFSET = 0x10000 - (0xD800 << 10) - 0xDC00;         *
//*                                                                            *
//* Computations:                                                              *
//*  UTF16 lead = LEAD_OFFSET + (codepoint >> 10);                             *
//*  UTF16 trail = 0xDC00 + (codepoint & 0x3FF);                               *
//*  UTF32 codepoint = (lead << 10) + trail + SURROGATE_OFFSET;                *
//*                                                                            *
// incomming 16-bit value:                                                     *
//*  a) if ( ! this->big_endian ) swap bytes                                   *
//*  b) if ( <= 0xD7FF || (>= 0xE000 && <= 0xFFFD) )  single, 16-bit code      *
//*  c) else is 2 16-bit codes high-then-low                                   *
//*       i) bring in the next 16-bit value and swap bytes if needed           *
//*      ii) apply the pair-decode                                             *
//*                                                                            *
//* Note: 'assert' is a macro that tests a condition i.e. an if statement.     *
//* #include <stdint.h>                                                        *
//*                                                                            *
//* void encode_utf16_pair(uint32_t character, uint16_t *units)                *
//* {                                                                          *
//*     unsigned int code;                                                     *
//*     assert(0x10000 <= character && character <= 0x10FFFF);                 *
//*     code = (character - 0x10000);                                          *
//*     units[0] = 0xD800 | (code >> 10);                                      *
//*     units[1] = 0xDC00 | (code & 0x3FF);                                    *
//* }                                                                          *
//*                                                                            *
//* uint32_t decode_utf16_pair(uint16_t *units)                                *
//* {                                                                          *
//*     uint32_t code;                                                         *
//*     assert(0xD800 <= units[0] && units[0] <= 0xDBFF);                      *
//*     assert(0xDC00 <= units[1] && units[1] <= 0xDFFF);                      *
//*     code = 0x10000;                                                        *
//*     code += (units[0] & 0x03FF) << 10;                                     *
//*     code += (units[1] & 0x03FF);                                           *
//*     return code;                                                           *
//* }                                                                          *
//*                                                                            *
//******************************************************************************

int Taggit::mptEncodeTextFrames ( short si, WriteFilter filter, 
                                  ofstream& tofs, ofstream& dbg )
{
   char obuff[gsMAXBYTES * 3] ;     // byte-formatted encoded-text output stream
   char hbuff[framehdrCNT] ;        // byte-formatted frame-header output stream
   gString gsw,                     // wchar_t text formatted for display
           gsid ;                   // frame ID string
   id3v2_framehdr frhdr ;           // frame header for data formatting
   TextEncode enc ;                 // text encoding
   int   outBytes = ZERO ;          // return value
   short fhdrBytes ;                // number of encoded bytes for frame header

   //* Note the early return if not saving text frames! *
   if ( filter == wfAUDIO_ONLY )
      return outBytes ;

   //* Get text-encoding instructions *
   if ( this->cfgOpt.txtEncode == ENCODE_UTF8 )
      enc = ENCODE_UTF8 ;        // Encode all text tags as UTF-8
   else if ( this->cfgOpt.txtEncode == ENCODE_UTF16 )
      enc = ENCODE_UTF16 ;       // Encode all text tags as UTF-16
   else if ( this->cfgOpt.txtEncode == ENCODE_UTF16BE )
      enc = ENCODE_UTF16BE ;     // Encode all text tags as UTF-16BE
   else  // If is ASCII, encode as ASCII, else encode as UTF-16 (big-endian)
   {
      if ( gsw.isASCII() ) enc = ENCODE_ASCII ;
      else                 enc = ENCODE_UTF16 ;
   }

   for ( short indx = ZERO ; indx < tfCOUNT ; ++indx )
   {
      //* Apply the frame filter *
      if ( ((filter == wfALL_DATA) && *this->tData.sf[si].sfTag.field[indx] != NULLCHAR)
           ||
           ((filter == wfALL_ACTDATA) && (this->tData.sffDisp[indx])
             && (*this->tData.sf[si].sfTag.field[indx] != NULLCHAR))
           ||
           ((filter == wfALL_ACTIVE) && (this->tData.sffDisp[indx]))
         )
      {
         //* If the field contains text data, convert it to a byte stream.*
         //* Else convert an empty place-holder record                    *
         gsw = this->tData.sf[si].sfTag.field[indx] ;

         #if 0    // NOT IMPLEMENTED
         //* The tfTxxx "Comment" field requires special processing:        *
         //* 4.2.2. User defined text information frame.                    *
         //* "This frame is intended for one-string text information        *
         //* concerning the audio file in a similar way to the other        *
         //* "T"-frames. The frame body consists of a description of the    *
         //* string, represented as a terminated string, followed by the    *
         //* actual string. There may be more than one "TXXX" frame in each *
         //* tag, but only one with the same description.                   *
         //* <Header for 'User defined text information frame', ID: "TXXX"> *
         //* Text encoding  $xx                                             *
         //* Description    <text string according to encoding> $00 (00)    *
         //* Value          <text string according to encoding>             *
         if ( indx == tfTxxx )
         {
            // Programmer's Note: We have never seen a file where this field 
            // is formatted in the way described above. Instead, it is always 
            // a simple string (no "description").
         }
         #endif   // NOT IMPLEMENTED

         frhdr.reset() ;               // reset the data
         if ( (frhdr.txtEncode( enc, indx, obuff, gsw )) > ZERO )
         {
            //* Format and write the frame header to temp file *
            fhdrBytes = mptEncodeFrameHeader ( frhdr, hbuff ) ;
            tofs.write( hbuff, fhdrBytes ) ;
            outBytes += fhdrBytes ;

            #if DEBUG_WMDATA != 0
            if ( dbg.is_open() )
            {
               gString gsdbg( "\n           %s %02hhX, bytes: %d"
                              "\n           %02hhX %02hhX %02hhX %02hhX - "
                              "%02hhX %02hhX %02hhX %02hhX - %02hhX %02hhX"
                              "\n           ", 
                              frhdr.frame_id, &frhdr.encoding, &frhdr.frame_size,
                              &hbuff[0], &hbuff[1], &hbuff[2], &hbuff[3], &hbuff[4], 
                              &hbuff[5], &hbuff[6], &hbuff[7], &hbuff[8], &hbuff[9] ) ;
               dbg << "Encoding: '" << gsw.ustr() << "'"
                   << gsdbg.ustr() ;
               //for ( short i = ZERO ; i < frhdr.frame_size ; ++i )
               //{
               //   gsdbg.compose( "%02hhX ", &obuff[i] ) ;
               //   dbg << gsdbg.ustr() ;
               //}
               dbg << endl ;
            }
            #endif   // DEBUG_WMDATA

            //* Write the frame text data to temp file *
            tofs.write( obuff, frhdr.frame_size ) ;
            outBytes += frhdr.frame_size ;
         }
      }
   }
   return outBytes ;

}  //* End mptEncodeTextFrames()

//*************************
//* mptEncodeImageFrames  *
//*************************
//******************************************************************************
//* Encode the set of image frames for the specified file.                     *
//*                                                                            *
//* Input  : si   : index into list of source data                             *
//*          tofs : handle for open output stream (temp file)                  *
//*          dbg  : handle for debugging output stream                         *
//*                 (file is opened by caller if debugging output is required) *
//*                                                                            *
//* Returns: number of bytes written to target file                            *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* 1) Image frames come from two sources:                                     *
//*    a) embedded in the source file                                          *
//*    b) specified as an external image file                                  *
//* 2) Images are stored in two parts:                                         *
//*    a) Image header data are stored in a linked list of header records,     *
//*       this->tData.sf[si].sfTag.ePic                                        *
//*    b) Image data files, which are either temp files created from image     *
//*       data extracted from the audio source file, OR external image files   *
//*       specified by the user.                                               *
//* 3) The header for each image file is encoded and written to the target     *
//*    file 'tofs'; then the image data file is appended without modification  *
//*    to 'tofs'.                                                              *
//*                                                                            *
//* <Header for 'Attached picture', ID: "APIC">                                *
//* Text encoding   $xx                                                        *
//* MIME type       <text string> $00                                          *
//* Picture type    $xx                                                        *
//* Description     <text string according to encoding> $00 (00)               *
//* Picture data    <binary data>                                              *
//*                                                                            *
//* Example:                                                                   *
//* [Frame Header, 10 bytes, plus image stats]                                 *
//* 'A' 'P' 'I' 'C' ssss sf ef en mype\0 pt desc                               *
//*                                                                            *
//*  where: "APIC" is the frame ID                                             *
//*         ssss     integer frame size                                        *
//*           sf     status-flags byte                                         *
//*           ef     encoding-flags byte                                       *
//*           en     text encoding byte                                        *
//*           mtype  MIME-type string (ASCII, null terminated)                 *
//*           pt     picture-type byte                                         *
//*           desc   description string according to encoding (null terminated)*
//*                                                                            *
//******************************************************************************

int Taggit::mptEncodeImageFrames ( short si, ofstream& tofs, ofstream& dbg )
{
   char obuff[gsMAXBYTES],                // image header buffer
        fhbuff[gsMAXBYTES] ;              // frame header buffer
   EmbeddedImage* p = &this->tData.sf[si].sfTag.ePic ; // pointer to head of list
   ifstream ifs ;                         // for reading image file
   int   outBytes = ZERO ;                // return value
   short cfgBytes = ZERO ;                // encoded image config bytes

   //* Encode the frame header *
   //* a) frame_id is always "APIC"
   //* b) status_flags and encode_flags are always reset
   //* c) big-endian is always 'true' (does not apply to image frame header)
   //* d) encoding is always ASCII (does not apply to image frame header)
   //* e) frame_size is calculated for each image frame
   id3v2_framehdr fhdr ;
   gString gs( IMAGE_TAG ) ;
   gs.copy( fhdr.frame_id, gs.gschars() ) ;
   fhdr.big_endian = true ;
   fhdr.encoding = ENCODE_ASCII ;

   #if DEBUG_WMDATA != 0
   gString gsdbg ;                        // text formatting
   if( this->tData.sf[si].sfTag.ePic.inUse != false )
   {
      if ( dbg.is_open() )
         dbg << "mptEncodeImageFrames\n--------------------" << endl ;
   }
   #endif   // DEBUG_WMDATA

   while ( (p != NULL) && (p->inUse != false) )
   {
      //* Open the image file *
      ifs.open( p->mp3img.picPath, ifstream::in ) ;
      if ( ifs.is_open() )
      {
         //* Encode the image header *
         cfgBytes = p->mp3img.encode ( obuff ) ;

         //* Calculate the frame size, encode the *
         //* frame header and write it to target. *
         fhdr.frame_size = cfgBytes + p->mp3img.picSize ;
         mptEncodeFrameHeader ( fhdr, fhbuff ) ;
         tofs.write( fhbuff, framehdrCNT ) ;
         outBytes += framehdrCNT ;

         //* Write the encoded image configuration data to target *
         tofs.write( obuff, cfgBytes ) ;
         outBytes += cfgBytes ;

         //* Append the image file to target *
         int inbytes ;
         do
         {
            ifs.read( obuff, gsMAXBYTES ) ;
            inbytes = ifs.gcount() ;
            tofs.write( obuff, inbytes ) ;
            outBytes += inbytes ;
         }
         while ( inbytes > ZERO ) ;

         ifs.close() ;              // close the image file

         #if DEBUG_WMDATA != 0
         if ( dbg.is_open() )
         {
            gsdbg.compose( "Frame Header\n"
                           "  id     : %s\n"
                           "  size   : %d\n"
                           "  %02hhX %02hhX %02hhX\n"
                           "Embedded Image:\n"
                           "  picPath: '%s'\n"
                           "  txtDesc: '%s'\n"
                           "  mimType: '%s'\n"
                           "  encode : %02hhX\n"
                           "  picType: %02hhX\n"
                           "  picExpl: '%s'\n"
                           "  picSize: %d\n"
                           "  cfgSize: %hd",
                           fhdr.frame_id, &fhdr.frame_size, &fhdr.encoding,
                           &fhdr.status_flags, &fhdr.encode_flags,
                           p->mp3img.picPath, p->mp3img.txtDesc,
                           p->mp3img.mimType, 
                           &p->mp3img.picType, &p->mp3img.encoding, 
                           p->mp3img.picExpl, &p->mp3img.picSize, &cfgBytes ) ;
            dbg << gsdbg.ustr() << endl ;
         }
         #endif   // DEBUG_WMDATA
      }
      #if DEBUG_WMDATA != 0
      else if ( dbg.is_open() )     // source image file not found
         dbg << "ERROR! Image file '" << p->mp3img.picPath << "' not found." << endl ;
      #endif   // DEBUG_WMDATA

      p = p->next ;
   }

   #if DEBUG_WMDATA != 0
   if( this->tData.sf[si].sfTag.ePic.inUse != false )
   {
      if ( dbg.is_open() )
         dbg << endl ;
   }
   #endif   // DEBUG_WMDATA

   return outBytes ;

}  //* End mptEncodeImageFrames() *

//*************************
//*  mptDecodeTagHeader   *
//*************************
//******************************************************************************
//* Read and decode the MP3 tag header. If it exists, it is usually at the     *
//* beginning of the audio file. The tag header is sometimes located at the    *
//* end of the file.                                                           *
//*                                                                            *
//* If the tag header is not the next 'taghdrCNT' bytes from the input stream, *
//* then we scan the first four(4) bytes of that input for a valid MPEG Audio  *
//* Frame Header, and initialize the 'afhdr' data members.                     *
//*                                                                            *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          thdr    : (by reference) receives the decoded header record       *
//*          afhdr   : (by reference) secondary format verification            *
//*                    If valid Tag Header is found (return value is 'true'):  *
//*                      all members of afhdr reset                            *
//*                    If no Tag Header found or invalid format (return value  *
//*                    is 'false':                                             *
//*                      members of afhdr are initialized                      *
//*                      afhdr::valid indicates whether it is a valid header   *
//*                                                                            *
//* Returns: 'true'  if tag header verified, else 'false'                      *
//******************************************************************************
//*  Structure of id3v2.4 tag.                                                 *
//*                                                                            *
//*  +-----------------------------+    Note that the 'Footer' is new for      *
//*  |      Header (10 bytes)      |    id3v2.4. This includes a newly-defined *
//*  +-----------------------------+    footer bit (bit 4) in the tag header.  *
//*  |       Extended Header       |                                           *
//*  | (variable length, OPTIONAL) |    A tag which preceeds the audio data may*
//*  +-----------------------------+    have an extended header OR a footer,   *
//*  |   Frames (variable length)  |    but not both.                          *
//*  +-----------------------------+                                           *
//*  |           Padding           |    A tag which _follows_ the audio data   *
//*  | (variable length, OPTIONAL) |    _must_ have a footer and _may not_     *
//*  +-----------------------------+    have padding.                          *
//*  | Footer (10 bytes, OPTIONAL) |                                           *
//*  +-----------------------------+                                           *
//*                                                                            *
//* Although this application is built around the id3v2.3 specification, we    *
//* recognize this next-generation header format.                              *
//* See mptDecodeExtendedHeader().                                             *
//*                                                                            *
//******************************************************************************

static bool mptDecodeTagHeader ( ifstream& ifs, id3v2_taghdr& thdr, 
                                 id3v2_audioframe& afhdr )
{
   char ibuff[gsMAXBYTES] ;   // input buffer
   UINT cnt ;                 // for validating header record
   bool validHdr = false ;    // return value

   //* Read header data from source file *
   ifs.read ( ibuff, taghdrCNT ) ;
   cnt = ifs.gcount() ;

   //* A valid ID3 tag header will be of the form: *
   //*    $49 44 43 yy yy xx zz zz zz zz           *
   //* where yy < FFh, xx==flags, zz < 80h         *
   if ( cnt == taghdrCNT )
   {
      thdr.file_id[0]   = ibuff[0] ;
      thdr.file_id[1]   = ibuff[1] ;
      thdr.file_id[2]   = ibuff[2] ;
      thdr.file_id[3]   = NULLCHAR ;
      thdr.major        = ibuff[3] ;
      thdr.rev          = ibuff[4] ;
      thdr.flags.setflags( ibuff[5] ) ;

      //* Tag size value is a 28-bit value encoded *
      //* in 32 bits: bytes 6, 7, 8, 9.            *
      thdr.decodeTagSize ( &ibuff[6] ) ;

      //* Validate the remainder of the header *
      gString gstmp( thdr.file_id ) ;
      if ( (gstmp.compare( "ID3" )) == ZERO &&
           (thdr.major != 0xFF && thdr.rev != 0xFF) )
      {
         validHdr = true ;
      }
      //* Tag Header not found. Check for an MPEG Audio Frame Header *
      else
      {
         afhdr.decode( ibuff ) ;
      }
   }
   return validHdr ;

}  //* End mptDecodeTagHeader() *

//*************************
//*  mptEncodeTagHeader   *
//*************************
//******************************************************************************
//* Convert the formatted tag header into a byte stream.                       *
//*                                                                            *
//* Input  : thdr   : formatted tag header                                     *
//*          obuff  : pointer to output buffer                                 *
//*                                                                            *
//* Returns: number of bytes encoded                                           *
//******************************************************************************
//* Notes:                                                                     *
//* -- We explicitly reset the undefined flag bits (0, 1, 2, 3).               *
//* -- We also explicitly reset the seldom-useful flag bits, 'unsynch' and     *
//*    'experimental'.                                                         *
//* -- Note that our implementation always uses EITHER a tag footer OR an      *
//*    extended header, but it is the caller's responsibility to initialize    *
//*    those flags appropriately.                                              *
//*                                                                            *
//* Format without extended header:                                            *
//* I D 3 m n f tttt  (10 bytes)                                               *
//* where "ID3" == a literal string                                            *
//*       m     == major version (0x03)                                        *
//*       n     == minor version (0x03 or 0x04)                                *
//*       f     == flag byte: %abc00000                                        *
//*       tttt  == tag size (4 bytes, with special formatting)                 *
//*                  See id3v2_taghdr::encodeTagSize() for details.            *
//*                                                                            *
//* Format with extended header: (see also mptEncodeExtendedHeader())          *
//* I D 3 m n f tttt hhhh gg pppp                                              *
//* where hhhh  == extended header size (4 bytes, big-endian)                  *
//*                this is 00 00 00 06 without CRC data,                       *
//*                     or 00 00 00 0A with CRC data                           *
//*       gg    == flag bytes: %a0000000 00000000                              *
//*       pppp  == number of padding bytes (4 byte, big-endian integer)        *
//* If the CRC flag is set (msb of 'gg' above), then the CRC value follows:    *
//*       cccc  == CRC-32 data (4 byte, big-endian integer)                    *
//*                Note that the current Taggit release DOES NOT calculate a   *
//*                CRC error-detection for the tag, so the CRC flag is always  *
//*                reset.                                                      *
//******************************************************************************

static short mptEncodeTagHeader ( const id3v2_taghdr& thdr, char* obuff )
{
   short indx = ZERO ;              // output index and return value

   obuff[indx++] = thdr.file_id[0] ;
   obuff[indx++] = thdr.file_id[1] ;
   obuff[indx++] = thdr.file_id[2] ;
   obuff[indx++] = thdr.major ;
   obuff[indx++] = thdr.rev ;
   obuff[indx++] = thdr.flags.getflags() ;
   indx += thdr.encodeTagSize( &obuff[indx] ) ;

   return indx ;

}  //* End mptEncodeTagHeader() *

//*************************
//*  mptEncodeTagFooter   *
//*************************
//******************************************************************************
//* Convert the formatted tag footer into a byte stream.                       *
//*                                                                            *
//* Input  : tftr   : formatted tag footer                                     *
//*          obuff  : pointer to output buffer                                 *
//*                                                                            *
//* Returns: number of bytes encoded                                           *
//******************************************************************************

static short mptEncodeTagFooter ( const id3v2_tagfooter& tftr, char* obuff )
{
   short indx = ZERO ;              // output index and return value

   obuff[indx++] = tftr.foot_id[0] ;
   obuff[indx++] = tftr.foot_id[1] ;
   obuff[indx++] = tftr.foot_id[2] ;
   obuff[indx++] = tftr.major ;
   obuff[indx++] = tftr.rev ;
   obuff[indx++] = tftr.flags.getflags() ;
   indx += tftr.encodeTagSize( &obuff[indx] ) ;
   
   return indx ;

}  //* End mptEncodeTagFooter() *

//***************************
//* mptDecodeExtendedHeader *
//***************************
//******************************************************************************
//* Read and decode the MP3 "extended header".                                 *
//* -- The Extended Header is primarily to indicate the amount of padding.     *
//* -- Typically, this header is not present even if padding is present in     *
//*    stream. The specification says that the extended header is optional,    *
//*    but we read that to mean that it may be omitted IF there is no padding  *
//*    to report. Unfortunately, most or all tag editors read this as meaning  *
//*    "don't bother to mention the padding". Oh, well, cowboys... need we     *
//*    say more?                                                               *
//* -- Also, if CRC is used, then the CRC lives at the end of the extended     *
//*    header                                                                  *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          thdr    : (by reference) receives the decoded header record       *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************

static short mptDecodeExtendedHeader ( ifstream& ifs, id3v2_taghdr& thdr )
{
   //* MSB of flags set indicates presence of CRC error detection *
   const short CRC_MASK = 0x8000 ;

   char ibuff[gsMAXBYTES] ;   // input buffer
   short bytesRead = ZERO ;   // return value

   //* Get size of extended header *
   ifs.read ( ibuff, 4 ) ;
   bytesRead += ifs.gcount() ;
   thdr.exBytes = thdr.intConv( (UCHAR*)&ibuff[0] ) ;

   //* Read the extended header *
   ifs.read ( ibuff, thdr.exBytes ) ;
   bytesRead += ifs.gcount() ;

   // Decode the flags (16 bits, big-endian) *
   thdr.exFlags = thdr.intConv16( (UCHAR*)(&ibuff[0]) ) ;
   thdr.crcFlag = (bool)(thdr.exFlags & CRC_MASK ) ;

   //* Decode the size of the padding (32 bits, big-endian) *
   thdr.padBytes= thdr.intConv( (UCHAR*)&ibuff[2] ) ;

   //* Decode the CRC value if present (32 bits, big-endian) *
   if ( thdr.crcFlag )
      thdr.tagCRC = thdr.intConv( (UCHAR*)&ibuff[6] ) ;

   return bytesRead ;

}  //* End mptDecodeExtendedHeader() *

//***************************
//* mptEncodeExtendedHeader *
//***************************
//******************************************************************************
//* Convert the decoded MP3 "extended header" to a byte stream.                *
//*                                                                            *
//* Input  : thdr    : (by reference) contains the decoded header record       *
//*          obuff   : receives the encoded binary stream                      *
//*                                                                            *
//* Returns: number of bytes written to the output stream buffer               *
//******************************************************************************

static short mptEncodeExtendedHeader ( const id3v2_taghdr& thdr, UCHAR* obuff )
{
   short indx = ZERO ;

   indx += thdr.intConv( thdr.exBytes, obuff ) ;
   indx += thdr.intConv( thdr.exFlags, &obuff[indx] ) ;
   indx += thdr.intConv( thdr.padBytes, &obuff[indx] ) ;
   if ( thdr.crcFlag )
      indx += thdr.intConv( thdr.tagCRC, &obuff[indx] ) ;

   return indx ;

}  //* End mptEncodeExtendedHeader() *

//***************************
//*  mptDecodeFrameHeader   *
//***************************
//******************************************************************************
//* Read and decode the MP3 frame header.                                      *
//* A frame header details the kind of data contained in the frame.            *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          fhdr    : (by reference) receives the decoded header record       *
//*          tftr    : (by reference) if a Tag Footer is identified INSTEAD OF *
//*                    a frame header, this object recieves its decoded data   *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************
//* Notes:                                                                     *
//* 1) For id3v2.4 an optional "Tag Footer" is defined.                        *
//*    a) In order to avoid corrupting it if found, we do a special test of    *
//*       the frame_id.                                                        *
//*    b) If the tag-footer ID is found, then 'tftr' will receive the decoded  *
//*       data. For efficiency, we do not reset the 'tftr' members if input    *
//*       data are not a tag footer. It is the caller's responsibility to      *
//*       notice if the members have been initialized.                         *
//*    c) By lucky chance, framehdrCNT == taghdrCNT, so the input stream       *
//*       remains synchronized.                                                *
//*                                                                            *
//* 2) Optional data appended to frame header.                                 *
//*    The frame header may have additional data appended if the flags:        *
//*    'flag_compress', 'flag_encrypt' or 'flag_grouped' are set.              *
//*    These optional data (if present) are not included in the header size,   *
//*    but ARE included in the reported frame size.                            *
//*    Therefore, if we have read any of the these optional data, we need to   *
//*    subtract it from the 'frame_size' reported to the caller so caller will *
//*    not read too many bytes for the frame contents. (see below)             *
//******************************************************************************

static short mptDecodeFrameHeader ( ifstream& ifs, id3v2_framehdr& fhdr, 
                                    id3v2_tagfooter& tftr )
{
   const UCHAR usefulFLAGS = 0xE0 ;
   char ibuff[gsMAXBYTES] ;   // input buffer
   short bytesRead = ZERO,    // return value
         i = ZERO ;           // input index

   fhdr.reset() ;       // clear previous data

   ifs.read ( ibuff, framehdrCNT ) ;   // read the frame header
   bytesRead += ifs.gcount() ;

   //* Test for Tag Footer (see note above) *
   if (    (ibuff[0] == '3')
        && ((ibuff[1] == 'D') || (ibuff[1] == 'd'))
        && ((ibuff[2] == 'I') || (ibuff[2] == 'i')) )
   {
      tftr.reset() ;
      tftr.foot_id[0] = ibuff[i++] ;
      tftr.foot_id[1] = ibuff[i++] ;
      tftr.foot_id[2] = ibuff[i++] ;
      tftr.major       = ibuff[i++] ;
      tftr.rev         = ibuff[i++] ;
      tftr.flags.setflags( ibuff[i++] ) ;
      tftr.decodeTagSize( &ibuff[i] ) ;
   }
   else        // frame header
   {
      fhdr.frame_id[0] = ibuff[i++] ;
      fhdr.frame_id[1] = ibuff[i++] ;
      fhdr.frame_id[2] = ibuff[i++] ;
      fhdr.frame_id[3] = ibuff[i++] ;
      fhdr.frame_size = fhdr.intConv( (UCHAR*)&ibuff[i] ) ;
      i += 4 ;    // step over the integer
      fhdr.status_flags = ibuff[i++] & usefulFLAGS ;
      fhdr.encode_flags = ibuff[i] & usefulFLAGS ;
      fhdr.flag_tag_pres  = (fhdr.status_flags & 0x80) ? true : false ;
      fhdr.flag_file_pres = (fhdr.status_flags & 0x40) ? true : false ;
      fhdr.flag_readonly  = (fhdr.status_flags & 0x20) ? true : false ;
      fhdr.flag_compress  = (fhdr.encode_flags & 0x80) ? true : false ;
      fhdr.flag_encrypt   = (fhdr.encode_flags & 0x40) ? true : false ;
      fhdr.flag_grouped   = (fhdr.encode_flags & 0x20) ? true : false ;

      //* Simple test to verify that frame_id is valid data *
      if ( ((*fhdr.frame_id >= 'A' && *fhdr.frame_id <= 'Z') ||
            (*fhdr.frame_id >= '0' && *fhdr.frame_id <= '9')) )
      {
         // Programmer's Note: The following optional additions are not 
         // included in frame header size, but are included in frame size.
         //* If 'compress' flag, 4 bytes added to header *
         //* indicating de-compressed size.              *
         if ( fhdr.flag_compress )
         {
            ifs.read( ibuff, 4 ) ;
            bytesRead += ifs.gcount() ;
            fhdr.decomp = fhdr.intConv( (UCHAR*)&ibuff[0] ) ;
         }
         //* If 'encrypt' flag,  1 byte added to header *
         //* indicating type of encryption.             *
         if ( fhdr.flag_encrypt )
         {
            ifs.read( ibuff, 1 ) ;
            bytesRead += ifs.gcount() ;
            fhdr.crypto = ibuff[0] ;
         }
         //* If 'grouped' flag,  1 byte added to header indicating group ID.*
         if ( fhdr.flag_grouped )
         {
            ifs.read( ibuff, 1 ) ;
            bytesRead += ifs.gcount() ;
            fhdr.group_id = ibuff[0] ;
         }
         //* Adjust reported 'frame_size' if needed. (see note above) *
         // Programmer's Note: Because we have never seen any of these 
         // optional data, we can only _hope_ that this adjustment works.
         if ( (ULONG)bytesRead > framehdrCNT )
            fhdr.frame_size -= (bytesRead - framehdrCNT) ;
      }
   }
   return bytesRead ;

}  //* End mptDecodeFrameHeader() *

//*************************
//* mptEncodeFrameHeader  *
//*************************
//******************************************************************************
//* Convert the formatted frame header into a byte stream.                     *
//*                                                                            *
//* Input  : fhdr   : formatted frame header                                   *
//*          obuff  : pointer to output buffer                                 *
//*                                                                            *
//* Returns: number of bytes encoded                                           *
//******************************************************************************
//* Notes:                                                                     *
//* a) We reset all flags which are undefined in id3v2.3.                      *
//* b) Since we modify/replace all text frames we can eliminate all the        *
//*    flags for these frames because we 'know' that none of them will be      *
//*    used for text data.                                                     *
//* c) Since we do not modify non-text frames, the flags for these frames are  *
//*    not changed. Note that optional data are appended to the header if the  *
//*    following flags are set:                                                *
//*    -- 'flag_compress' (four (4) extra bytes) 'decomp' member               *
//*    -- 'flag_encrypt'  (one (1) extra byte)   'crypto' member               *
//*    -- 'flag_grouped'  (one (1) extra byte)   'group_id' member             *
//*                                                                            *
//******************************************************************************

static short mptEncodeFrameHeader ( id3v2_framehdr& fhdr, char* obuff )
{
   const UCHAR usefulFLAGS = 0xE0 ;
   short indx = ZERO ;        // output index and return value

   obuff[indx++] = fhdr.frame_id[0] ;
   obuff[indx++] = fhdr.frame_id[1] ;
   obuff[indx++] = fhdr.frame_id[2] ;
   obuff[indx++] = fhdr.frame_id[3] ;
   indx += fhdr.intConv( fhdr.frame_size, &obuff[indx] ) ;

   //* For text frames, reset all flags *
   if ( *fhdr.frame_id == 'T' || *fhdr.frame_id == 't' )
   {  //* Note that for text frames, we reset all flags.*
      obuff[indx++] = ZERO ;
      obuff[indx++] = ZERO ;
   }
   //* Appended data for non-text frames? (see note above) *
   else
   {
      obuff[indx++] = fhdr.status_flags & usefulFLAGS ;
      obuff[indx++] = fhdr.encode_flags & usefulFLAGS ;

      if ( fhdr.flag_compress )
         indx += fhdr.intConv( fhdr.decomp, &obuff[indx] ) ;
      if ( fhdr.flag_encrypt )
         obuff[indx++] = fhdr.crypto ;
      if ( fhdr.flag_grouped )
         obuff[indx++] = fhdr.group_id ;
   }

   return indx ;

}  //* End mptEncodeFrameHeader() *

//***************************
//*   mptDecodeTextFrame    *
//***************************
//******************************************************************************
//* Read and decode an MP3 text frame.                                         *
//*   1) Read the raw data.                                                    *
//*   2) Convert the raw text to wchar_t (gString).                            *
//*                                                                            *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          fhdr    : (by reference) receives the decoded header record       *
//*          gsOut   : (by reference) receives the decoded text string         *
//*          dbg     : handle to open debug file (FOR DEBUGGING ONLY)          *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* A text frame is encoded as:   (see 'enum TextEncode')                      *
//*   a) ISO8859-1 (ASCII + Latin-1 extensions) or                             *
//*   b) UTF-16 with Byte-Order-Mark (BOM)  id3v2.2 and id3v2.3                *
//*      -- big-endian, read as   : 0xFE 0xFF, or                              *
//*      -- little-endian, read as: 0xFF 0xFE).                                *
//*   c) UTF-16BE (no BOM) id3v2.4                                             *
//*   d) UTF-8   id3v2.4                                                       *
//*                                                                            *
//* UTF-16 is inflexible and outrageously old-fashioned by Linux standards.    *
//* The only reason it is still used is because the Microsoft(tm) libraries    *
//* use it as the default encoding. (Bill Gates screws us again.)              *
//*   Note that this IS NOT the "UCS-2" encoding used in the id3v1 standard    *
//*   which was limited to 16-bit, fixed-length encoding.                      *
//* We must decode/encode the entire range of UTF-16 codepoints.               *
//*   a) For the Basic Multilingual Plane (BMP), character codepoints are a    *
//*      single, 16-bit value between 0000h - D7FFh or E000h - FFFDh.          *
//*      -- To test codepoints from the BOM, use any string of characters that *
//*         is known to be within the BOM. For instance:                       *
//*                "Hello World, and to you too, 马伦教授!"                     *
//*   b) For the Supplimentary Planes (astral planes), two 16-bit values are   *
//*      combined according to a dumbass scheme which allows for efficient     *
//*      character lookup, but must otherwise be seen as second-rate           *
//*      engineering.                                                          *
//*       -- To test codepoints from the Supplimentary planes, we use some of  *
//*          the characters which lie outside the BMP but which are recognized *
//*          and displayed properly by the en_US.utf8 locale.                  *
//*          These are found in the "Supplimentary Ideographic Plane":         *
//*            CJK Compatibility Ideographs Supplement (2F800–2FA1F hex).      *
//*            "丽丸乁你 侮侻倂偺 備僧像㒞 免兔兤具 㒹內再冗 冤仌冬况 凵刃㓟𡠠"          *
//*          The following table describes these test characters.              *
//*          (Note that many of these characters are seldom used and our)      *
//*          (translations are approximate because some aren't in our   )      *
//*          (dictionary as independent characters, so we had to guess  )      *
//*          (based on how they are combined.                           )      *
//*     (Yes, our Chinese sucks, thanks for noticing. 你是对的，我的中文很烂。)    *
//*                                                                            *
//*                  PINYIN                                                    *
//* CODEPOINT  CHAR  (Phonetic)  TRANSLATION                                   *
//* ---------  ----  ----------  -----------                                   *
//* 0x02F800    丽    lì          Korea  (as in 美丽 , "beautiful")             *
//* 0x02F801    丸    wán         pill   (as in 丸子 , "ball")                  *
//* 0x02F802    乁    yí          (combining glyph)                            *
//* 0x02F804    你    nǐ          you    (as in 你好 , "hello")                 *
//* 0x02F805    侮    wǔ          insult (as in 侮辱)                           *
//* 0x02F806    侻    tuì                                                      *
//* 0x02F807    倂    bìng                                                     *
//* 0x02F808    偺    zá          we/our group                                 *
//* 0x02F809    備    bèi         prepared (as in 備份 , "backup")              *
//* 0x02F80A    僧    sēng        monk     (as in 僧侶)                         *
//* 0x02F80B    像    xiàng       similarity (as in 像这样 , "like this")       *
//* 0x02F80C    㒞    jùn                                                      *
//* 0x02F80E    免    miǎn        free     (as in 免疫 , "immune")              *
//* 0x02F80F    兔    tù          rabbit   (as in 兔子)                         *
//* 0x02F810    兤    huǎng                                                    *
//* 0x02F811    具    jù          with/possessing (as in 具体 , "specific")     *
//* 0x02F813    㒹                                                             *
//* 0x02F814    內    nèi         inside     (as in 內衣 , "underwear")         *
//* 0x02F815    再    zài         again      (as in zàijiàn , "goodbye")       *
//* 0x02F817    冗    rǒng        trivial    (as in 冗余的 , "redundant")       *
//* 0x02F818    冤    yuān        injustice  (as in 冤家 , "enemy")             *
//* 0x02F819    仌    bīng                                                     *
//* 0x02F81A    冬    dōng        winter     (as in 冬天)                       *
//* 0x02F81B    况    kuàng       situation  (as in 况且 , "moreover")          *
//* 0x02F81D    凵    qiǎn                                                     *
//* 0x02F81E    刃    rèn         blade                                        *
//* 0x02F81F    㓟    pí                                                       *
//* 0x02F820    𡠠                                                             *
//******************************************************************************

static UINT mptDecodeTextFrame ( ifstream& ifs, id3v2_framehdr& fhdr, 
                                 gString& gsOut, ofstream& dbg )
{
   char ibuff[gsMAXBYTES] ;   // input buffer
   UINT bytesRead = ZERO ;    // return value

   #if DEBUG_EMDATA != 0
   gString gstmp ;            // text formatting
   bool  ascii = true ;       // if encoded as ascii
   bool verbose = true ;      // controls how much debugging info is written
   #endif   // DEBUG_EMDATA

   if ( fhdr.frame_size < gsMAXBYTES )
   {
      //* Read the raw text *
      ifs.read( ibuff, fhdr.frame_size ) ;
      bytesRead += ifs.gcount() ;
   }
   else
   {
      // Programmer's Note: If text frame is larger than our buffer,
      // (unlikely), we will display a truncated text string.
      // Also, declaring the data captured as less than a full buffer
      // prevents buffer overrun during Unicode conversion.
      ifs.read( ibuff, gsMAXBYTES ) ;
      char ctmp[gsMAXBYTES] ;
      int loop = (fhdr.frame_size - gsMAXBYTES) / gsMAXBYTES, 
          remainder = (fhdr.frame_size - gsMAXBYTES) % gsMAXBYTES ;
      for ( int i = ZERO ; i < loop ; ++i )
         ifs.read( ctmp, gsMAXBYTES ) ;
      if ( remainder > ZERO )
         ifs.read( ctmp, remainder ) ;
      bytesRead += fhdr.frame_size ;
      fhdr.frame_size = gsMAXBYTES - 3 ;
   }

   #if DEBUG_EMDATA != 0
   if ( dbg.is_open() )
   {
      if ( ibuff[0] == ENCODE_ASCII )
         gstmp = "(ASCII)" ;
      else
      {
         ascii = false ;

         if (   (ibuff[0] == ENCODE_UTF16)
             && (fhdr.frame_size >= UTF16_MIN)
             && (ibuff[1] == UTF_MSB || ibuff[1] == UTF_LSB)
             && (ibuff[2] == UTF_MSB || ibuff[2] == UTF_LSB)
            )
         {
            gstmp.compose( "(UTF-16 %s-endian)", ((ibuff[1] == UTF_MSB) ? "big" : "little") ) ;
         }

         else if ( (ibuff[0] == ENCODE_UTF16BE) && (fhdr.frame_size >= UTF16BE_MIN) )
         {  //* UTF-16 big-endian with no byte-order-mark (BOM) *
            //* Note that this option was introduced in id3v2.4.*
            gstmp = "(UTF-16 big-endian with no BOM)" ;
         }
         else
         {
            //* Assume that source is UTF-8 encoded, even if    *
            //* not explicitly indicated.                       *
            //* Note that this option was introduced in id3v2.4.*
            if ( ibuff[0] == ENCODE_UTF8 )
               gstmp = "(UTF-8)" ;
            else
               gstmp = "(non-standard text encoding)" ;
         }
      }

      gsOut.compose( "** TEXT FRAME(%d bytes): ", &fhdr.frame_size ) ;
      dbg << gsOut.ustr() ;

      if ( ! ascii && verbose )  // verbose debugging
      {
         gsOut.compose( "%02hhX %02hhX %02hhX %02hhX %02hhX ",
                        &ibuff[0], &ibuff[1], &ibuff[2], &ibuff[3], &ibuff[4] ) ;
         dbg << gsOut.ustr() ;
      }
      dbg << gstmp.ustr() << endl ;
   }
   #endif   // DEBUG_EMDATA

   //* Convert the raw byte stream to text *
   fhdr.txtConv( ibuff, gsOut ) ;
#if 0    // TEMP TEMP TEMP - TEST ISO8859-1 DECODING
   if ( (gsOut.find( L"Low Rider" )) >= ZERO )
   {
      dbg << "***** ISO 8859-1 TEST *****" << endl ;
      gsOut.clear() ;
      short i = ZERO ;
      ibuff[i++] = ENCODE_ASCII ;
      ibuff[i++] = '!' ;
      for ( UCHAR x = 0x7F ; x < iso8859LAST ; ++x )
         ibuff[i++] = x ;
      ibuff[i++] = iso8859LAST ;
      ibuff[i++] = '!' ;
      ibuff[i++] = '\0' ;
      fhdr.frame_size = i ;
      for ( short i = ZERO ; i < fhdr.frame_size ; ++i )
      {
         gsOut.compose( "%02hhX ", &ibuff[i] ) ;
         dbg << gsOut.ustr() ;
      } dbg << endl ;

      fhdr.txtConv( ibuff, gsOut ) ;
   }
#elif 0  // TEMP TEMP TEMP - TEST UTF-16 DECODING
   if ( (gsOut.find( L"Low Rider" )) >= ZERO )
   {
      dbg << "***** UTF-16 (little-endian) TEST *****" << endl ;
//      gsOut = "Hello World, and to you too, 马伦教授!" ;
      gsOut = "SUPPLIMENTARY CODEPOINTS:"
              " 丽丸乁你 侮侻倂偺 備僧像㒞 免兔兤具 㒹內再冗 冤仌冬况 凵刃㓟𡠠" ;
      fhdr.utf16leEncode( ibuff, gsOut ) ;
      fhdr.txtConv( ibuff, gsOut ) ;
   }
#elif 0  // TEMP TEMP TEMP - TEST UTF-16 DECODING
   if ( (gsOut.find( L"Low Rider" )) >= ZERO )
   {
      dbg << "***** UTF-16 (big-endian) TEST *****" << endl ;
//      gsOut = "Hello World, and to you too, 马伦教授!" ;
      gsOut = "SUPPLIMENTARY CODEPOINTS:"
              " 丽丸乁你 侮侻倂偺 備僧像㒞 免兔兤具 㒹內再冗 冤仌冬况 凵刃㓟𡠠" ;
      fhdr.utf16Encode( ENCODE_UTF16, ibuff, gsOut ) ;
      fhdr.txtConv( ibuff, gsOut ) ;
   }
#elif 0  // TEMP TEMP TEMP - TEST UTF-8 DECODING
   if ( (gsOut.find( L"Low Rider" )) >= ZERO )
   {
      dbg << "***** UTF-8 TEST *****" << endl ;
//      gsOut = "Hello World, and to you too, 马伦教授!" ;
      gsOut = "SUPPLIMENTARY CODEPOINTS:"
              " 丽丸乁你 侮侻倂偺 備僧像㒞 免兔兤具 㒹內再冗 冤仌冬况 凵刃㓟𡠠" ;
      ibuff[0] = ENCODE_UTF8 ;
      gsOut.copy( &ibuff[1], gsMAXBYTES ) ;
      fhdr.frame_size = gsOut.utfbytes() + 1 ;
      fhdr.txtConv( ibuff, gsOut ) ;
   }
#endif   // TEMP TEMP TEMP - TEST UTF-8 DECODING

   return bytesRead ;

}  //* End mptDecodeTextFrame() *

//***************************
//*   mptDecodeImageFrame   *
//***************************
//******************************************************************************
//* Read and decode an MP3 image frame.                                        *
//*   1) Read the setup and description data and save it for later.            *
//*   2) Read the binary image data and save it to the specified file.         *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          fhdr    : (by reference) receives the decoded header record       *
//*          imgData : (by reference) receives the decoded image data          *
//*          lang    : user interface language                                 *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************
//* 4.15. Attached picture                                                     *
//* This frame contains a picture directly related to the audio file.          *
//* Image format is the MIME type and subtype for the image. In the event that *
//* the MIME media type name is omitted, "image/" will be implied.             *
//* The "image/png" or "image/jpeg" picture format should be used when         *
//* interoperability is wanted. Description is a short description of the      *
//* picture, represented as a terminated text string. The description has a    *
//* maximum length of 64 characters, but may be empty. There may be several    *
//* pictures attached to one file, each in their individual "APIC" frame, but  *
//* only one with the same content descriptor. There may only be one picture   *
//* with the picture type declared as picture type $01 and $02 respectively.   *
//* There is the possibility to put only a link to the image file by using the *
//* 'MIME type' "-->" and having a complete URL instead of picture data.       *
//* The use of linked files should however be used sparingly since there is    *
//* the risk of separation of files.                                           *
//*                                                                            *
//*   Programmer's Note: Even though the specification does not make it        *
//*   clear, the MIME type is a null-terminated ASCII (or possibly UTF-8)      *
//*   string.                                                                  *
//*                                                                            *
//*   Programmer's Note: "content descriptor"                                  *
//*   The content descriptor is the text 'description' string. There seems     *
//*   to be no coherent reason for forcing unique image descriptions.          *
//*   This is especially true when there are multiple images with NO           *
//*   description which is actually a violation of the standard.               *
//*                                                                            *
//*   Programmer's Note: The specification excerpt above states that           *
//*   the image description may be up to "64 characters", HOWEVER, the         *
//*   number of _bytes_ required to encode those characters depends            *
//*   upon the text encoding and on the actual codepoints used.                *
//*   -- That is why we read text until the null terminator (00h  or 00-00h)   *
//*      is found. We must assume that the null terminator is present even     *
//*      if the string is empty.                                               *
//*   -- Note that 64 UTF16-encoded "characters" can require anywhere between  *
//*      128 bytes and 258 bytes.                                              *
//*   -- For comfort, we have sized the target buffer, 'txtDesc' as += four(4) *
//*      times 64 _bytes_ so that up to 64 UTF-8-encoded characters (including *
//*      the null terminator) can be captured.                                 *
//*                                                                            *
//* <Header for 'Attached picture', ID: "APIC">                                *
//* Text encoding   $xx                                                        *
//* MIME type       <text string> $00                                          *
//* Picture type    $xx                                                        *
//* Description     <text string according to encoding> $00 (00)               *
//* Picture data    <binary data>                                              *
//*                                                                            *
//* Example:                                                                   *
//* e image/aaaa\0 t some_description\0 [picture data...]                      *
//*                                                                            *
//* Picture Types: (see the 'pType' array)                                     *
//* $00     Other                                                              *
//* $01     32x32 pixels 'file icon' (PNG only)                                *
//* $02     Other file icon                                                    *
//* $03     Cover (front)                                                      *
//* $04     Cover (back)                                                       *
//* $05     Leaflet page                                                       *
//* $06     Media (e.g. label side of CD)                                      *
//* $07     Lead artist/lead performer/soloist                                 *
//* $08     Artist/performer                                                   *
//* $09     Conductor                                                          *
//* $0A     Band/Orchestra                                                     *
//* $0B     Composer                                                           *
//* $0C     Lyricist/text writer                                               *
//* $0D     Recording Location                                                 *
//* $0E     During recording                                                   *
//* $0F     During performance                                                 *
//* $10     Movie/video screen capture                                         *
//* $11     A bright coloured fish                                             *
//* $12     Illustration                                                       *
//* $13     Band/artist logotype                                               *
//* $14     Publisher/Studio logotype                                          *
//*                                                                            *
//* Programmer's Note: If the MIME type is a URL rather than an actual MIME    *
//* type string, then it will probably be truncated to fit the 'mimType' field.*
//* In any case, this application does not handle references to                *
//* externally-located images. Note also that if this is a URL, the size of    *
//* the image will be ZERO which may cause problems further downstream in the  *
//* processing cycle. This needs to be investigated.                           *
//******************************************************************************

static UINT mptDecodeImageFrame ( ifstream& ifs, const id3v2_framehdr& fhdr, 
                                  id3v2_image& imgData, AppLang lang )
{
   char ibuff[gsMAXBYTES + 1] ;  // input buffer
   gString gstmp ;               // data formatting
   ofstream ofs ;                // temp-file access
   bool toFile = (*imgData.picPath != NULLCHAR ? true : false) ;
   UINT bytesRead = ZERO ;       // return value

   //* Read and decode the text data *
   ifs.read( ibuff, 1 ) ;           // text-encoding indicator
   bytesRead += ifs.gcount() ;
   imgData.encoding = (TextEncode)(char)ibuff[0] ;
   for ( short i = ZERO ; i < gsMAXBYTES ; ++i )   // MIME type
   {
      ifs.read( &ibuff[i], 1 ) ;
      bytesRead += ifs.gcount() ;
      if ( ibuff[i] == NULLCHAR )
         break ;
   }
   gstmp = ibuff ;
   gstmp.copy( imgData.mimType, gsMAXBYTES ) ;
   #if 0    // UNDER CONSTRUCTION - TEST FOR URL MIME TYPE
   //* Test for a URL to an external image within the MIME type string *
   gString gsurl ;
   short urlindex = ZERO ;
   if ( (urlindex = gstmp.after( mimeLnk )) > ZERO )
   {
      gsurl = &gstmp.gstr()[urlindex] ;

      //* If the external file is a LOCAL file, we can copy its contents.*
      if ( ((gsurl.find( "file://" )) == ZERO) || ((gsurl.find( ".." )) == ZERO) )
      {
      }
      //* Else, external file is a network location      *
      //* (http, https, ftp etc.) Don't try to access it.*
      else
      {
      }
   }
   #endif   // UNDER CONSTRUCTION - TEST FOR URL MIME TYPE

   ifs.read( ibuff, 1 ) ;           // picture-type indicator
   bytesRead += ifs.gcount() ;
   imgData.picType = ibuff[0] ;
   gstmp = pType[lang][imgData.picType] ;
   gstmp.copy( imgData.picExpl, imgMAX_DESC ) ;

   //* Read the description string according to text encoding *
   if ( (imgData.encoding == ENCODE_ASCII) || (imgData.encoding == ENCODE_UTF8) )
   {
      for ( short i = ZERO ; i < imgMAX_DESC ; ++i )
      {
         ifs.read( &ibuff[i], 1 ) ;
         bytesRead += ifs.gcount() ;
         if ( ibuff[i] == NULLCHAR )
            break ;
         ibuff[i + 1] = NULLCHAR ;     // ensure that string is terminated
      }
   }
   else     // one of the UTF-16 flavours
   {
      for ( short i = ZERO ; i < imgMAX_DESC ; i += 2 )
      {
         ifs.read( &ibuff[i], 2 ) ;
         bytesRead += ifs.gcount() ;
         if ( ibuff[i] == ZERO && ibuff[i+1] == ZERO )
            break ;
         ibuff[i + 2] = ibuff[i + 3] = NULLCHAR ; // ensure that string is terminated
      }
   }
   imgData.txtConv( ibuff, imgData.encoding ) ;

   imgData.picSize = fhdr.frame_size - bytesRead ;
   int loop = imgData.picSize / gsMAXBYTES, 
       remainder = imgData.picSize % gsMAXBYTES ;

   if ( toFile )     // open the temp file
   {
      ofs.open( imgData.picPath, (ofstream::out | ofstream::trunc) ) ;
   }
   for ( int i = ZERO ; i < loop ; ++i )
   {
      ifs.read( ibuff, gsMAXBYTES ) ;
      bytesRead += ifs.gcount() ;
      if ( toFile && (ofs.is_open()) )
         ofs.write( ibuff, ifs.gcount() ) ;
   }
   if ( remainder > ZERO )
   {
      ifs.read( ibuff, remainder ) ;
      bytesRead += ifs.gcount() ;
      if ( toFile && (ofs.is_open()) )
         ofs.write( ibuff, ifs.gcount() ) ;
   }
   if ( toFile )     // close the temp file
   {
      ofs.close() ;
   }

   return bytesRead ;

}  //* End mptDecodeImageFrame() *

//***************************
//*    mptDecodePopFrame    *
//***************************
//******************************************************************************
//* Read and decode an MP3 POPM or PCNT frame.                                 *
//*                                                                            *
//* a) Note that even though the standard allows for multiple popularimeters   *
//*    (POPM), we will only retain the one most recently read. This should not *
//*    be a problem since only a complete fool would encode more than one.     *
//* b) Only one play counter (PCNT) is allowed by the standard, but if         *
//*    multiple instances are encountered, only the most recent will be        *
//*    retained.                                                               *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          fhdr    : (by reference) receives the decoded header record       *
//*          popData : (by reference) receives the decoded frame data          *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************
//* From the id3v2.3.0 standard:                                               *
//* ============================                                               *
//* <Header for 'Play counter', ID: "PCNT">                                    *
//* Counter         $xx xx xx xx (xx ...)                                      *
//*                                                                            *
//* "This is simply a counter of the number of times a file has been played.   *
//*  The value is increased by one every time the file begins to play.         *
//*  There may only be one "PCNT" frame in each tag. When the counter reaches  *
//*  all one's, one byte is inserted in front of the counter thus making the   *
//*  counter eight bits bigger. The counter must be at least 32-bits long to   *
//*  begin with."                                                              *
//* This explanation _implies_ that the value is a big-endian integer.         *
//*                                                                            *
//* <Header for 'Popularimeter', ID: "POPM">                                   *
//* Email to user   <text string> $00                                          *
//* Rating          $xx                                                        *
//* Counter         $xx xx xx xx (xx ...)                                      *                                                          *
//*                                                                            *
//* "The purpose of this frame is to specify how good an audio file is.        *
//*  Many interesting applications could be found to this frame such as a      *
//*  playlist that features better audiofiles more often than others or it     *
//*  could be used to profile a person's taste and find other 'good' files     *
//*  by comparing people's profiles. The frame is very simple. It contains     *
//*  the email address to the user, one rating byte and a four byte play       *
//*  counter, intended to be increased with one for every time the file is     *
//*  played. The email is a terminated string.                                 *
//*  The rating is 1-255 where 1 is worst and 255 is best. 0 is unknown.       *
//*  If no personal counter is wanted it may be omitted. When the counter      *
//*  reaches all one's, one byte is inserted in front of the counter thus      *
//*  making the counter eight bits bigger in the same way as the play          *
//*  counter ("PCNT"). There may be more than one "POPM" frame in each tag,    *
//*  but only one with the same email address.                                 *
//* This explanation _implies_ that not only is the integer a big-endian value,*
//* but that the email address (if any) is an ASCII (or UTF-8) string.         *
//*                                                                            *
//******************************************************************************

static UINT mptDecodePopFrame ( ifstream& ifs, const id3v2_framehdr& fhdr, 
                                popMeter& popData )
{
   char ibuff[gsMAXBYTES] ;            // input buffer
   gString gstmp( fhdr.frame_id ) ;    // data formatting
   UINT bytesRead = ZERO ;             // return value

   //* Read the frame data *
   ifs.read( ibuff, fhdr.frame_size ) ;
   bytesRead = ifs.gcount() ;

   if ( (gstmp.compare( POP_TAG )) == ZERO )
   {
      //* Reinitialize the target fields (possible previous tag field data) *
      popData.popEmail[0] = NULLCHAR ;
      popData.popStar = ZERO ;
      popData.popCount = ZERO ;

      //* Scan the email address(if any). If valid, it will be null terminated.*
      short i = ZERO ;
      while ( i < fhdr.frame_size )
      {
         popData.popEmail[i] = ibuff[i] ;
         if ( popData.popEmail[i++] == NULLCHAR )
            break ;
         else if ( i >= (MAX_FNAME - 2) )  // prevent buffer overrun
            break ;
      }
      popData.popEmail[i] = NULLCHAR ;    // be sure string is terminated

      if ( i < fhdr.frame_size )
      {
         popData.popStar = ibuff[i++] ;
         if ( i <= (fhdr.frame_size - 4) )
         {  //* This is a big-endian integer of at least 32 bits (4 bytes). *
            //* Data beyond 32 bits is silently discard.                    *
            popData.popCount = popData.intConv( (UCHAR*)&ibuff[fhdr.frame_size - 4] ) ;
         }
      }
      popData.popdata = true ;            // set data flag
   }
   else     // CNT_TAG
   {  //* This is a big-endian integer of at least 32 bits (4 bytes). *
      //* Data beyond 32 bits is silently discard.                    *
      popData.playCount = popData.intConv( (UCHAR*)&ibuff[fhdr.frame_size - 4] ) ;
      popData.cntdata = true ;            // set data flag
   }
   return bytesRead ;

}  //* End mptDecodePopFrame() *

//***************************
//*    mptEncodePopFrame    *
//***************************
//******************************************************************************
//* Convert the decoded MP3 Popularimeter AND/OR the Play Count to a binary    *
//* output stream.                                                             *
//*                                                                            *
//* Input  : si    : index into list of source data                            *
//*          tofs  : handle for open output stream (temp file)                 *
//*          dbg   : handle for debugging output stream                        *
//*                    (file opened by caller if debugging output is required) *
//*                                                                            *
//* Returns: number of bytes written to the output stream buffer               *
//******************************************************************************

short Taggit::mptEncodePopFrame ( short si, ofstream& tofs, ofstream& dbg )
{
   short outBytes = ZERO ;                // return value

   if ( (this->tData.sf[si].sfTag.pop.popdata != false) ||
        (this->tData.sf[si].sfTag.pop.cntdata != false) )
   {
      char obuff[gsMAXBYTES],                // frame data buffer
           fhbuff[gsMAXBYTES] ;              // frame header buffer
      gString gs ;                           // text formatting
      popMeter* popPtr = &this->tData.sf[si].sfTag.pop ;

      //* Initialize the frame header *
      //* a) frame_id is either "POPM" or "PCNT"
      //* b) status_flags and encode_flags are always reset
      //* c) big-endian is always 'true'
      //* d) encoding is always ASCII
      //* e) frame_size is calculated
      id3v2_framehdr fhdr ;
      fhdr.big_endian = true ;
      fhdr.encoding = ENCODE_ASCII ;

      #if DEBUG_WMDATA != 0
      gString gsdbg ;                        // text formatting
      if ( dbg.is_open() )
         dbg << "mptEncodePopFrame\n-----------------\n" ;
      #endif   // DEBUG_WMDATA

      //* POPM frame *
      if ( popPtr->popdata != false )
      {
         //* Encode POPM frame data *
         gs = popPtr->popEmail ;
         gs.copy( obuff, gs.utfbytes() ) ;
         short indx = gs.utfbytes() ;
         obuff[indx++] = popPtr->popStar ;
         indx += popPtr->intConv( popPtr->popCount, &obuff[indx] ) ;

         //* Update frame header, encode it and write it to target *
         gs = POP_TAG ;             // set the frame ID
         gs.copy( fhdr.frame_id, gs.gschars() ) ;
         fhdr.frame_size = indx ;   // insert frame size into frame header
         mptEncodeFrameHeader ( fhdr, fhbuff ) ;
         tofs.write( fhbuff, framehdrCNT ) ;
         outBytes += framehdrCNT ;

         //* Write frame data to target *
         tofs.write( obuff, indx ) ;
         outBytes += indx ;

         #if DEBUG_WMDATA != 0
         if ( dbg.is_open() )
         {
            gsdbg.compose( "%s (%d bytes):\n"
                           "  '%s'  star:%02hhu  count:%d\n",
                           fhdr.frame_id, &fhdr.frame_size,
                           popPtr->popEmail, &popPtr->popStar, &popPtr->popCount ) ;
            dbg << gsdbg.ustr() ;
         }
         #endif   // DEBUG_WMDATA
      }

      //* PCNT frame *
      if ( popPtr->cntdata != false )
      {
         //* Encode PCNT frame data *
         short indx = popPtr->intConv( popPtr->playCount, obuff ) ;

         //* Update frame header, encode it and write it to target *
         gs = CNT_TAG ;          // set the frame ID
         gs.copy( fhdr.frame_id, gs.gschars() ) ;
         fhdr.frame_size = indx ;   // insert frame size into frame header
         mptEncodeFrameHeader ( fhdr, fhbuff ) ;
         tofs.write( fhbuff, framehdrCNT ) ;
         outBytes += framehdrCNT ;

         //* Write frame data to target *
         tofs.write( obuff, indx ) ;
         outBytes += indx ;

         #if DEBUG_WMDATA != 0
         if ( dbg.is_open() )
         {
            gsdbg.compose( "%s (%d bytes):\n"
                           "  count:%d\n",
                           fhdr.frame_id, &fhdr.frame_size, &popPtr->playCount ) ;
            dbg << gsdbg.ustr() ;
         }
         #endif   // DEBUG_WMDATA
      }

      #if DEBUG_WMDATA != 0
      if ( dbg.is_open() )
         dbg << endl ;
      #endif   // DEBUG_WMDATA
   }
   return outBytes ;

}  //* End mptEncodePopFrame()


#undef DEBUG_EMDATA
#undef DEBUG_WMDATA

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

