//******************************************************************************
//* File       : TagOgg.cpp                                                    *
//* Author     : Mahlon R. Smith                                               *
//*              Copyright (c) 2016-2020 Mahlon R. Smith, The Software Samurai *
//*                  GNU GPL copyright notice located in Taggit.hpp            *
//* Date       : 19-Sep-2020                                                   *
//* Version    : (see AppVersion string)                                       *
//*                                                                            *
//* Description: This module supports parsing, display and modification of     *
//*              Ogg/Vorbis audio files.                                       *
//*                                                                            *
//******************************************************************************
//* Notes on Ogg/Vorbis formatting:                                            *
//*  1) The first page of an Ogg stream contains the Identification header.    *
//*       (https://xiph.org/vorbis/doc/Vorbis_I_spec.html#x1-620004.2.1)       *
//*     This page is exactly 58 bytes in length.                               *
//*     00-03   "OggS"                                                         *
//*     04      stream structure version (always 00h)                          *
//*     05      header type (always 02h i.e. begin stream)                     *
//*     06-13   granpos (always zero)                                          *
//*     14-17   stream serial number                                           *
//*     18-21   page sequence number (zero)                                    *
//*     22-25   page checksum                                                  *
//*     26      segment count (01h)                                            *
//*     27      segment table (number of bytes in ID header: 1Eh)              *
//*     28      packet type (01h for Identification header)                    *
//*     29-34   'v' 'o' 'r' 'b' 'i' 's'                                        *
//*     35-38   vorbis version (00h for Vorbis I)                              *
//*     39      audio channels                                                 *
//*     40-43   sample rate                                                    *
//*     44-47   bitrate maximum                                                *
//*     48-51   bitrate nominal                                                *
//*     52-55   bitrate minimum                                                *
//*     56      blocksize 0 (top 4 bits)                                       *
//*             blocksize 1 (bottom 4 bits)                                    *
//*     57      framing flag (1 byte, in LSb)                                  *
//*                                                                            *
//*  2) The second page of an Ogg stream contains the Comment header and the   *
//*     Setup header. (https://xiph.org/vorbis/doc/framing.html)               *
//*     00-26   Page Header                                                    *
//*             00-03   "OggS"                                                 *
//*             04      stream structure version (always 00h)                  *
//*             05      flag byte ('ptype')                                    *
//*                     0x01: unset = fresh packet                             *
//*                             set = continued packet                         *
//*                     0x02: unset = not first page of logical bitstream      *
//*                             set = first page of logical bitstream (bos)    *
//*                     0x04: unset = not last page of logical bitstream       *
//*                             set = last page of logical bitstream (eos)     *
//*                     (remaining bits reserved)                              *
//*             06-13   'granpos'    absolute granule position                 *
//*                                  (little-endian, LSByte first)             *
//*             14-17   'streamser'  stream serial number (little-endian)      *
//*             18-21   'pageseq     page sequence number (little-endian)      *
//*             22-25   CRC (cksum)  page checksum (little-endian)             *
//*             26      'segcount'   page segment count (2-255 segments)       *
//*     27-nn           Segment Table indicates size of each segment           *
//*     nn+1 (7 bytes)  Packet Header                                          *
//*     nn+8 (4 bytes)  Vendor Record size                                     *
//*     nn+12           Vendor Record string                                   *
//*     mm   (4 bytes)  Comment count                                          *
//*     mm+4            Comment Vectors                                        *
//*                     Note that length of last comment segment               *
//*                       is always < 255 bytes.                               *
//*     pp              Framing Bit (1 byte, in LSb)                           *
//*     ss              Setup Header (copied, not decoded here)                *
//*                     Note that length of last setup segment                 *
//*                       is always < 255 bytes.                               *
//*     qq              Framing Bit (1 byte, in LSb)                           *
//*     (CRC is calculated from the top of Page 2 through the end of the)      *
//*     (setup data.                                                    )      *
//*  3) Page 3+ - audio stream data                                            *
//*                                                                            *
//*  -   -  -   -  -   -  -   -  -   -  -   -  -   -  -   -  -   -  -   -  -   *
//* Programmer's Note: The code here was created only from reading the         *
//* specification. We did not peek at the libVorbis source code on the grounds *
//* that a project should be fun and a way of exercising our logical thought   *
//* processes, and not just a way of getting things done.                      *
//*                                                                            *
//* The audio files with modified comment data were tested for playback and    *
//* comment decoding using both VLC Media Player and Totem ("Videos").         *
//* We refuse to install the poorly-designed Rhythmbox media player on any of  *
//* our systems, but some of the beta testers do, and they report successful   *
//* playback using Rhythmbox.                                                  *
//*                                                                            *
//* ========================================================================== *
//* From: http://age.hobba.nl/audio/mirroredpages/ogg-tagging.html             *
//* This list of tags and their implementation is from a community-based       *
//* (non-authoritative) source. In general, it echos the id2v3 (MP3) tag       *
//* fields. Currently, we use ONLY tag names actually defined by the standard. *
//*                   --  --  --  --  --  --  --  --  --                       *
//* Singleton tags, which should only appear once. If one of these tags        *
//* appears more than once, its last appearance should be displayed if there   *
//* is only room to display one instance of the tag.                           *
//*                                                                            *
//* ALBUM                                                                      *
//*     if appropriate, an album name                                          *
//* ARTIST                                                                     *
//*     for information to be displayed on systems with limited display        *
//*     capabilities. it is not a replacement for the ENSEMBLE and PERFORMER   *
//*     tags, but typically will summarize them.                               *
//* PUBLISHER                                                                  *
//*     who publishes the disc the track came from                             *
//* COPYRIGHT                                                                  *
//*     who holds copyright to the track or disc the track is on               *
//* DISCNUMBER                                                                 *
//*     if part of a multi-disc album, put the disc number here                *
//* ISRC                                                                       *
//*     this number lets you order a CD over the phone from a record shop.     *
//* EAN/UPN                                                                    *
//*     there may be a barcode on the CD; it is most likely an EAN or UPN      *
//*     (Universal Product Number).                                            *
//* LABEL                                                                      *
//*     the record label or imprint on the disc                                *
//* LABELNO                                                                    *
//*     record labels often put the catalog number of the source media         *
//*     somewhere on the packaging. use this tag to record it.                 *
//* LICENSE                                                                    *
//*     the license, or URL for the license the track is under. for instance,  *
//*     the Open Audio license.                                                *
//* OPUS                                                                       *
//*     the number of the work; eg, Opus 10, BVW 81, K6                        *
//* SOURCEMEDIA                                                                *
//*     the recording media the track came from. eg, CD, Cassette, Radio       *
//*     Broadcast, LP, CD Single                                               *
//* TITLE                                                                      *
//*     "the work", whether a symphony, or a pop song                          *
//* TRACKNUMBER                                                                *
//*     the track number on the CD                                             *
//* VERSION                                                                    *
//*     Make sure you don't put DATE or LOCATION information in this tag.      *
//*     "live", "acoustic", "Radio Edit" "12 inch remix" might be typical      *
//*     values of this tag                                                     *
//* ENCODED-BY                                                                 *
//*     The person who encoded the Ogg file. May include contact information   *
//*     such as email address and phone number.                                *
//* ENCODING                                                                   *
//*     Put the settings you used to encode the Ogg file here.                 *
//*     This tag is NOT expected to be stored or returned by cddb type         *
//*     databases. It includes information about the quality setting,          *
//*     bit rate, and bitrate management settings used to encode the Ogg.      *
//*     It also is used for information about which encoding software was      *
//*     used to do the encoding.                                               *
//*                                                                            *
//* The remaining tags are multiples; if they are present more than once,      *
//* all their occurances are considered significant.                           *
//*                                                                            *
//* COMPOSER                                                                   *
//*     composer of the work. eg, Gustav Mahler                                *
//* ARRANGER                                                                   *
//*     the person who arranged the piece, eg, Ravel                           *
//* LYRICIST                                                                   *
//*     the person who wrote the lyrics, eg Donizetti                          *
//* AUTHOR                                                                     *
//*     for text that is spoken, or was originally meant to be spoken, the     *
//*     author, eg JRR Tolkien                                                 *
//* CONDUCTOR                                                                  *
//*     conductor of the work; eg Herbert von Karajan. choir directors would   *
//*     also use this tag.                                                     *
//* PERFORMER                                                                  *
//*     individual performers singled out for mention; eg, Yoyo Ma (violinist) *
//* ENSEMBLE                                                                   *
//*     the group playing the piece, whether orchestra, singing duo, or rock   *
//*     and roll band.                                                         *
//* PART                                                                       *
//*     a division within a work; eg, a movement of a symphony. Some tracks    *
//*     contain several parts. Use a single PART tag for each part contained   *
//*     in a track. ie, PART="Oh sole mio"                                     *
//* PARTNUMBER
//*     The part number goes in here. You can use any format you like, such as *
//*      Roman numerals, regular numbers, or whatever. The numbers should be   *
//*      entered in such a way that an alphabetical sort on this tag will      *
//*      correctly show the proper ordering of all the oggs that contain the   *
//*      contain the piece of music.                                           *
//* GENRE                                                                      *
//*     like the genre tag from the cddb but without the limitations. You can  *
//*     put any genre you want in this tag. If you think "Pink Floyd" are a    *
//*     genre unto themselves, say so here. For crossover works, or ambiguous  *
//*     works, use as many GENRE tags as you think it takes to describe the    *
//*     styles used.                                                           *
//* DATE                                                                       *
//*     date or date-time of relevance to the track. The date must be in       *
//*     ISO 8601 format, but may be followed by a space character, then any    *
//*     text you wish, including the same date in any other format. None of    *
//*     the alternate formats in ISO 8601 may be used. Only the primary format *
//*     in ISO 8601 is to be used.                                             *
//*     q.v. http://www.cl.cam.ac.uk/~mgk25/iso-time.html eg, DATE="1999-08-16 *
//*     (recorded)" or DATE="1999-08-16 recorded August 16, 1999"              *
//* LOCATION                                                                   *
//*     location of recording, or other location of interest                   *
//* COMMENT                                                                    *
//*     additional comments of any nature.                                     *
//*                                                                            *
//* Note: See also "METADATA_BLOCK_PICTURE" and "COVERART" comment names       *
//*       described in the header of the oggReadImageVector() method.          *
//*                                                                            *
//******************************************************************************


//****************
//* Header Files *
//****************
#include "Taggit.hpp"         // Taggit-class definitions and data,
                              // plus general definitions and NcDialogAPI definition
#include "CRC_Gen.hpp"        // Definition of CRC generator class


//**************************************
//* Set to non-zero for debugging only *
//**************************************
#define DEBUG_WMDATA (0)      // Debug WriteMetadata_OGG()
#define DEBUG_EMDATA (0)      // Debug ExtractMetadata_OGG()

//* Used during development of OGG image support *
#define OGG_IMAGES (1)

//**********************
//* Non-member methods *
//**********************
static TagFields oggMapField ( const gString& fldName ) ;
static bool   oggDecodeID_Header ( ifstream&ifs, Ogg_ID& oid, 
                                 Ogg_PageHdr& oph, Ogg_Packet& opkt ) ;
static void   oggDecodePageHeader ( ifstream& ifs, Ogg_PageHdr& oph ) ;
static UINT32 oggDecodePacketHeader ( ifstream& ifs, Ogg_Packet& opkt ) ;
static UINT32 oggDecodeVendorRecord ( ifstream& ifs, Ogg_Comment& vrogc ) ;
static UINT32 oggReadSegmentTable ( ifstream& ifs, Ogg_Segments& seg ) ;
static void   oggEncodePageHeader ( Ogg_Segments& seg, const Ogg_PageHdr& oph ) ;
static void   oggAdjustSegmentTable ( Ogg_Segments& seg, ofstream& dbg ) ;
static void   oggEncodeSegmentTable ( Ogg_Segments& seg ) ;
static void   oggEncodePacketHeader ( Ogg_Segments& seg ) ;
static void   oggEncodeVendorRecord ( Ogg_Segments& seg ) ;
static void   oggEncodeCommentVectors ( Ogg_Segments& seg, const EmbeddedImage* eiPtr ) ;
static void   oggEncodeSetupHeader ( ifstream& ifs, Ogg_Segments& seg ) ;
static UINT32 oggCalculateCRC ( Ogg_Segments& seg ) ;
#if OGG_IMAGES != 0
static UINT32 oggImageVectorSize ( const EmbeddedImage* eiPtr ) ;
#endif   // OGG_IMAGES
static bool   oggWriteMetadata ( ofstream& ofs, Ogg_Segments& seg ) ;


#if DEBUG_WMDATA != 0   // DEBUG ONLY: WriteMetadata_OGG()
static void dbgDisplaySegmentTable ( ofstream& dbg, const Ogg_Segments& seg ) ;
#endif                  // DEBUG_WMDATA

//**************
//* Local data *
//**************
//* Duplicate comment titles are allowed and encouraged under the OGG/Vorbis   *
//* specification. Unfortunately, there is only one display field for a given  *
//* title. Therefore we must concatenate the duplicate entries for display.    *
//* This sequence is a delimiter for data scanned from multiple source fields  *
//* with the same field title.                                                 *
//* Example: If a file contains multiple "DESCRIPTION" entries they will be    *
//* concatenated for display:                                                  *
//*    DESCRIPTION: Howdy Pardner!                                             *
//*    DESCRIPTION: How are you doin'?                                         *
//*    is displayed as: Howdy Pardner! /How are you doin'?                     *
//* For output to a target file, these delimited messages can be separated and *
//* written individually in the same format from which they were scanned.      *
//* See the oggMapField() method for more information.                         *
const wchar_t* oggFIELD_DELIMITER = L" /" ;
const short oggFD_BYTES = 2 ;

//* Basic list of OGG Comment-field names *
const short OFMA_COUNT = 23 ;    // number of pre-defined field names
const short OFMB_COUNT = 10 ;    // number of search substrings defined
class OggFieldMap
{
   public:
   const wchar_t* const fName ;  // OGG field name
   const short fLen ;            // number of characters to compare
   const TagFields fIndex ;      // display-field index
} ;

static OggFieldMap ofma[OFMA_COUNT] = 
{
   //* Basic, authorized Comment-field names for OGG/Vorbis I *
   { L"TITLE",          5, tfTit2 },   // Title
   { L"ARTIST",         6, tfTpe1 },   // Artist
   { L"ALBUM",          5, tfTalb },   // Album
   { L"TRACKNUMBER",   11, tfTrck },   // Track
   { L"GENRE",          5, tfTcon },   // Genre "(nn) xxx..."
   { L"PERFORMER",      9, tfTpe2 },   // Guest Artist
   { L"ORGANIZATION",  12, tfTpub },   // Publisher, Record Label
   { L"VERSION",        7, tfTpe4 },   // Remixed By
   { L"COPYRIGHT",      9, tfTcop },   // Copyright "YYYY xxx..."
   { L"LICENSE",        7, tfTown },   // Owner/Licensee
   { L"DESCRIPTION",   11, tfTxxx },   // Freeform comment (unmapped fields should default to this)
   { L"DATE",           7, tfTrda },   // Freeform recording date(s)
   { L"CONTACT",        7, tfTrsn },   // Contact ('Internet radio station name')
   { L"LOCATION",       8, tfTsee },   // Settings for encoding (location, studio)
   { L"ISRC",           4, tfTsrc },   // ISRC recording code
   //* The following comment names are proposed additions or defacto standards.*
   //* This list is posted on the XiphWiki page: xiph.org.                     *
   //*   (See also: http://age.hobba.nl/audio/mirroredpages/ogg-tagging.html)  *
   { L"COMPOSER",       8, tfTcom },   // Composer
   { L"TRACKTOTAL",    10, tfTrck },   // Number of tracks (complements TRACKNUMBER)
                                       // (under MP3, this is combined with TRACKNUMBER: trk/tot)
   { L"DISCNUMBER",    10, tfTpos },   // Sequence number of disc in set
                                       // (combine with "DISCTOTAL": dscnum/dsctot)
   { L"DISCTOTAL",      9, tfTpos },   // Total number of discs in set
   { L"ENCODE",         6, tfTenc },   // Encoder (covers both "ENCODED-BY" and "ENCODER")
   { L"SOURCEMEDIA",   11, tfTmed },   // Media type
   { L"PRODUCTNUMBER", 13, tfTrso },   // UPC code? ('Internet radio station owner')
   { L"RATING",         6, tfTsrc },   // Popularity rating (1-5 stars)
} ;                                 
static OggFieldMap ofmb[OFMB_COUNT] = 
{
   { L"LYRI",           4, tfText },   // Lyricist
   { L"YEAR",           4, tfTyer },   // Year recorded
   { L"COMPOSE",        7, tfTcom },   // Composer(s)
   { L"TIME",           4, tfTlen },   // playback time
   { L"LENGTH",         6, tfTlen },   // audio data size (bytes)
   { L"SIZE",           4, tfTsiz },   //  "                 "
   { L"SUBTITLE",       8, tfTit3 },   // Subtitle/description refinement
   { L"CONDUCT",        7, tfTpe2 },   // Conductor/performer refinement
   { L"WEB",            3, tfTrsn },   // Download site, internet radio station, etc.
   { L"NET",            3, tfTrsn },   //  "                 "
} ;

//* Embedded image comment name (APIC in MP3 world) ('fIndex' field not used) *
const short OFMI_COUNT = 2 ;     // number of pre-defined field names
static OggFieldMap ofmi[OFMI_COUNT] = 
{
   { L"METADATA_BLOCK_PICTURE", 22, tfCOUNT },  // standard comment name
   { L"COVERART",                8, tfCOUNT },  // ad-hoc and obsolete comment name
} ;

//* Used in the case of a corrupted Comment header, *
//* i.e. if the Vendor record invalid or missing.   *
static const char* defaultVENDOR = "Xiph.Org libVorbis I (Taggit 2020)" ;
static const short defaultVENDOR_BYTES = 34 ;



//*************************
//*  ExtractMetadata_OGG  *
//*************************
//******************************************************************************
//* Read the media file and extract the metadata tag fields.                   *
//*                                                                            *
//* Input  : si  : index into list of source data                              *
//*                                                                            *
//* Returns: 'true' if valid OGG header,                                       *
//*          'false' invalid header record, file not found, access error, etc. *
//******************************************************************************
//* Ogg/Vorbis 'I' files contain three metadata headers:                       *
//*  1) Identification - identifies the file as Ogg/Vorbis format.             *
//*     We decode this to verify that the file can be decoded.                 *
//*  2) Comments - text fields describing the audio data                       *
//*     We decode and report the fields within the Comments header.            *
//*  3) Setup - codec setup and bitstream definitions                          *
//*     This is encoded binary data which is not decoded here.                 *
//*                                                                            *
//*                                                                            *
//* The fields of the Identification header:                                   *
//*     [vorbis_version] = read 32 bits as unsigned integer            (0)     *
//*     [audio_channels] = read 8 bit integer as unsigned              (>0)    *
//*     [audio_sample_rate] = read 32 bits as unsigned integer         (>0)    *
//*     [bitrate_maximum] = read 32 bits as signed integer                     *
//*     [bitrate_nominal] = read 32 bits as signed integer                     *
//*     [bitrate_minimum] = read 32 bits as signed integer                     *
//*       a) The fields are meaningful only when greater than zero.            *
//*          None set indicates the encoder does not care to speculate.        *
//*       b) All three fields set to the same value implies a fixed rate, or   *
//*          tightly bounded, nearly fixed-rate bitstream.                     *
//*       c) Only nominal set implies a VBR or ABR stream that averages the    *
//*          nominal bitrate.                                                  *
//*       d) Maximum and or minimum set implies a VBR bitstream that obeys the *
//*          bitrate limits.                                                   *
//*     [blocksize_0] = power-of-2 (read 4 bits as unsigned integer)   (<=bs_1)*
//*                     [64 | 128 | 256 | 512 | 1024 | 2048 | 4096 | 8192      *
//*     [blocksize_1] = power-of-2 (read 4 bits as unsigned integer)   (>bs_0) *
//*     [framing_flag] = read one bit (see note below on framing flag)         *
//*   Note that the integer fields are defined as little-endian (LSByte first).*
//*                                                                            *
//* The Comments header begins with:                                           *
//*     a) [vendor\_length] = unsigned 32-bit integer                          *
//*     b) [vendor\_string] = UTF-8 vector of [vendor\_length] bytes.          *
//*     c) [number of comment fields] = unsigned 32-bit integer                *
//*                  The number of fields is limited to 2 to the 32nd - 1.     *
//*     d) Zero or more comment fields as specified by (c)                     *
//*         i) [field length] = unsigned 32-bit integer (little-endian)        *
//*                         This (apparently) includes the FIELDNAME, equals   *
//*                         character (0x3D) and the UTF-8 text.               *
//*        ii) [field contents] = FIELDNAME + '=' + UTF-8(unterminated) string *
//*     e) [framing\_bit] = read a single bit as boolean                       *
//*        If ( [framing\_bit] unset or end-of-packet ) then ERROR             *
//*        Note that as stated, this is completely insane because a single bit *
//*        cannot be read from a byte stream. This is the problem with         *
//*        letting morons write the documentation. In practice, this is the    *
//*        LSbit of the byte.                                                  *
//*     f) End of Comments header                                              *
//*                                                                            *
//* The individual fields contained the Comments header are of the form:       *
//*           nnnnFIELDNAME=<field contents>                                   *
//*     a) 'nnnn' is a 4-byte, little-endian integer indicating the length of  *
//*        the text record.                                                    *
//*     b) The 'FIELDNAME' may be any 7-bit ASCII sequence.                    *
//*        The field name is case-insensitive and may consist of ASCII 0x20    *
//*        through 0x7D, 0x3D (’=’) excluded. ASCII 0x41 through 0x5A inclusive*
//*        (characters A-Z) is to be considered equivalent to ASCII 0x61       *
//*        through 0x7A inclusive (characters a-z).                            *
//*                                                                            *
//*        The basic list includes:                                            *
//*         TITLE : Track/Work name                                            *
//*         VERSION : The version field may be used to differentiate multiple  *
//*                   versions of the same track title in a single collection. *
//*                  (e.g. remix info)                                         *
//*         ALBUM : The collection name to which this track belongs            *
//*         TRACKNUMBER : The track number of this piece if part of a specific *
//*                       larger collection or album.                          *
//*         ARTIST : The artist generally considered responsible for the work. *
//*                  In popular music this is usually the performing band or   *
//*                  singer. For classical music it would be the composer.     *
//*                  For an audio book it would be the author of the original  *
//*                  text.                                                     *
//*         PERFORMER : The artist(s) who performed the work. In classical     *
//*                     music this would be the conductor, orchestra, soloists.*
//*                     In an audio book it would be the actor who did the     *
//*                     reading. In popular music this is typically the same   *
//*                     as the ARTIST and is omitted.                          *
//*         COPYRIGHT : Copyright attribution, e.g., ’2001 Nobody’s Band’ or   *
//*                     ’1999 Jack Moffitt’                                    *
//*         LICENSE : License information, eg, ’All Rights Reserved’,          *
//*                   ’Any Use Permitted’, a URL to a license such as a        *
//*                   Creative Commons license                                 *
//*                   (”www.creativecommons.org/blahblah/license.html”) or the *
//*                   EFF Open Audio License (’distributed under the terms of  *
//*                   the Open Audio License.                                  *
//*                   see http://www.eff.org/IP/Open_licenses/eff_oal.html     *
//*                   for details’), etc.                                      *
//*         ORGANIZATION : Name of the organization producing the track (i.e.  *
//*                        the ’record label’)                                 *
//*         DESCRIPTION : A short text description of the contents             *
//*         GENRE : A short text indication of music genre                     *
//*         DATE : Date the track was recorded                                 *
//*         LOCATION : Location where track was recorded                       *
//*         CONTACT : Contact information for the creators or distributors of  *
//*                   the track. This could be a URL, an email address, the    *
//*                   physical address of the producing label.                 *
//*         ISRC : International Standard Recording Code for the track; see    *
//*                the ISRC intro page for more information on ISRC numbers.   *
//*     b) The '=' character (0x3D) terminates the 'FIELDNAME'.                *
//*     c) The 'field contents' are UTF-8 encoded text (NOT null terminated).  *
//*        Length is limited to 2 to the 32nd - 1 bytes; HOWEVER, the          *
//*        recommendation is a length of no more than a short paragraph (about *
//*        1,000 bytes).                                                       *
//*                                                                            *
//******************************************************************************

bool Taggit::ExtractMetadata_OGG ( short si )
{
   ofstream dbg ;

   #if DEBUG_EMDATA != 0
   const char* BadOgg = "Error: Invalid Ogg/Vorbis format.\n" ;
   const char* noComment =  "No Ogg metadata available.\n" ;
   const short CNAME_COLS = 18 ; // output column alignment
   gString dbgFile( this->tData.sf[si].sfPath ) ;
   short indx = (dbgFile.findlast( L'/' )) + 1 ;
   dbgFile.limitChars( indx ) ;
   dbgFile.append( "emd.txt" ) ;
   dbg.open( dbgFile.ustr(), 
                 ofstream::out | (si == ZERO ? ofstream::trunc : ofstream::app) ) ;
   if ( dbg.is_open() )
   {
      if ( si == ZERO )
         dbg << "Debug ExtractMetadata_OGG()\n"
                "---------------------------\n" ;
      dbg << "si(" << si << ") '" 
          << this->tData.sf[si].sfName << "'\n" << endl ;
   }
   #endif   // DEBUG_EMDATA

   gString gsOut, gstmp ;
   bool status = false ;

   //* Open the source file *
   ifstream ifs( this->tData.sf[si].sfPath, ifstream::in ) ;
   if ( ifs.is_open() )
   {
      char ibuff[gsMAXBYTES + 4];// input buffer
      Ogg_ID oid ;               // Identification header data
      Ogg_PageHdr oph ;          // Page header data
      Ogg_Packet opkt ;          // Packet header data
      bool valid_hdr = true ;    // true if valid header record, else false

      //* Read the Ogg ID header *
      valid_hdr = oggDecodeID_Header ( ifs, oid, oph, opkt ) ;
      if ( valid_hdr )
      {
         #if DEBUG_EMDATA != 0
         if ( dbg.is_open() )
         {
            gsOut.compose( "Page 1 Header\n"
                           "--------------------\n"
                           "Packet Header : %02hhX '%s'\n"
                           "Ogg/Vorbis tag:'%s'\n"
                           "Version       : %02hhX\n"
                           "Page Type     : %02hhXh\n"
                           "Segment Count : %hhu\n"
                           "granpos       : %016llXh\n"
                           "StreamID      : %08Xh\n"
                           "Page Sequence : %08Xh\n"
                           "Checksum      : %08Xh\n"
                           "Segment Bytes : %u\n"
                           "Valid Page Hdr: %s\n",
                           &opkt.ptype, opkt.pid, 
                           oph.tag, &oph.version, &oph.ptype, &oph.segcount,
                           &oph.granpos, &oph.streamser, &oph.pageseq, 
                           &oph.cksum, &oph.segBytes,
                           (char*)(valid_hdr ? "true" : "false") ) ;
            dbg << gsOut.ustr() << endl ;

            gsOut.compose( "Ogg/Vorbis ID Header\n"
                           "--------------------\n"
                           "version   : %u\n"
                           "channels  : %hhu\n"
                           "samprate  : %d\n"
                           "bitratemax: %d\n"
                           "bitratenom: %d\n"
                           "bitratemin: %d\n"
                           "blksize0  : %u\n"
                           "blksize1  : %u\n"
                           "frameflag : %hhx\n",
                           &oid.version, &oid.channels, &oid.samprate,
                           &oid.bitratemax, &oid.bitratenom, &oid.bitratemin,
                           &oid.blksize0, &oid.blksize1, &oid.frameflag ) ;
            dbg << gsOut.ustr() << endl ;
         }
         #endif   // DEBUG_EMDATA

         //* ID header is valid, continue to Comment Header *
         //* Note that an error in the Comment header is    *
         //* defined as a non-fatal error.                  *
         status = true ;         // file is OGG/Vorbis format

         //* Read header of second page *
         oggDecodePageHeader ( ifs, oph ) ;
         //* Read the segment table (0-255 bytes) *
         //* and count the total segment bytes.   *
         Ogg_Segments seg ;
         seg.segCount = oph.segcount ;
         oph.segBytes = oggReadSegmentTable ( ifs, seg ) ;

         //* Accumulate total segment bytes actually read. *
         //* This should match seg.segBytes.               *
         UINT32 segBytesRead = ZERO ;

         //* Read Page 2 packet header *
         segBytesRead += oggDecodePacketHeader ( ifs, seg.opkt ) ;

         //* Validate Page 2 header.                            *
         //* (For the comment header, a fail is only a warning, *
         //* (but it means that we can't edit the comment data. *
         gstmp = oph.tag ;
         gString gstmp2( seg.opkt.pid ) ;
         if ( (gstmp.compare( Ogg_Tag ) == ZERO) && (oph.version == ZERO) &&
              (seg.opkt.ptype == 3) && (gstmp2.compare( Ogg_PktHdr ) == ZERO) )
         {
            //* Read the "Vendor" record. *
            segBytesRead += oggDecodeVendorRecord ( ifs, seg.vendor ) ;

            //* Get the number of comment vectors (records). *
            ifs.read ( ibuff, 4 ) ;
            segBytesRead += ifs.gcount() ;
            if ( (ifs.gcount()) == 4 )
               seg.oldCVectors = seg.vendor.intConv( (UCHAR*)ibuff ) ;

            #if DEBUG_EMDATA != 0   // FOR DEBUGGING ONLY
            if ( dbg.is_open() )
            {
               gsOut.compose(
                  "Page 2 Header\n"
                  "--------------------\n"
                  "Packet Header : %02hhX '%s'\n"
                  "Ogg/Vorbis tag:'%s'\n"
                  "Version       : %02hhXh\n"
                  "Page Type     : %02hhXh\n"
                  "Segment Count : %hhu\n"
                  "granpos       : %016llXh\n"
                  "StreamID      : %08Xh\n"
                  "Page Sequence : %08Xh\n"
                  "Checksum      : %08Xh\n"
                  "Segment Bytes : %u\n"
                  "Valid Page Hdr: true\n\n"
                  "Vendor Record : (%u) %s\n",
                  &seg.opkt.ptype, seg.opkt.pid, 
                  oph.tag, &oph.version, &oph.ptype, &oph.segcount,
                  &oph.granpos, &oph.streamser, &oph.pageseq, 
                  &oph.cksum, &seg.segBytes,
                  &seg.vendor.clen, seg.vendor.ctxt ) ;
               dbg << gsOut.ustr() << endl ;
            }
            #endif                  // DEBUG_EMDATA

            //* Read and report each vector *
            if ( seg.oldCVectors > ZERO )
            {
               seg.cvAlloc( seg.oldCVectors ) ;
               seg.oldCBytes = this->oggReadCommentVectors ( si, ifs, seg.oldCVectors, 
                                                             dbg, seg.newComm ) ;
               segBytesRead += seg.oldCBytes ;

               #if DEBUG_EMDATA != 0   // FOR DEBUGGING ONLY
               if ( dbg.is_open() )
               {
                  gsOut.compose( "Total Comments: %d", &seg.oldCVectors ) ;
                  dbg << gsOut.ustr() << endl ;
               }
               #endif                  // DEBUG_EMDATA

               TagFields tfMap ;    // for mapping comment data to a display field
               short cIndex ;       // index of UTF-8 data

               for ( int v = ZERO ; v < seg.oldCVectors ; ++v )
               {
                  //* Special Note: If the vector was an Image Vector, then *
                  //* the value returned in 'ctxt' DOES NOT contain the '='.*
                  //* Therefore, the following test will fail, and the      *
                  //* unneeded vector will be discarded.                    *
                  // Programmer's Note: The specification states that the
                  // comment name is ASCII, so 1 byte == 1 character.
                  // The comment text however, is UTF-8.
                  gstmp = seg.newComm[v].ctxt ;
                  if ( (cIndex = gstmp.find( L'=' )) >= ZERO ) // if valid comment format
                  {
                     gstmp.limitChars( cIndex ) ;
                     tfMap = oggMapField ( gstmp ) ;
                     gsOut = &seg.newComm[v].ctxt[cIndex + 1] ;

                     //* OGG comment names are not required to be unique,      *
                     //* and in fact duplication is encouraged. Unfortunately, *
                     //* we have only one display field, per field name so we  *
                     //* must combine the duplications for display.            *
                     if ( (*this->tData.sf[si].sfTag.field[tfMap]) != NULLCHAR )
                     {
                        //* Append new data to existing field data *
                        gsOut.append( "%S%S", oggFIELD_DELIMITER, 
                                      this->tData.sf[si].sfTag.field[tfMap] ) ;

                        #if DEBUG_EMDATA != 0 && false
                        gString gsdbg( this->tData.sf[si].sfTag.field[tfMap] ) ;
                        dbg << "Duplicate Field Title\n"
                            << "    '" << gsdbg.ustr() << "'\n"
                            << "to: '" << gsOut.ustr() << endl ;
                        #endif   // DEBUG_EMDATA
                     }

                     //* Write the captured comment to its display field.*
                     gsOut.copy( this->tData.sf[si].sfTag.field[tfMap], gsMAXCHARS ) ;

                     #if DEBUG_EMDATA != 0
                     gstmp.append( L' ' ) ;
                     while ( gstmp.gschars() < CNAME_COLS )
                        gstmp.append( L'-' ) ;
                     gstmp.append( L' ' ) ;
                     do
                     {
                        gsOut.append( L' ' ) ;
                     }
                     while ( (gsOut.gscols()) < 30 ) ;
                     dbg << gstmp.ustr() << gsOut.ustr() << "(" 
                         << TextFrameID[tfMap] << ")\n" ;
                     #endif   // DEBUG_EMDATA
                  }
                  #if DEBUG_EMDATA != 0
                  else
                  {
                     if ( (gstmp.compare( ofmi[0].fName )) == ZERO )
                        dbg << "(Image Vector scanned and saved)\n" ;
                     else
                        dbg << "Non-standard encoding: '" << ibuff << "'\n" ;
                  }
                  #endif   // DEBUG_EMDATA
               }
            }
            else
            {
               #if DEBUG_EMDATA != 0
               dbg << noComment << endl ;
               #endif   // DEBUG_EMDATA
            }
         }
         //* Page 2 header is corrupted *
         else
         {
            #if DEBUG_EMDATA != 0
            dbg << BadOgg << endl ;
            #endif   // DEBUG_EMDATA

            //* Create an empty comment header. *
            // WHAT CAN WE DO TO SALVAGE the Setup header and audio data?
            // a) create a default page header
            // b) create a default packet header
            // c) create a default vendor record (see 'defaultVendor' above)
            // d) create a segmentTable[0] entry for packet header and vendor 
            //    record, but with no comments
            // e) find the segment table entries for the Setup Header
            //    and append them
            // f) format the first segment
            // g) find the Setup Header data and append it after first segment
         }
      }
      else
      {
         #if DEBUG_EMDATA != 0
         dbg << BadOgg << endl ;
         #endif   // DEBUG_EMDATA
      }

      ifs.close() ;           // close the source file
   }

   #if DEBUG_EMDATA != 0
   dbg << endl ;
   dbg.close() ;              // close the debugging file
   #endif   // DEBUG_EMDATA

   return status ;

}  //* End ExtractMetadata_OGG() *

//*************************
//*   WriteMetadata_OGG   *
//*************************
//******************************************************************************
//* Write edited metadata to target file, or alternatively write only the      *
//* headers and audio data.                                                    *
//*                                                                            *
//* Input  : si       : index into list of source data                         *
//*          filter   : (member of enum WriteFilter)                           *
//*                     determines which metadata will be written to target    *
//*          preserve : (optional, 'false' by default)                         *
//*                     If specified, retain the backup file after the target  *
//*                     has been successfully written.                         *
//*                                                                            *
//* Returns: 'true' if file successfully written                               *
//*          'false' if: a) invalid source header                              *
//*                      b) unable to encode the data                          *
//*                      c) file not found, access error, out-of-storage-space,*
//*                         etc.                                               *
//******************************************************************************
//* Notes:                                                                     *
//* 1) Rename source file:                                                     *
//*    a) If 'altName' NOT specified, then rename source as backup             *
//*       If a file of that name already exists, it will be deleted before     *
//*       the rename.                                                          *
//*    b) If 'altName' IS specified, then skip this step.                      *
//*                                                                            *
//* 2) Open the source file and read the 'Common Header Packet' and            *
//*    'Identification Header' (Page 1) data.                                  *
//*    These data were validatated on startup, so no need to re-validate.      *
//*                                                                            *
//* 3) Open the target file and write the Page 1 data unmodified.              *
//*                                                                            *
//* 4) Create a temporary buffer large enough to hold 256 x 256 bytes i.e. the *
//*    maximum size of the Page 2 data (minus headers and segment table).      *
//*    For technical reasons, the specification limits the maximum size of the *
//*    page to 65307 bytes (just under 64KB).
//*    a) The entire Page 2 stream is validated by a CRC checksum, so it is    *
//*       necessary to read the entire page (discarding any existing comment   *
//*       vectors) into the buffer in order to recalculate the checksum after  *
//*       modifications have been made.                                        *
//*    b) The most likely modifications:                                       *
//*       - Set the checksum field to ZERO for recalculating the checksum.     *
//*       - Modify the size of the segment(s) which hold the comment vectors.  *
//*         This also means updating the segment table for those segments.     *
//*       - Add or remove one or more segments to accomodate changes in the    *
//*         size of the comment data.                                          *
//*         - This also means adding or removing corresponding entries in the  *
//*           segment table. (The segment count cannot exceed 255.)            *
//*    c) IMPORTANT NOTE: It is possible, though unlikely that the comment     *
//*       data will extend beyond Page 2. This is indicated by the last entry  *
//*       in the segment table being exactly 255, meaning that the last        *
//*       segment containing comments is on the next page. While this is       *
//*       allowed by the specification, it is very unlikely to happen because  *
//*       Page 2 segments can include up to 64KBytes, while the combined size  *
//*       of all comments is unlikely to be more than 4KBytes. For this reason,*
//*       if it happens, we will note it in the debugging log (if enabled),    *
//*       but in the current version of the application, we don't handle that  *
//*       scenario and don't report it to the user.                            *
//*                                                                            *
//* 5) Read the existing source metadata including headers, tables and         *
//*    comment vectors.                                                        *
//*    a) Packet Header: Always the same, no modifications needed (unless its  *
//*       format is invalid).                                                  *
//*    b) Page Header: CRC checksum and segment count live here and will       *
//*       likely be modified when the old data are removed and the new data    *
//*       are added.                                                           *
//*    c) Segment Table: The segment(s) which contained the original comment   *
//*       vectors may change in size and number and this must be reflected in  *
//*       the segment table.                                                   *
//*       table.                                                               *
//*    d) Comment Vectors: The new comment vectors will replace the previous   *
//*       comments, and the segments where they live will probably change      *
//*       in size, and may change in number.                                   *
//*    e) Framing Bit: Marks the end of the comment vector list.               *
//*                                                                            *
//* 6) Write the edited metadata to the file:                                  *
//*    a) If caller wants to write the edited metadata to the file, format the *
//*       new data and write it to the target file.                            *
//*    b) If caller specified to write only the audio data, then skip this     *
//*       step and write only the placeholder data (as needed).                *
//* 7) Read the source audio data and append it _verbatim_ to the target file. *
//* 8) Close both source and target files.                                     *
//* 9) Unless caller has specified otherwise, delete the backup file.          *
//*    a) Note that if 'altName' was specified, the 'preserve' is implied.     *
//*                                                                            *
//* Reading the source:                                                        *
//* -------------------                                                        *
//* The stream begins with three(3) header packets:                            *
//*  1) ID packet - This takes the entire first page (Page 1)                  *
//*  2) Comments header packet - This begins the second page (Page 2) and      *
//*     can extend for the whole page or more; however in practice, all        *
//*     will live comfortably in a small fraction of Page 2.                   *
//*  3) Setup header packet - Begins immediately after the comment packet.     *
//*     This could be on the same or succeeding page .                         *
//* All three packets must be present for proper decoding.                     *
//*                                                                            *
//* Inserting Comments:                                                        *
//* -------------------                                                        *
//* 1) Actual bytes occupied by the comment-vector data:                       *
//*    a) the Page 2 packet header,                                            *
//*    b) Vendor record                                                        *
//*    c) comment-vector count                                                 *
//*    d) the comment vectors                                                  *
//*    e) This should equal the total length of the segments used,             *
//*       minus one(1) byte. The last byte of the last segment used for        *
//*       comments is the framing bit i.e. 01h.                                *
//*    f) Special Case: If the comment data extend beyond the current page,    *
//*       then the value represents all comment data in the current page.      *
//*       This is very unlikely to happen, but the specification allows for it.*
//*       For comments to extend to the next page, the comment data would have *
//*       to be greater than the recommended page size, (4KB).                 *
//*       This is unlikely since there are only 33 defined comment names and   *
//*       if ALL were used AND each used 100 bytes that is still only          *
//*                    100 x 33 == 3,300 bytes plus headers                    *
//*       This would easily fit into 4KB.                                      *
//*                                                                            *
//* Programer's Note: What does this mean?                                     *
//* a) if all comments fit into segTable[0] bytes, then the                    *
//*    setup header should begin at the next segment.                          *
//*    -- To discard comments:                                                 *
//*       -- write the placeholder comment                                     *
//*       -- reduce the size shown in segment[0] of the table                  *
//*    -- To replace comments                                                  *
//*       -- total comments < 255 bytes                                        *
//*          --adjust the size shown in segment[0] of the table                *
//*       -- total comments >= 255 bytes                                       *
//*          -- set segment[0] to 255                                          *
//*          -- increase segcount by 1 or more as needed                       *
//*          -- insert the new segment values into the table                   *
//*          -- be sure the last segment containing comments                   *
//*             is < 255                                                       *
//* b) The comment data may extend for an arbitrary number of                  *
//*    segments. The first segment of <255 bytes ends the                      *
//*    comment packet.                                                         *
//*    -- The last byte of that packet is the framing bit: 01h                 *
//        "Framing bit" (LSB set 01h) always follows the comment data:         *
//        If data remain within Page 2, then:                                  *
//        a) byte following the framing bit should be 05h                      *
//*          i.e. setup header is beginning.                                   *
//        b) 'v''o''r''b''i''s'                                                *
//* b) if segTable[0] < 255 it means end-of-packet;                            *
//*    not necessarily the end of the page. The setup header                   *
//*    can be (and usually is) on Page 2.                                      *
//* c) Start of the new packet will be 01 05 v o r b i s                       *
//*                                                                            *
//******************************************************************************

bool Taggit::WriteMetadata_OGG ( short si, WriteFilter filter, bool preserve )
{
   Ogg_PageHdr oph ;          // Page header data
   Ogg_Comment ogc ;          // Comment vector
   gString gstmp ;            // data formatting
   ofstream dbg ;             // handle for debugging output
   bool status = true ;       // Return value

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   //* For debugging, do not overwrite the source file *
   static short anSequence = ZERO ;
   char altName[MAX_FNAME] = "" ;
   #if 1    // ENABLED BY DEFAULT
   gString gsdbg( "wmd%02hd.%02hd.oga", &si, &anSequence ) ;
   gsdbg.copy( altName, MAX_FNAME ) ;
   ++anSequence ;
   #endif   // ENABLED BY DEFAULT

   gString dbgFile( this->tData.sf[si].sfPath ) ;
   short indx = (dbgFile.findlast( L'/' )) + 1 ;
   dbgFile.limitChars( indx ) ;
   dbgFile.append( "wmd.txt" ) ;
   dbg.open( dbgFile.ustr(), 
             ofstream::out | (si == ZERO ? ofstream::trunc : ofstream::app) ) ;
   if ( dbg.is_open() )
   {
      if ( si == ZERO )
         dbg << "Debug WriteMetadata_OGG()\n"
                "-------------------------\n" ;
      dbg << "si(" << si << ")  filter(" 
          << (filter == wfALL_DATA ? "wfALL_DATA" : 
              filter == wfALL_ACTDATA ? "wfALL_ACTDATA" :
              filter == wfALL_ACTIVE ? "wfALL_ACTIVE" : "wfAUDIO_ONLY")
          << ")  preserve(" << preserve << ")\n"
          << " sfName: '" << this->tData.sf[si].sfName << "'\n" ;
      if ( *altName != NULLCHAR )
          dbg << "altName: '" << altName << "'\n" ;
      dbg << endl ;
   }
   #endif                  // DEBUG_WMDATA

   gString srcPath( this->tData.sf[si].sfPath ),      // source filespec
           bacPath( "%S~", srcPath.gstr() ),          // backup filespec
           trgPath = srcPath ;                        // target filespec

   //* If user has specified a different target filename,   *
   //* adjust the target filespec. Because the source file  *
   //* will not be modified, backup filename is unnecessary.*
   if ( this->tData.sf[si].sfMod != false )
   {
      short fnameIndex = (trgPath.findlast( L'/' )) + 1 ;
      trgPath.limitChars( fnameIndex ) ;
      trgPath.append( this->tData.sf[si].sfName ) ;
   }

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   //* If alternate target filename *
   if ( *altName != NULLCHAR )
   {
      short indx = ((trgPath.findlast( L'/')) + 1) ;
      trgPath.limitChars( indx ) ;
      trgPath.append( altName ) ;

      if ( dbg.is_open() )
         dbg << "         Source backup not needed." << endl ;
   }
   else
   #endif   // DEBUG_WMDATA
   {
      //* Rename the source file as a backup  *
      //* If backup already exists, delete it *
      short bac = OK ;
      fmFType fType ;
      if ( (this->TargetExists ( bacPath, fType )) != false )
      {
         bac = this->DeleteFile ( bacPath ) ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {  if ( bac == OK )
               dbg << "         Existing backup deleted." << endl ;
            else
               dbg << "         Existing backup delete error!" << endl ;
         }
         #endif                  // DEBUG_WMDATA
      }

      //* Rename the source file *
      if ( bac == OK )
      {
         bac = this->RenameFile ( srcPath, bacPath ) ;
         srcPath = bacPath ;  // original source no longer exists
      }
      if ( bac != OK )        // access error, cannot continue
      {
         status = false ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            dbg << "Rename to backup failed. Abort.\n" << endl ;
         }
         #endif                  // DEBUG_WMDATA
      }
   }

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   if ( dbg.is_open() )
   {
      dbg << "srcPath: '" << srcPath.ustr() << "'\n"
          << "bacPath: '" << bacPath.ustr() << "'\n"
          << "trgPath: '" << trgPath.ustr() << "'\n" << endl ;
   }
   #endif                  // DEBUG_WMDATA

   //*****************************************************
   //* If setup successful, open source and target files *
   //*****************************************************
   if ( status != false )
   {
      ifstream ifs( srcPath.ustr(), ifstream::in ) ;
      ofstream ofs( trgPath.ustr(), ofstream::trunc ) ;
      if ( (ifs.is_open() != false) && (ofs.is_open() != false) )
      {
         char ibuff[gsMAXBYTES + 4] ;   // input buffer

         //******************************************
         //* Read and copy the file header (Page 1) *
         //******************************************
         ifs.read ( ibuff, OGG_PAGE1_BYTES ) ;
         if ( (ifs.gcount()) == OGG_PAGE1_BYTES )
         {
            //* Write Page 1 header *
            ofs.write( ibuff, OGG_PAGE1_BYTES ) ;
         }
         else
            ; // read error (unlikely because file has already been validated)

         //* Page 2 captured data input and adjusted data for output *
         Ogg_Segments seg ;

         //* Read header of second (Comment) page *
         oggDecodePageHeader ( ifs, oph ) ;
         seg.segCount = oph.segcount ;    // synchronize the segment count

         //* Read the segment table and calculate the combined size of all *
         //* page segments by summing the elements of the segment table.   *
         oph.segBytes = oggReadSegmentTable ( ifs, seg ) ;

         //* Accumulate total segment bytes actually read. *
         //* This should match seg.segBytes.               *
         UINT32 segBytesRead = ZERO ;

         //* Read the packet header *
         segBytesRead += oggDecodePacketHeader ( ifs, seg.opkt ) ;

         //* Validate Page 2 header                           *
         //* (For the comment page, a fail is only a warning.)*
         gstmp = oph.tag ;
         gString gstmp2( seg.opkt.pid ) ;
         if ( (gstmp.compare( Ogg_Tag ) == ZERO) && (oph.version == ZERO) &&
              (seg.opkt.ptype == 3) && (gstmp2.compare( Ogg_PktHdr ) == ZERO) )
         {
            //* Read the "Vendor" record. *
            segBytesRead += oggDecodeVendorRecord ( ifs, seg.vendor ) ;

            //* Get the number of comment vectors (records). *
            ifs.read ( ibuff, 4 ) ;
            segBytesRead += ifs.gcount() ;
            if ( (ifs.gcount()) == 4 )
               seg.oldCVectors = ogc.intConv( (UCHAR*)ibuff ) ;

            #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
            if ( dbg.is_open() )
            {
               gsdbg.compose(
                  "Page 2 Header\n"
                  "--------------------\n"
                  "Packet Header : %02hhX '%s'\n"
                  "Ogg/Vorbis tag:'%s'\n"
                  "Version       : %02hhXh\n"
                  "Page Type     : %02hhXh\n"
                  "Segment Count : %hhu\n"
                  "granpos       : %016llXh\n"
                  "StreamID      : %08Xh\n"
                  "Page Sequence : %08Xh\n"
                  "Checksum      : %08Xh\n"
                  "Segment Bytes : %u\n"
                  "Valid Page Hdr: true\n\n"
                  "Vendor Record : (%u) %s\n\n",
                  &seg.opkt.ptype, seg.opkt.pid, 
                  oph.tag, &oph.version, &oph.ptype, &oph.segcount,
                  &oph.granpos, &oph.streamser, &oph.pageseq, 
                  &oph.cksum, &seg.segBytes,
                  &seg.vendor.clen, seg.vendor.ctxt ) ;
               dbg << gsdbg.ustr() ;

               gsdbg.compose( "Total Old Comments: %d", &seg.oldCVectors ) ;
               dbg << gsdbg.ustr() << endl ;
            }
            #endif                  // DEBUG_WMDATA

            //* Read and discard existing comment vectors (if any).*
            //* Track the number of bytes occupied by the comments.*
            #if DEBUG_WMDATA == 0   // Production
            seg.oldCBytes = this->oggReadCommentVectors ( si, ifs, seg.oldCVectors, dbg ) ;
            segBytesRead += seg.oldCBytes ;
            #else    // DEBUG_WMDATA (FOR DEBUGGING ONLY)
            // Programmer's Note: This debugging code is used to validate the 
            // scan-and-discard of source file comment vectors. However, there 
            // is a logical problem: by sending a valid pointer to an Ogg_Comment 
            // array, we are enabling a re-save of any source file Image Vectors 
            // causing duplication of images. For this reason, we count the 
            // Image Vector placeholders in 'ocDebug' and if > ZERO, we delete 
            // that many images from the tail of the ePic linked list.
            UINT32 extraImage = ZERO ;
            Ogg_Comment *ocDebug = new Ogg_Comment[seg.oldCVectors + 1] ;
            seg.oldCBytes = this->oggReadCommentVectors ( si, ifs, seg.oldCVectors, 
                                                          dbg, ocDebug ) ;
            segBytesRead += seg.oldCBytes ;
            if ( dbg.is_open() )
            {
               for ( int v = ZERO ; v < seg.oldCVectors ; ++v )
               {
                  gstmp = ocDebug[v].ctxt ;  // test for image placeholder
                  if ( (gstmp.compare( ofmi[0].fName )) == ZERO )
                     ++extraImage ;

                  gstmp.compose( "(%02u) '%s' (%u bytes)", 
                                 &v, ocDebug[v].ctxt, &ocDebug[v].clen ) ;
                  dbg << gstmp.ustr() << endl ;
               }
               gstmp.compose( "       Total Bytes: %u", &seg.oldCBytes ) ;
               dbg << gstmp.ustr() << endl ;

               while ( extraImage > ZERO )   // remove duplicated images
               {
                  EmbeddedImage* eiPtr = &this->tData.sf[si].sfTag.ePic ;
                  while ( eiPtr->next != NULL )
                     eiPtr = eiPtr->next ;
                  this->tData.sf[si].sfTag.delPic( eiPtr ) ;
                  --extraImage ;
               }
            }
            delete [] ocDebug ;
            #endif   // DEBUG_WMDATA (FOR DEBUGGING ONLY)

            //* When all the comment data (if any) have been read, only     *
            //* the framing bit (01h) should remain unread in the last      *
            //* segment scanned.                                            *
            //* -- NOTE: If we reached the end of the page, the comment     *
            //*          data MAY continue on the next page. Currently, we  *
            //*          ignore that possibility. (see note above)          *
            ifs.read ( ibuff, 1 ) ; // read and verify the framing bit
            segBytesRead += ifs.gcount() ;

            #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
            if ( dbg.is_open() )
            {
               if ( (ifs.gcount() == 1) && (ibuff[0] == FRAMING_BIT) )
                  dbg << "Framing bit scanned and verified." ;
               else
                  dbg << "Error! Comment vector packet framing bit missing." ;
               dbg << endl ;

               UINT32 tableCBytes = ZERO ; // total comment bytes according to the table
               for ( UINT32 i = ZERO ; i < oph.segcount ; ++i )
               {
                  tableCBytes += (UCHAR)seg.segTable[i] ;
                  if ( seg.segTable[i] < 255 )
                     break ;
               }
               gstmp.compose( "\ntableCBytes: %u segBytesRead: %u of %u segment bytes\n", 
                              &tableCBytes, &segBytesRead, &seg.segBytes ) ;
               dbg << gstmp.ustr() << endl ;
               dbgDisplaySegmentTable ( dbg, seg ) ;
            }
            #endif                  // DEBUG_WMDATA

         }
         //* Invalid format for Page 2 header. *
         else
         {  // Programmer's Note: If we are this far down the rabbit hole,
            // there may be no comming back, so at this time, we just report it.
            // Fortunately, the source data were validated when read by the 
            // application, so a broken header is unlikely at this point.
            #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
            if ( dbg.is_open() )
            {
               gsdbg.compose( "Page 2 header validation failed!\n"
                              "Packet Header : %02hhX '%s'\n"
                              "Ogg/Vorbis tag:'%s'\n"
                              "Version       : %02hhXh (s/b 00h)\n"
                              "Page Type     : %02hhXh (s/b 03h)\n",
                              &seg.opkt.ptype, seg.opkt.pid, 
                              oph.tag, &oph.version, &oph.ptype ) ;
               dbg << gsdbg.ustr() << endl ;
            }
            #endif                  // DEBUG_WMDATA
         }

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            dbg << "\n=======================\n"
                     "BEGIN OUTPUT FORMATTING\n"
                     "=======================\n" << endl ;
         }
         #endif                  // DEBUG_WMDATA

         //* If edited metadata is to be inserted into the target file, *
         //* prescan the edited data for number of comment vectors      *
         //* (including Image Vectors), then collect, and format the    *
         //* comment records. Calculate total size.                     *
         this->oggPrescanMetadata ( si, filter, seg, dbg ) ;

         //* Adjust the segment table and number number of segments *
         //* as necessary to accomodate the new comment data.       *
         oggAdjustSegmentTable ( seg, dbg ) ;
         oph.segcount = seg.segCount ;    // synchronize the segment count

         //* Allocate a buffer for formatting the output stream.*
         seg.pbAllocate () ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            gsdbg.formatInt( seg.pbAlloc, 7, true ) ;
            gsdbg.insert( "Allocated: pageBuff[" ) ;
            gsdbg.append( "]\n" ) ;
            dbg << gsdbg.ustr() << endl ;
         }
         #endif                  // DEBUG_WMDATA

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            gsdbg.compose( "Formatted Comment Vectors (%2u)\n"
                           "------------------------------\n",
                           &seg.newCVectors ) ;
            dbg << gsdbg.ustr() ;
            UINT32 streamLen ;
            if ( seg.newCVectors > ZERO )
            {
               // (see notes in oggPrescanMetadata())
               for ( int v = ZERO ; v < seg.newCVectors ; ++v )
               {
                  streamLen = seg.newComm[v].intConv( (UCHAR*)seg.newComm[v].ctxt ) ;
                  #if OGG_IMAGES != 0
                  gsdbg = &seg.newComm[v].ctxt[4] ;
                  if ( (gsdbg.find( ofmi[0].fName )) == ZERO )
                     gsdbg.compose( "(%4u - %4u) 'embedded image'\n", 
                                    &streamLen, &seg.newComm[v].clen ) ;
                  else
                  #endif   // OGG_IMAGES
                     gsdbg.compose( "(%4u - %4u) '%s'\n", &streamLen, 
                                    &seg.newComm[v].clen, &seg.newComm[v].ctxt[4] ) ;
                  dbg << gsdbg.ustr() ;
               }
               gsdbg.compose( "------------------\n"
                              "  Total:%4d bytes\n", &seg.newCBytes ) ;
               dbg << gsdbg.ustr() ;
            }
            else
               dbg << "None, writing audio data only.\n" ;
            dbg << endl ;
         }
         #endif                  // DEBUG_WMDATA

         //*******************************************
         //* Write Page 2 data to the output buffer: *
         //*******************************************
         //*  a) Page 2 header
         oggEncodePageHeader ( seg, oph ) ;
         //*  b) segment table: segTable[seg.segCount]
         oggEncodeSegmentTable ( seg ) ;
         //*  c) packet header:
         oggEncodePacketHeader ( seg ) ;
         //*  d) Vendor record: contained in 'seg.vendor' object
         oggEncodeVendorRecord ( seg ) ;
         //*  e) number of comment vectors (4 bytes 'nn nn nn nn')
         //*  f) array of comment vectors (if any)
         //*  g) framing bit '01'
         oggEncodeCommentVectors ( seg, &this->tData.sf[si].sfTag.ePic ) ;
         //*  h) Setup Header (remaining segments in Page 2)
         oggEncodeSetupHeader ( ifs, seg ) ;
         //* Page 2 is now fully encoded.
         //* Calculate the CRC and insert it into the stream. *
         #if DEBUG_WMDATA == 0   // Production
         oggCalculateCRC ( seg ) ;
         #else    // DEBUG_WMDATA - DEBUGGING ONLY
         UINT32 oldCRC = oph.cksum,
                newCRC = oggCalculateCRC ( seg ) ;
         #endif   // DEBUG_WMDATA

         #if DEBUG_WMDATA != 0   // DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            #if 0    // Reformat the page header for comparison
            Ogg_PageHdr xph ;
            xph.reset();
            xph.tag[0] = seg.pageBuff[0] ;
            xph.tag[1] = seg.pageBuff[1] ;
            xph.tag[2] = seg.pageBuff[2] ;
            xph.tag[3] = seg.pageBuff[3] ;
            xph.tag[4] = '\0' ;
            xph.version   = seg.pageBuff[4] ;
            xph.ptype     = seg.pageBuff[5] ;
            xph.granpos   = xph.intConv64( (UCHAR*)&seg.pageBuff[6] ) ;
            xph.streamser = xph.intConv( (UCHAR*)&seg.pageBuff[14] ) ;
            xph.pageseq   = xph.intConv( (UCHAR*)&seg.pageBuff[18] ) ;
            xph.cksum     = xph.intConv( (UCHAR*)&seg.pageBuff[22] ) ;
            xph.segcount  = seg.pageBuff[26] ;
            gsdbg.compose(
               "Page 2 Header (modified)\n"
               "--------------------\n"
               "Ogg/Vorbis tag:'%s'\n"
               "Version       : %02hhXh\n"
               "Page Type     : %02hhXh\n"
               "Segment Count : %hhu\n"
               "granpos       : %016llXh\n"
               "StreamID      : %08Xh\n"
               "Page Sequence : %08Xh\n"
               "Checksum      : %08Xh\n",
               xph.tag, &xph.version, &xph.ptype, &xph.segcount,
               &xph.granpos, &xph.streamser, &xph.pageseq, 
               &xph.cksum ) ;
               dbg << gsdbg.ustr() ;
            #endif   // Reformat the page header for comparison

            gsdbg.compose( "oldCRC: %08X\n"
                           "newCRC: %08X\n"
                           "Writing %d bytes to target file... ", 
                           &oldCRC, &newCRC, &seg.pbIndex ) ;
            dbg << gsdbg.ustr() ;
         }
         #endif                  // DEBUG_WMDATA

         //* Write the Page 2 buffer to the target file *
         status = oggWriteMetadata ( ofs, seg ) ;

         #if DEBUG_WMDATA != 0   // DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            if ( status )
               dbg << "DONE\n" << endl ;
            else
               dbg << "WRITE ERROR!\n" << endl ;
         }
         #endif                  // DEBUG_WMDATA

         //* Copy the remainder of the source file to target
         do
         {
            ifs.read ( ibuff, gsMAXBYTES ) ;
            if ( (ifs.gcount()) > ZERO )
               ofs.write ( ibuff, (ifs.gcount()) ) ;
         }
         while ( ! (ifs.eof()) ) ;
         ofs.flush() ;

         //* When save is complete, then the target becomes the source for *
         //* any subsequent updates.                                       *
         //* If target filespec != source filespec, update source filespec.*
         if ( this->tData.sf[si].sfMod != false )
            trgPath.copy( this->tData.sf[si].sfPath, gsMAXBYTES ) ;

         //* Reset the edits-pending flags and update the display *
         this->tData.sf[si].sfTag.tfMod = 
         this->tData.sf[si].sfMod       = false ;

      }     // files opened successfully
      else                    // access error - Abort!
      {
         status = false ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            dbg << "ifs and/or ofs failed to open. Abort.\n" << endl ;
         }
         #endif                  // DEBUG_WMDATA
      }
      if ( ifs.is_open() )    // close source file
         ifs.close() ;
      if ( ofs.is_open() )    // close target file
         ofs.close() ;

      //* Delete the backup file *
      // Programmer's Note: Under some circumstances (see above) the backup 
      // file will not have been created, but it does no harm to delete a 
      // file which does not exist.
      if ( ! preserve
         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         && (*altName == NULLCHAR)
         #endif   // DEBUG_WMDATA
         )
      {
         #if DEBUG_WMDATA == 0   // Production

         this->DeleteFile ( bacPath ) ;

         #else    // DEBUG_WMDATA
         if ( (this->DeleteFile ( bacPath )) == OK )
            dbg << "         Backup file deleted" << endl ;
         else
            dbg << "         Error deleting backup file." << endl ;
         #endif   // DEBUG_WMDATA
      }
      #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
      else if ( *altName != NULLCHAR )
      {  //* If our dummy filename was used for output, update *
         //* the source path and filename to point to it.      *
         gsdbg = this->tData.sf[si].sfPath ;
         indx = gsdbg.findlast( L'/' ) ;
         gsdbg.limitChars( indx + 1 ) ;
         gsdbg.append ( altName ) ;
         gsdbg.copy( this->tData.sf[si].sfPath, gsMAXCHARS ) ;
         gsdbg = altName ;
         gsdbg.copy( this->tData.sf[si].sfName, MAX_FNAME ) ;
      }
      #endif   // DEBUG_WMDATA
   }

   #if DEBUG_WMDATA != 0
   dbg << endl ;
   dbg.close() ;              // close the debugging file
   #endif   // DEBUG_WMDATA

   return status ;

}  //* End WriteMetadata_OGG() *

//*************************
//*     oggUnmapField     *
//*************************
//******************************************************************************
//* Return the OGG Comment header corresponding to the specified MP3 tag       *
//* field.                                                                     *
//*                                                                            *
//* Input  : fldCode : member of enum TagFields                                *
//*                                                                            *
//* Returns: pointer to OGG Comment field name                                 *
//******************************************************************************

const wchar_t* Taggit::oggUnmapField ( short fldCode ) const
{
   const wchar_t* wPtr = NULL ;

   //* Scan the list of predefined field names *
   for ( short i = ZERO ; i < OFMA_COUNT ; ++i )
   {
      if ( fldCode == ofma[i].fIndex )
      {
         wPtr = ofma[i].fName ;
         if ( ofma[i].fIndex == tfTenc )   // special-case processing
            wPtr = OGG_BLIST_01 ;
         break ;
      }
   }

   //* If match not found, scan the B-list *
   if ( wPtr == NULL )
   {
      for ( short i = ZERO ; i < OFMB_COUNT ; ++i )
      {
         if ( fldCode == ofmb[i].fIndex )
         {
            switch ( fldCode )
            {
               case tfTyer:   wPtr = OGG_BLIST_02 ;   break ;
               case tfTcom:   wPtr = OGG_BLIST_03 ;   break ;
               default:                               break ;
            } ;
            break ;
         }
      }
   }

   //* If no match found, default to tfTxxx "DESCRIPTION" *
   if ( wPtr == NULL )
   {
      wPtr = OGG_DFLT_COMMENT ;
   }
   return wPtr ;

}  //* End oggUnmapField() *

//*************************
//*  oggPrescanMetadata   *
//*************************
//******************************************************************************
//* Format the edited metadata in preparation for writing to target file.      *
//*                                                                            *
//* Input  : si    : index of source data to scan                              *
//*          filter: (member of enum WriteFilter)                              *
//*                  determines which metadata will be written to target       *
//*          ocPtr : pointer to 'Ogg_Comment'                                  *
//*                  -- an array of 'Ogg_Comment' objects is instantiated      *
//*                     and attached to this pointer                           *
//*                  -- any array previously instantiated will be deleted      *
//*          dbg   : handle for debugging output stream                        *
//*                (file is opened by caller if debugging output is required)  *
//*                                                                            *
//* Returns: number of comment vectors formatted for output                    *
//******************************************************************************
//* Notes: These notes apply primarily to MP3 comments, and special processing *
//*        for other encodings are listed under subheadings.                   *
//* Comment vectors are selected according to the value of 'filter'.           *
//*  1) wfALL_DATA (default)                                                   *
//*     a) All fields which contain non-null data will be returned.            *
//*     OGG/Vorbis:                                                            *
//*      i) Fields which map directly to a defined comment name will be        *
//*         returned using that name.                                          *
//*     ii) Fields which do not map directly to a defined comment name will    *
//*         be returned using the default comment name (see OGG_DFLT_COMMENT)  *
//*                                                                            *
//*  2) wfALL_ACT_DATA:                                                        *
//*     a) All 'active' fields which contain non-null data will be returned.   *
//*     b) All inactive fields, regardless of content will be ignored.         *
//*     OGG/Vorbis:                                                            *
//*      i) Fields which map directly to a defined comment name will be        *
//*         returned using that name.                                          *
//*     ii) Fields which do not map directly to a defined comment name will    *
//*         be returned using the default comment name (see OGG_DFLT_COMMENT)  *
//*                                                                            *
//*  3) wfALL_ACTIVE:                                                          *
//*     a) All 'active' fields will be returned, regardless of content.        *
//*     b) All inactive fields, regardless of content will be ignored.         *
//*     OGG/Vorbis:                                                            *
//*      i) For OGG/Vorbis comment vectors, this option is interpreted as      *
//*         meaning all fields which are directly mapped to OGG/Vorbis         *
//*         comment-vector names. These name-to-field mappings are defined     *
//*         in the OggFieldMap object 'ofma'.                                  *
//*         If a match is found when 'fldi == ofma[n].fIndex                   *
//*     ii) Note that we allow 0 or 1 empty default mapping (OGG_DFLT_COMMENT).*
//*         - If a non-empty default field is found, then no empty default     *
//*           fields will be accepted.                                         *
//*         - If no default field with data is found, then exactly one empty   *
//*           default field will be appended to the list.                      *
//*                                                                            *
//*  4) wfAUDIO_ONLY:                                                          *
//*     a) Discard all metadata and returns cVectors == ZERO.                  *
//*        Caller does not need to call with this option; however, as a        *
//*        convenience we simply return without performing a data scan.        *
//*                                                                            *
//*   -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -   *
//*                                                                            *
//* Mapping and unmapping OGG/Vorbis comments to/from MP3 display fields:      *
//*                                                                            *
//* a) Source fields that had the same OGG comment vector name will have been  *
//*    concatenated for display into the corresponding MP3 tag field when      *
//*    loaded from the source file.                                            *
//* b) When writing to target, we separate the concatenated tag fields into    *
//*    separate comment vectors.                                               *
//*    -- We have no choice but to trust the user to concatenate items         *
//*       according to a consistent pattern.                                   *
//*    -- If the user removes delimiters, then the data will not be split into *
//*       separate comment vectors. No harm done except that the user may be   *
//*       surprised at the result.                                             *
//*    -- If the user uses the delimiter incorrectly, we will parse the field  *
//*       incorrectly. Sorry about that, but reading the user's mind is not    *
//*       scheduled for implementation until version 4.5.00.                   *
//* c) For each display field that contains data, the MP3 field designation    *
//*    is unmapped from its 'enum TagFields' index to a corresponding OGG      *
//*    comment-vector name.                                                    *
//* d) Images (if any) for the file are stored in the linked list:             *
//*              Taggit::SrcFile[si].tagData::tagFields::ePic                  *
//*    This storage format is optimized for MP3 image data; however, these     *
//*    images and their associated setup data are easily converted into OGG    *
//*    comment records. The only complex portion is encoding the binary data   *
//*    as "Base64" pseudotext. This is handled by the 'Ogg_Image' class.       *
//*                                                                            *
//* Output formatting:                                                         *
//* ------------------                                                         *
//* 1) Formatted comment records are returned to caller in an array of         *
//*    Ogg_Comment objects.                                                    *
//*    a) seg.newComm[ocIndex].ctxt receives the formatted BINARY stream.      *
//*       This is not a text string, it is a binary stream. The first four(4)  *
//*       bytes are a 32-bit unsigned integer (little-endian) indicating the   *
//*       length of the comment title + the equals character ('=') + the length*
//*       of the UTF-8 argument.                                               *
//*    b) seg.newComm[ocIndex].clen is the number of binary bytes in ctxt.     *
//*       Length does not include a null terminator because the null terminator*
//*       is not written to the target file.                                   *
//*       When written to the target file byte stream, it would appear as:     *
//*                        b0b1b2b3ARTIST=Bon Jovi                             *
//*                  08 00 00 00 A R T I S T = B o n _ J o v i                 *
//*                  ----------- ----------- - ---------------                 *
//*                       |          |       |         |                       *
//* little-endian int --->|          |       |         |                       *
//*   (four(4) bytes)                |       |         |                       *
//* comment name ------------------->|       |         |                       *
//* the 'equals' characters ---------------->|         |                       *
//* comment text (UTF-8) ----------------------------->|                       *
//*                                                                            *
//*    c) OGG/Vorbis embedded images.                                          *
//*       Embedded images are encoded as rather large, but otherwise ordinary  *
//*       OGG Comments. The image will not fit into the comment vector format; *
//*       therefore, a placholder comment vector is inserted for each image to *
//*       be embedded. The placeholder contains:                               *
//*          'ctxt': The Image Vector size integer is encoded and written,     *
//*                  followed by the "METADATA_BLOCK_PICTURE" comment name.    *
//*                  This alerts the caller that an image is to be inserted    *
//*                  into the binary stream.                                   *
//*          'clen': the overall size of the Image Vector in bytes.            *
//*                  This is used to calculate the size of the output buffer.  *
//*                                                                            *
//*       When the Comment Vectors are written to the output buffer, the       *
//*       placeholder vector is replaced by the Image Vector data including    *
//*       the Image Vector header and the binary image data converted to       *
//*       "Base64" pseudotext. See oggEncodeCommentVectors() for details.      *
//*                                                                            *
//******************************************************************************

short Taggit::oggPrescanMetadata ( short si, WriteFilter filter, Ogg_Segments& seg, 
                                   ofstream& dbg)
{
   gString gs, gsc, gsa ;     // data formatting
   UINT32 utfLen ;            // number of UTF-8 bytes in comment argument (! incl null)
   short  ocIndex = ZERO,     // index into output array
          deIndex = ZERO,     // index for delimiter scan
          iVectors = ZERO,    // number of images to be embedded
          cVectors = ZERO ;   // return value
   bool   hasData,            // 'true' if source field contains data
          isActive,           // 'true' if source field is active (displayed)
          oggDflt,            // 'true' if mapped to default comment name
          nedName = false ;   // 'true' if a non-empty default comment record found

   seg.cvAlloc ( ZERO ) ;     // release any old allocation and reset counters

   //*******************************************************
   //* If audio only is to be saved, do not scan the data. *
   //*              NOTE THE EARLY RETURN                  *
   //*******************************************************
   if ( filter == wfAUDIO_ONLY )
      return seg.newCVectors ;

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   gString gsdbg ;            // text formatting
   if ( dbg.is_open() )
   {
      dbg << "oggPrescanMetadata\n"
             "------------------\n" ;
   }
   #endif                  // DEBUG_WMDATA

   //* Pre-scan the data and count the number of *
   //* comment vectors which meet the criteria.  *
   for ( short fldi = ZERO ; fldi < tfCOUNT ; ++fldi )
   {
       //* Get the comment name associated with this field *
      gsc = this->oggUnmapField ( fldi ) ;
      //* Mapped to OGG default comment name? *
      oggDflt = (gsc.compare( OGG_DFLT_COMMENT ) == ZERO) ;

      //* Does field contain data? *
      hasData = ((*this->tData.sf[si].sfTag.field[fldi]) != NULLCHAR) ;
      //* Is the field active? *
      isActive = this->tData.sffDisp[fldi] ;
      //* Default OGG comment name AND contains data *
      if ( ! nedName && oggDflt && hasData )
         nedName = true ;

      //* See notes above for the logic of this test *
      if (    (hasData && isActive)                      // wfALL_ACTDATA
           || ((filter == wfALL_ACTIVE) && isActive      // wfALL_ACTIVE
                && !oggDflt)                             // (except empty defaults)
           || (hasData && (filter == wfALL_DATA))        // wfALL_DATA
         )
      {
         ++cVectors ;      // at least one vector in this display field

         //* Scan the source string for field delimiters *
         gsa = this->tData.sf[si].sfTag.field[fldi] ;
         deIndex = ZERO ;
         do
         {
            if ( (deIndex = gsa.find( oggFIELD_DELIMITER, deIndex )) >= ZERO )
            {
               if ( gsa.gstr()[deIndex + oggFD_BYTES] != '\0' )
                  ++cVectors ;
               deIndex += oggFD_BYTES ;
            }
         }
         while ( deIndex >= ZERO ) ;
      }
   }
   //* If user specified to save all data fields which map directly to OGG *
   //* comment names, BUT the default field contains no data (or is not    *
   //* active) then allocate space for an empty default comment vector.    *
   if ( (filter == wfALL_ACTIVE) && ! nedName )    // empty default field
      ++cVectors ;

#if OGG_IMAGES != 0  // UNDER CONSTRUCTION
   //* If embedded images captured or if user has added external images.    *
   //* Note that the image data will be too large for the allocated comment *
   //* vector object, so we will create a placeholder for each image.       *
   //* (see notes above)                                                    *
   if ( (iVectors = this->tData.sf[si].sfTag.countPics()) > ZERO )
   {
      cVectors += iVectors ;
   }
#endif   // OGG_IMAGES

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   if ( dbg.is_open() )
   {
      gsdbg.compose( "allocating %hd comment vectors (incl. %hd images)\n", &cVectors, &iVectors ) ;
      dbg << gsdbg.ustr() ;
   }
   #endif                  // DEBUG_WMDATA

   //* Allocate a comment vector object for each identified data field *
   seg.newCVectors = cVectors ;        // update the caller's internal count
   seg.cvAlloc ( cVectors ) ;

   //* Scan all display fields for data to be included in target file.*
   for ( short fldi = ZERO ; fldi < tfCOUNT ; ++fldi )
   {
       //* Get the comment name associated with this field *
      gsc = this->oggUnmapField ( fldi ) ;
      //* Mapped to OGG default comment name? *
      oggDflt = (gsc.compare( OGG_DFLT_COMMENT ) == ZERO) ;

      //* Does field contain data? *
      hasData = ((*this->tData.sf[si].sfTag.field[fldi]) != NULLCHAR) ;
      //* Is the field active? *
      isActive = this->tData.sffDisp[fldi] ;

      if (    (hasData && isActive)                      // wfALL_ACTDATA
           || ((filter == wfALL_ACTIVE) && isActive      // wfALL_ACTIVE
                && !oggDflt)                             // (except empty defaults)
           || (hasData && (filter == wfALL_DATA))        // wfALL_DATA
         )
      {
         //* Scan the source string for field delimiters *
         gsa = this->tData.sf[si].sfTag.field[fldi] ;
         if ( (gsa.find( oggFIELD_DELIMITER )) < ZERO )
         {  //* No delimiters, so this is a single-entry field *
            //* Compose the comment name + '=' + comment text *
            gsa.strip() ;              // strip leading and trailing whitespace
            gs.compose( "%S=%S", gsc.gstr(), gsa.gstr() ) ;
            utfLen = ((gs.utfbytes()) - 1) ;          // length of formatted comment
            seg.newComm[ocIndex].clen = utfLen + 4 ;  // length of byte stream
            //* Encoded length of byte-stream *
            seg.newComm[ocIndex].intConv( utfLen, (UCHAR*)seg.newComm[ocIndex].ctxt ) ;
             //* Append the text to byte stream *
            gs.copy( &seg.newComm[ocIndex++].ctxt[4], gsMAXBYTES ) ;

            #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
            if ( dbg.is_open() )
            {
               gsdbg.compose( "SE: '%S'\n", gs.gstr() ) ;
               dbg << gsdbg.ustr() ;
            }
            #endif                  // DEBUG_WMDATA
         }
         else
         {
            gString gst ;
            do
            {
               if ( (deIndex = gsa.find( oggFIELD_DELIMITER )) >= ZERO )
               {
                  gst = gsa ;                   // isolate the delimited field
                  gst.limitChars( deIndex ) ;
                  gst.strip() ;                 // strip leading and trailing whitespace
                  if ( (gst.gschars()) > 1 )    // if a non-empty string
                  {
                     gs.compose( "%S=%S", gsc.gstr(), gst.gstr() ) ;
                     utfLen = ((gs.utfbytes()) - 1) ;
                     seg.newComm[ocIndex].clen = utfLen + 4 ;
                     seg.newComm[ocIndex].intConv( utfLen, (UCHAR*)seg.newComm[ocIndex].ctxt ) ;
                     gs.copy( &seg.newComm[ocIndex++].ctxt[4], gsMAXBYTES ) ;
                  }
                  gsa.shiftChars( -(deIndex + oggFD_BYTES) ) ; // discard the extracted field
               }
               else                             // last delimited entry
               {
                  gsa.strip() ;                 // strip leading and trailing whitespace
                  gs.compose( "%S=%S", gsc.gstr(), gsa.gstr() ) ;
                  utfLen = ((gs.utfbytes()) - 1) ;
                  seg.newComm[ocIndex].clen = utfLen + 4 ;
                  seg.newComm[ocIndex].intConv( utfLen, (UCHAR*)seg.newComm[ocIndex].ctxt ) ;
                  gs.copy( &seg.newComm[ocIndex++].ctxt[4], gsMAXBYTES ) ;
               }
               #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
               if ( dbg.is_open() )
               {
                  gsdbg.compose( "SD: '%S'\n", gs.gstr() ) ;
                  dbg << gsdbg.ustr() ;
               }
               #endif                  // DEBUG_WMDATA
            }
            while ( deIndex >= ZERO ) ;
         }
      }
   }

   //* If we did not find the full set of vectors, scanned on the first pass, *
   //* then create an EMPTY default Comment Vector to fill the missing record.*
   if ( ocIndex < (cVectors - iVectors) )
   {
      utfLen = ZERO ;                                 // length of UTF-8 argument
      gs.compose( "%S=", OGG_DFLT_COMMENT ) ;         // comment name plus '='
      seg.newComm[ocIndex].clen = (gs.utfbytes() - 1 + 4) ; // length of byte stream
      seg.newComm[ocIndex].intConv( utfLen, (UCHAR*)seg.newComm[ocIndex].ctxt ) ;// encoded length of byte-stream
      gs.copy( &seg.newComm[ocIndex++].ctxt[4], gsMAXBYTES ) ; // create encoded byte stream
   }

#if OGG_IMAGES != 0  // UNDER CONSTRUCTION
   //* If images are to be embedded, then a comment vector has been   *
   //* allocated for each. Create a placeholder record for each image.*
   if ( iVectors > ZERO )
   {
      #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
      if ( dbg.is_open() )
      {
         gsdbg.compose( "    Embedded Image:\n"
                        "    ---------------" ) ;
         dbg << gsdbg.ustr() << endl ;
      }
      #endif   // DEBUG_WMDATA

      EmbeddedImage* eiPtr = &this->tData.sf[si].sfTag.ePic ;
      for ( UINT32 i = ZERO ; (i < (UINT32)iVectors) && (eiPtr != NULL) ; ++i )
      {
         //* Copy placeholder text to 'ctxt'. Caller will see this as an *
         //* indicator that at least one image is assigned to the file,  *
         //* and will replace this placeholder with the Image Vector.    *
         //* 'clen' receives total bytes in encoded Image Vector. This is*
         //* used below to help calculate the size of the output buffer. *
         seg.newComm[ocIndex].clen = oggImageVectorSize ( eiPtr ) ;
         seg.newComm[ocIndex].intConv( seg.newComm[ocIndex].clen, 
                                       (UCHAR*)seg.newComm[ocIndex].ctxt ) ;
         gs = ofmi[0].fName ;
         gs.copy( &seg.newComm[ocIndex++].ctxt[4], gsMAXBYTES ) ;

         #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
         if ( dbg.is_open() )
         {
            UINT32 indx = ocIndex -1 ; // index most recent entry
            gsdbg.compose( "    ctxt: '%s'\n"
                           "    clen: %u",
                           &seg.newComm[indx].ctxt[4],
                           &seg.newComm[indx].clen
                         ) ;
            dbg << gsdbg.ustr() << endl ;
         }
         #endif   // DEBUG_WMDATA

         ++cVectors ;
         eiPtr = eiPtr->next ;
      }
   }
#endif   // OGG_IMAGES

   //* Calculate the total size of the comment vectors *
   seg.newCBytes = ZERO ;
   for ( int v= ZERO ; v < seg.newCVectors ; ++v )
      seg.newCBytes += seg.newComm[v].clen ;

   #if DEBUG_WMDATA != 0   // FOR DEBUGGING ONLY
   if ( dbg.is_open() )
   {
      gsdbg.compose( "  %d vector bytes\n", &seg.newCBytes ) ;
      dbg << gsdbg.ustr() << endl ;
   }
   #endif                  // DEBUG_WMDATA

   return seg.newCVectors ;

}  //* End oggPrescanMetadata() *

//*************************
//*     oggVerifyCRC      *
//*************************
//******************************************************************************
//* Scan the specified source file:                                            *
//*  a) extract the Page 2 CRC from the page header                            *
//*  b) calculate the CRC for the remainder of the page                        *
//*  c) compare extracted and calculated values                                *
//*                                                                            *
//* Input  : si     : index of source data to scan                             *
//*                                                                            *
//* Returns: 'true' if CRC verified, else 'false'                              *
//******************************************************************************

bool Taggit::oggVerifyCRC ( short si )
{
   bool status = false ;

   return status ;

}  //* End oggVerifyCRC() *

//*************************
//*      oggMapField      *
//*************************
//******************************************************************************
//* Map the specified OGG field name to one of the defined display fields      *
//* (member of enum TagFields).                                                *
//*                                                                            *
//* Input  : fldName : name of OGG comment field                               *
//*                    -- This is a case-insensitive ASCII string (see notes)  *
//*                                                                            *
//* Returns: member of enum TagFields corresponding to the field in which to   *
//*          display the tag data                                              *
//*          -- If unable t map field, return tfCOUNT                          *
//******************************************************************************
//* Matching Comment-field name with display field:                            *
//* -- The basic display-field list is based on the MP3 specification for      *
//*    Tag fields.                                                             *
//* -- OGG has a much cleaner implementation for text tags, but the added      *
//*    flexibility means that each encoder, web database or audio playback     *
//*    application is free to create it own field names.                       *
//*    -- This could easily bite us on the ass because there may be multiple   *
//*       fields with similar names for which our algorithm would declare a    *
//*       match. In practical terms, this means that we could overwrite one    *
//*       mapped field with another causing loss of data.                      *
//* -- OGG defines a basic list of Comment-field names, and these are mapped   *
//*    to the display fields with the 'ofma[]' array.                          *
//* -- In addition, we can compare the OGG Comment-field name to common        *
//*    substrings which can be mapped to display fields. These substrings are  *
//*    listed in the 'ofmb[]' array.                                           *
//* -- If a scan of both arrays yields no match, then there would be a         *
//*    potential loss of data.                                                 *
//*    -- Our solution is to assign a default field identifier for data that   *
//*       cannot be definitively mapped to a field.                            *
//*    -- We have chosen "DESCRIPTION" which maps to the MP3 "Txxx" field.     *
//*          Example: UNKNOWN_FIELD_TITLE=abcdefg                              *
//*          maps to: DESCRIPTION=abcdefg                                      *
//* -- Duplicate vector names ARE allowed. Therefore the caller will           *
//*    concatenate the data for fields that map to the same display field.     *
//*    -- When the display field is written back to the file, the individual   *
//*       entries can be separated (assuming that the user hasn't corrupted    *
//*       our delimiter sequence during edit of the field.)                    *
//*       See definition of 'oggFIELD_DELIMITER' at the top of this module.    *
//*                                                                            *
//******************************************************************************

static TagFields oggMapField ( const gString& fldName )
{
   TagFields tfMap = tfCOUNT ;

   //* Scan the list of predefined field names *
   for ( short i = ZERO ; i < OFMA_COUNT ; ++i )
   {
      if ( (fldName.compare( ofma[i].fName, false, ofma[i].fLen )) == ZERO )
      {
         tfMap = ofma[i].fIndex ;
         break ;
      }
   }

   //* If field name is not one of the predefined strings,*
   //* scan for a matching substring.                     *
   if ( tfMap == tfCOUNT )
   {
      for ( short i = ZERO ; i < OFMB_COUNT ; ++i )
      {
         if ( (fldName.find( ofmb[i].fName, false, ofmb[i].fLen )) >= ZERO )
         {
            tfMap = ofmb[i].fIndex ;
            break ;
         }
      }
   }

   //* If no match found, assign field contents   *
   //* to the most generic field (see note above).*
   if ( tfMap == tfCOUNT )
   {
      tfMap = tfTxxx ;
   }
   return tfMap ;

}  //* End oggMapField() *

//*************************
//*  oggDecodeID_Header   *
//*************************
//******************************************************************************
//* Decode the ID Header data stream into it components.                       *
//* This header constitutes the entire Page 1 of an OGG/Vorbis audio file.     *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          oid     : (by reference) receives the ID header                   *
//*          oph     : (by reference) receives the Page header                 *
//*          opkt    : (by reference) receives the Packet header               *
//*                                                                            *
//* Returns: 'true'  if valid ID Header, else 'false'                          *
//******************************************************************************

static bool oggDecodeID_Header ( ifstream& ifs, Ogg_ID& oid, 
                                 Ogg_PageHdr& oph, Ogg_Packet& opkt )
{
   char ibuff[gsMAXBYTES] ;         // input buffer
   bool valid_hdr = true ;          // return value

   //* Read and decode the page header *
   oggDecodePageHeader ( ifs, oph ) ;

   //* Count the total segment bytes.                             *
   //* This should be exactly one byte with a value of 0x1E (30); *
   //* however, we do it the hard way to avoid making assumptions.*
   ifs.read ( ibuff, oph.segcount ) ;
   for ( UCHAR i = ZERO ; i < oph.segcount ; ++i )
      oph.segBytes += (UCHAR)(ibuff[i]) ;

   //* Read and decode the packet header *
   oggDecodePacketHeader ( ifs, opkt ) ;

   //* Read and decode the identification header *
   int readCount = OGG_PAGE1_BYTES - OGG_PAGEHDR_BYTES - OGG_PKTHDR_BYTES - oph.segcount ;
   ifs.read ( ibuff, readCount ) ;
   oid.reset() ;
   oid.version = oid.intConv( (UCHAR*)&ibuff[0] ) ;
   oid.channels = (UCHAR)ibuff[4] ;
   oid.samprate = oid.intConv( (UCHAR*)&ibuff[5] ) ;
   oid.bitratemax = oid.intConv( (UCHAR*)&ibuff[9] ) ;
   oid.bitratenom = oid.intConv( (UCHAR*)&ibuff[13] ) ;
   oid.bitratemin = oid.intConv( (UCHAR*)&ibuff[17] ) ;
   //* Each nybble is interpreted as an exponent of 2 *
   UINT32 blk = oid.intConv( (UCHAR*)&ibuff[21] ),
          blk1exp = (blk & 0x0000000F),
          blk2exp = ((blk & 0x000000F0) >> 4) ;
   oid.blksize0 = ((UINT32)0x00000001) << blk1exp ;
   oid.blksize1 = ((UINT32)0x00000001) << blk2exp ;
   oid.frameflag = (bool)(ibuff[22] & 0x01) ;

   //* Validate the page header data *
   gString gstmp( oph.tag ) ;
   if ( (gstmp.compare( Ogg_Tag ) == ZERO) &&
        (oph.version == ZERO) && (oph.ptype == 0x02) && 
        (oph.segcount == 1) && (oph.segBytes == 30)
      )
   {
      //* Validate the ID header *
      if ( (oid.version != ZERO) ||
           (oid.channels <= ZERO) ||
           (oid.samprate <= ZERO) ||
           !((oid.blksize0 == 64)   || (oid.blksize0 == 128) || 
             (oid.blksize0 == 256)  || (oid.blksize0 == 512) || 
             (oid.blksize0 == 1024) || (oid.blksize0 == 2048) || 
             (oid.blksize0 == 4096) || (oid.blksize0 == 8192)) ||
           !((oid.blksize1 == 64)   || (oid.blksize1 == 128) || 
             (oid.blksize1 == 256)  || (oid.blksize1 == 512) || 
             (oid.blksize1 == 1024) || (oid.blksize1 == 2048) || 
             (oid.blksize1 == 4096) || (oid.blksize1 == 8192)) ||
           (oid.frameflag == false) )
         valid_hdr = false ;
   }
   else
      valid_hdr = false ;

   return valid_hdr ;

}  //* End oggDecodeID_Header() *

//*************************
//*  oggDecodePageHeader  *
//*************************
//******************************************************************************
//* Read and decode any of the three required page headers for an OGA file     *
//* into its components.                                                       *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          oph     : (by reference) receives the Page header                 *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void oggDecodePageHeader ( ifstream& ifs, Ogg_PageHdr& oph )
{
   char ibuff[gsMAXBYTES] ;         // input buffer

   //* Initialize the receiving object *
   oph.reset() ;

   //* Read the raw data *
   ifs.read ( ibuff, OGG_PAGEHDR_BYTES ) ;

   //* Get tag indicating a OGG-format file.                    *
   //* The version number _should_ also be the null terminator, *
   //* but in case it's not, trim to four characters (+null).   *
   gString gs( ibuff ) ;
   gs.limitChars( OGG_TAG_BYTES ) ;          // (4 characters)
   gs.copy( oph.tag, (OGG_TAG_BYTES + 1) ) ; // (4 ASCII chars + null)

   oph.version = ibuff[4] ;      // stream structure revision
   oph.ptype   = ibuff[5] ;      // flags byte
   oph.granpos   = oph.intConv64( (UCHAR*)&ibuff[6] ) ; // granpos
   oph.streamser = oph.intConv( (UCHAR*)&ibuff[14] ) ;  // streamser
   oph.pageseq   = oph.intConv( (UCHAR*)&ibuff[18] ) ;  // pageseq
   oph.cksum     = oph.intConv( (UCHAR*)&ibuff[22] ) ;  // cksum
   oph.segcount = ibuff[26] ;    // page segment bytes

}  //* End oggDecodePageHeader() *

//*************************
//* oggDecodePacketHeader *
//*************************
//******************************************************************************
//* Read and decode the packet header which should consist of the type code    *
//* and the word 'vorbis' (case insensitive and not null terminated).          *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          opkt    : (by reference) receives the packet header               *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************

static UINT32 oggDecodePacketHeader ( ifstream& ifs, Ogg_Packet& opkt )
{
   char   ibuff[gsMAXBYTES] ;       // input buffer
   UINT32 bytesRead = ZERO ;        // return value

   ifs.read ( ibuff, OGG_PKTHDR_BYTES ) ;
   bytesRead = ifs.gcount() ;

   opkt.reset() ;
   opkt.ptype  = ibuff[0] ;
   opkt.pid[0] = ibuff[1] ;
   opkt.pid[1] = ibuff[2] ;
   opkt.pid[2] = ibuff[3] ;
   opkt.pid[3] = ibuff[4] ;
   opkt.pid[4] = ibuff[5] ;
   opkt.pid[5] = ibuff[6] ;

   return bytesRead ;

}  //* End oggDecodePacketHeader() *

//*************************
//* oggDecodeVendorRecord *
//*************************
//******************************************************************************
//* Read and decode the "Vendor" record which should consist of:               *
//*  a) an integer indicating the length of the vendor string                  *
//*  b) the vendor string (not null terminated)                                *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          vrogc   : (by reference) an Ogg_Comment object to receive record  *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************

static UINT32 oggDecodeVendorRecord ( ifstream& ifs, Ogg_Comment& vrogc )
{
   char   ibuff[gsMAXBYTES] ;       // input buffer
   UINT32 bytesRead = ZERO ;        // return value

   //* Initialize the receiving object to a default vendor record *
   vrogc.reset() ;
   vrogc.clen = defaultVENDOR_BYTES ;
   gString gs = defaultVENDOR ;
   gs.copy( vrogc.ctxt, (defaultVENDOR_BYTES + 1) ) ;

   ifs.read ( ibuff, 4 ) ;                // read length of Vendor record
   bytesRead = ifs.gcount() ;
   if ( bytesRead == 4 )
   {
      vrogc.clen = vrogc.intConv( (UCHAR*)ibuff ) ;
      ifs.read ( ibuff, vrogc.clen ) ;    // read the Vendor record
      bytesRead += ifs.gcount() ;
      if ( (ifs.gcount()) == vrogc.clen )
      {
         //* Null-terminate and save the record *
         ibuff[vrogc.clen] = NULLCHAR ;
         gs = ibuff ;
         gs.copy( vrogc.ctxt, gsMAXBYTES ) ;
      }
      else  // else restore placeholder length
         vrogc.clen = defaultVENDOR_BYTES ;
   }
   return bytesRead ;

}  //* End oggDecodeVendorRecord() *

//*************************
//*  oggReadSegmentTable  *
//*************************
//******************************************************************************
//* Read the segment table and calculate the combined size of all page         *
//* segments by summing the elements of the segment table.                     *
//*                                                                            *
//* Input  : ifs     : handle for open input stream                            *
//*          seg     : (by reference) receives segment data                    *
//*                                                                            *
//* Returns: total number of segment bytes                                     *
//******************************************************************************

static UINT32 oggReadSegmentTable ( ifstream& ifs, Ogg_Segments& seg )
{
   seg.segBytes = ZERO ;      // reset the accumulator

   ifs.read ( seg.segTable, seg.segCount ) ;
   for ( UCHAR i = ZERO ; i < seg.segCount ; ++i )
      seg.segBytes += (UCHAR)(seg.segTable[i]) ;

   return seg.segBytes ;

}  //* End oggReadSegmentTable() *

//*************************
//* oggReadCommentVectors *
//*************************
//******************************************************************************
//* Read the comment vectors from the input stream and optionally save them to *
//* caller's Ogg_Comment buffer.                                               *
//*                                                                            *
//*   NOTE: Any (text) comment vector larger than four(4) KBytes will be       *
//*         truncated. This is not likely to happen, but be aware.             *
//*         See below for notes on Image Vectors.                              *
//*   NOTE: If an embedded image is encountered, it is probably much larger    *
//*         than 4 KBytes, but it will be captured to an image object rather   *
//*         to a standard comment vector, and thus will be captured intact.    *
//*         The placeholder comment vector will receive a length of zero,      *
//*         indicating that the vector data have been saved as an image and    *
//*         that the placeholder may be discarded.                             *
//*                                                                            *
//* Input  : si      : index of source file with focus                         *
//*          ifs     : handle for open input stream                            *
//*          cvCount : expected number of comments                             *
//*          dbg     : handle to open output file to hold debugging data       *
//*          ocPtr   : (optional, NULL pointer by default)                     *
//*                    if specified, pointer to an array of Ogg_Comment objects*
//*                    to receive the captured comment vectors.                *
//*                                                                            *
//* Returns: number of bytes read from the input stream                        *
//******************************************************************************
//* Image Vectors:                                                             *
//* The OGG/Vorbis standard embeds cover art and other images as if they were  *
//* comment vectors; however, these Image Vectors require special processing.  *
//*                                                                            *
//* For image vectors, if we are saving the data, capture and decode the image *
//* setup data and binary picture data to an EmbeddedImage object in our       *
//* 'ePic' data member (linked list of images).                                *
//* If however this is simply a scan-and-discard operation (ocPtr == NULL),    *
//* then treat the image vector as a normal comment vector in order to avoid   *
//* the extra processing overhead.                                             *
//*                                                                            *
//* For additional information, see notes in oggReadImageVector(), below.      *
//******************************************************************************

int Taggit::oggReadCommentVectors ( short si, ifstream& ifs, int cvCount, 
                                    ofstream& dbg, Ogg_Comment* ocPtr )
{
   char ibuff[gsMAXBYTES + 1] ;  // input buffer
   gString gs ;                  // data formatting
   Ogg_Comment ogc ;             // temp comment buffer
   int  bytesRead = ZERO ;       // return value

   for ( int v = ZERO ; v < cvCount ; ++v )
   {
      ogc.reset() ;           // clear the receiving object

      ifs.read ( ibuff, 4 ) ; // read length of comment vector
      bytesRead += ifs.gcount() ;
      // NOTE: FOR THE SIMULATED IMAGE DATA, THIS IS THE WRONG BYTE COUNT...
      ogc.clen = ogc.intConv( (UCHAR*)ibuff ) ;

      if ( ogc.clen < gsMAXBYTES )        // prevent buffer overflow
      {
         ifs.read ( ibuff, ogc.clen ) ;   // read vector text
         bytesRead += ifs.gcount() ;

         #if OGG_IMAGES != 0  // UNDER CONSTRUCTION
         //* Test for an embedded image encoded as a comment vector.*
         UINT32 additionalBytes = ZERO ;
         if ( (ocPtr != NULL) &&
              (this->oggReadImageVector ( si, ibuff, ifs.gcount(), ogc.clen, 
                                          ifs, additionalBytes, dbg )) )
         {
            bytesRead += additionalBytes ; // additionalBytes should be ZERO

            //* Because the image data are saved to an EmbeddedImage *
            //* object, this comment vector does not contain a valid *
            //* comment. Instead, we copy "METADATA_BLOCK_PICTURE"   *
            //* as Ogg_Comment::ctxt, _BUT_ we retain the actual     *
            //* length of the Image Vector in Ogg_Comment::clen.     *
            // Programmer's Note: For the simulated image data, 'clen' will be incorrect.
            gs = ofmi[0].fName ;
            gs.copy( ogc.ctxt, gsMAXBYTES ) ;
         }
         else
         #endif   // OGG_IMAGES
         {
            ibuff[ogc.clen] = NULLCHAR ;
            gs = ibuff ;
            gs.copy( ogc.ctxt, gsMAXBYTES ) ;
         }
      }     // vector < gsMAXBYTES

      //* Comment is very large. For non-image comments, save the first    *
      //* gmMAXBYTES chunk and discard the rest. For image vectors, if we  *
      //* we are saving the data, we must capture, decode and save the     *
      //* entire vector; else treat it as a normal comment vector.         *
      else
      {
         ifs.read ( ibuff, (gsMAXBYTES - 2) ) ;
         bytesRead += ifs.gcount() ;

         #if OGG_IMAGES != 0  // UNDER CONSTRUCTION
         //* Test for an embedded image encoded as a comment vector.*
         UINT32 additionalBytes = ZERO ;
         if ( (ocPtr != NULL) &&
              (this->oggReadImageVector ( si, ibuff, ifs.gcount(), ogc.clen, 
                                          ifs, additionalBytes, dbg )) )
         {
            bytesRead += additionalBytes ; // additional bytes of source file read

            //* Because the image data are saved to an EmbeddedImage *
            //* object, this comment vector does not contain a valid *
            //* comment. Instead, we copy "METADATA_BLOCK_PICTURE"   *
            //* as Ogg_Comment::ctxt, _BUT_ we retain the actual     *
            //* length of the Image Vector in Ogg_Comment::clen.     *
            gs = ofmi[0].fName ;
            gs.copy( ogc.ctxt, gsMAXBYTES ) ;
         }
         else
         #endif   // OGG_IMAGES
         {
            ibuff[(gsMAXBYTES - 2)] = NULLCHAR ;
            gs = ibuff ;
            gs.copy( ogc.ctxt, gsMAXBYTES ) ;
            int remaining = ogc.clen - (gsMAXBYTES - 2) ;
            ogc.clen = gs.utfbytes() - 1 ;
            while ( remaining > ZERO )
            {
               ifs.read ( ibuff, (remaining >= (gsMAXBYTES - 2)) ? (gsMAXBYTES - 2) : remaining ) ;
               bytesRead += ifs.gcount() ;
               remaining -= ifs.gcount() ;
            }
         }
      }     // large vector

      //* If specified, save comment to caller's array *
      if ( ocPtr != NULL )
      {
         ocPtr[v] = ogc ;
      }
   }     // for(;;)

   return bytesRead ;

}  //* End oggReadCommentVectors() *

//*************************
//*  oggReadImageVector   *
//*************************
//******************************************************************************
//* Called only by oggReadCommentVectors(), determine whether the header       *
//* specifies an Image Vector or a standard Comment Vector.                    *
//*                                                                            *
//* If an Image Vector, then:                                                  *
//*   1) decode the header.                                                    *
//*   2) decode the binary image data                                          *
//*   3) if specified, save setup data and write image data to a temp file.    *
//*                                                                            *
//*                                                                            *
//* Input  : si       : index of source file with focus                        *
//*          ibuff    : caller's input buffer containing the first gsMAXBYTES  *
//*                     of data for the vector. This includes the vector name, *
//*                     and if it is an image vector, the remainder of the     *
//*                     header and at least some (possibly all) of the encoded *
//*                     binary image data.                                     *
//*          ibLen    : number of data bytes in 'ibuff' on entry               *
//*          veLen    : total length of the vector in bytes                    *
//*          ifs      : handle for an open input stream containing remainder   *
//*                     of encoded binary image data (if any)                  *
//*          srcCount : (by reference, initial value ignored)                  *
//*                     receives the number of _additional_ bytes read from    *
//*                     the input stream                                       *
//*          dbg      : handle to open output file to hold debugging data      *
//*                                                                            *
//* Returns: 'true'  if image vector found and scanned                         *
//*          'false' if vector is not an image vector                          *
//******************************************************************************
//* Embedded Images:                                                           *
//* From: <https://wiki.xiph.org/VorbisComment>                                *
//* ===========================================                                *
//* https://en.wikipedia.org/wiki/Base64 describes the encoding of binary data *
//* to base64 pseudotext.                                                      *
//*                                                                            *
//* [See Ogg_Image class for additional information.]                          *
//*                                                                            *
//* Cover art                                                                  *
//* METADATA_BLOCK_PICTURE                                                     *
//* ======================                                                     *
//* The binary FLAC picture structure is base64 encoded and placed within a    *
//* VorbisComment with the tag name "METADATA_BLOCK_PICTURE". This is the      *
//* preferred and recommended way of embedding cover art within VorbisComments.*
//* It has the following benefits:                                             *
//*                                                                            *
//* Easy to use for developers since the identical (or similar) structure is   *
//* also used by FLAC and MP3.                                                 *
//* The cover art can either be linked or embedded within the stream.          *
//* Common picture file formats are supported (jpg and png).                   *
//* A description may be included and the picture type (front cover,           *
//* back cover...) and image mime type are provided.                           *
//* Base64 encoded data is invariant under UTF-8 and a valid UTF-8 string, so  *
//* obeys the rules for comment data.                                          *
//*                                                                            *
//* Implementations interpreting or writing picture blocks should note the     *
//* following details:                                                         *
//* General encoding/decoding                                                  *
//* -------------------------                                                  *
//* Failure to decode a picture block should not prevent playback of the file  *
//* (failure to deal with the particularly large packet required by the        *
//* comment header is a separate problem with the player implementation).      *
//* Base64 encoding is used as in section 4 of RFC4648. We note that line      *
//* feeds are not allowed and padding characters ('=') are required.           *
//* Applications adding picture blocks should inform users that some           *
//* applications or hardware may not support them and should provide a method  *
//* to remove the blocks (this is expected to be trivial for applications      *
//* capable of adding them).                                                   *
//*                                                                            *
//* Block handling                                                             *
//* --------------                                                             *
//* The unencoded format is that of the FLAC picture block. The fields are     *
//* stored in big endian order as in FLAC, picture data is stored according    *
//* to the relevant standard.                                                  *
//* Picture data should be stored in PNG or JPEG formats or linked separately. *
//* It is recommended readers support both PNG and JPEG.                       *
//* Allowed values for the MIME string are "image/", "image/png", "image/jpeg" *
//* and "-->" (the link indicator) and "" (length 0). An empty MIME string     *
//* indicates type "image/".                                                   *
//* Fields present in the ID3V2.4.0 Attached Picture Frame (APIC Frame) take   *
//* the same interpretation as in the ID3V2.4.0 format with the following      *
//* exceptions (following the FLAC format):                                    *
//* The description field is UTF-8 (encoded without ID3V2's initial            *
//* 'encoding byte')                                                           *
//* String fields are not null terminated: their preceding length fields are   *
//* used instead.                                                              *
//*                                                                            *
//* ========================================================================== *
//*                                                                            *
//* Format for METADATA_BLOCK_PICTURE                                          *
//* =================================                                          *
//* This is the FLAC (free lossless audio codec) format. We have been unable   *
//* to find a definitive format for OGG embedded images, but we believe this   *
//* is the same format.                                                        *
//* <https://xiph.org/flac/format.html#metadata_block_picture>                 *
//*                                                                            *
//* DATA                            FORMAT                                     *
//* ------------------------------  -----------------------------------------  *
//* pic type code                   32-bit integer (big-endian)                *
//* mime string length              32-bit integer (big-endian)                *
//* mime string                     ASCII "image/jpeg","image/png","image/"    *
//* description string length       32-bit integer (big-endian)                *
//* description string              UTF-8 encoded                              *
//* width of image (pixels)         accurate value OR ZERO (big-endian)        *
//* height of image (pixels)        accurate value OR ZERO (big-endian)        *
//* color depth (bits per pixel)    32-bit integer OR ZERO (big-endian)        *
//* color indexing (GIF)            32-bit integer (0, jpeg && png)(big-endian)*
//* length of picture (bytes)       32-bit integer (big-endian)                *
//* picture data                    Base64 encoded                             *
//*                                                                            *
//*                                                                            *
//* Linked images                                                              *
//* -------------                                                              *
//* Support for linked images is optional for applications handling picture    *
//* blocks. When a linked picture is indicated the following rules are         *
//* observed:                                                                  *
//*                                                                            *
//* The picture data is a complete URL indicating the picture to be used,      *
//* relative URLs are allowed (note relative URLs do not start with a protocol *
//* specifier and are retrieved with the same protocol as the file being       *
//* processed).                                                                *
//* Links are ISO-8859-1 encoded                                               *
//* Applications MAY retrieve linked images via the file:// protocol.          *
//* Applications MUST obtain user approval if they wish to retrieve images     *
//* via remote protocols.                                                      *
//* Link targets may become unavailable: applications supporting linked images *
//* SHOULD recover gracefully from this and MAY report the absence to the user.*
//* The type of the linked file is not restricted to JPEG and JFIF and         *
//* applications MAY support other formats.                                    *
//* If the application does not support linked images, the target is           *
//* unavailable, not permitted or an unknown format the picture block should   *
//* be skipped.                                                                *
//* Applications may make links available to users, this is of particular use  *
//* when links are unsupported or of unsupported type.                         *
//*                                                                            *
//* Image dimension fields                                                     *
//* ----------------------                                                     *
//* The height, width, colour depth and 'number of colours' fields are for     *
//* purely informational purposes. Applications MUST NOT use them for decoding *
//* purposes, but MAY display them to the user and MAY use them to make a      *
//* decision whether to skip the block (for example if selecting the most      *
//* appropriate among multiple blocks).                                        *
//* Applications writing picture blocks MUST set these fields correctly OR set *
//* them all to zero.                                                          *
//*                                                                            *
//* Multiple blocks                                                            *
//* ---------------                                                            *
//* Multiple image blocks MAY be included as separate                          *
//*              METADATA_BLOCK_PICTURE comments.                              *
//* There may only be one each of picture type (APIC type) 1 and 2 in a        *
//* Vorbis stream. [these are icons]                                           *
//* Block order is significant for some types and applications should preserve *
//* the comment order when reading or writing VorbisComment headers. The block *
//* order may be used to determine the order pictures are presented to the user*
//*                                                                            *
//* Playback tests                                                             *
//* --------------                                                             *
//* Embedding a picture into the file might break playback of existing players *
//* (especially hardware players, software players could be updated easily).   *
//* A workaround would be to link the picture within the tag. Furthermore      *
//* users should become informed in some way that embedding a picture COULD    *
//* cause problems (as stated above).                                          *
//*                                                                            *
//* In order to test if there are playback problems, there are test files      *
//* available here (http://www.audioranger.com/coverart_mk.ogg) and            *
//* here (http://www.audioranger.com/coverart_im.ogg). You're invited to       *
//* download one of these test files (or both), test playback on your software *
//* and hardware players, and report the results here on the wiki.             *
//*     [NOTE: Both of the above links are broken.]                            *
//*                                                                            *
//* Unofficial COVERART field (deprecated)                                     *
//* --------------------------------------                                     *
//* There also exists an unofficial, not well supported comment field named    *
//* "COVERART". It includes a base64-encoded string of the binary picture data *
//* (usually a JPEG file, but this could be a different file format too).      *
//* The disadvantages are that:                                                *
//* a) no additional information like a description about the cover art or     *
//*    its type (front cover, back cover etc.) is provided.                    *
//* b) the cover art can't be linked                                           *
//* c) the base64 string is displayed within many tag editors as plain text    *
//*    because of their missing support for this "COVERART" field              *
//* d) it may breaks the playback on hardware players because of a large       *
//*    VorbisComment header                                                    *
//*                                                                            *
//* The unofficial "COVERART" field is supported for example by such software  *
//* as AudioShell (http://www.softpointer.com/AudioShell.htm) - read/write,    *
//* and Total Recorder (http://www.totalrecorder.com/)- only read.             *
//* Conversion to METADATA_BLOCK_PICTURE
//* ------------------------------------                                       *
//* Old "COVERART" tags should be converted to the new METADATA_BLOCK_PICTURE  *
//* tag (see above for its specification). This conversion is straightforward  *
//* and is suggested to be done the following way:                             *
//*                                                                            *
//* a) Decode the COVERART tag. A program MAY check the signature of the       *
//*    embedded picture in order to determine whether it is an allowed type.   *
//*    Lossless conversion from disallowed types to allowed types MAY be       *
//*    carried out.                                                            *
//* b) Fill out the FLAC block with the binary picture data. If the MIME type  *
//*    of the picture is unknown or can't be determined, the MIME type         *
//*    "image/" MAY be used instead.                                           *
//* c) Supplying image dimensions, color depth etc. is optional (see           *
//*    specification above).                                                   *
//* d) In the absence of other information the picture type 'Other' should be  *
//*    used. Applications may want to allow users to select a default type or  *
//*    specify the type to use.                                                *
//* e) Encode the new picture block, remove the COVERART tag from the comments *
//*    and add the METADATA_BLOCK_PICTURE entry.                               *
//* f) If multiple tags are being converted the order of the                   *
//*    METADATA_BLOCK_PICTURE tags should be the same as that of the COVERART  *
//*    tags they are replacing.                                                *
//******************************************************************************

bool Taggit::oggReadImageVector ( short si, char* ibuff, UINT32 ibLen, UINT32 veLen, 
                                  ifstream& ifs, UINT32& srcCount, ofstream& dbg )
{
   #define SIMULATED_IMAGE (0)   // SET TO NON-ZERO FOR DEBUGGING ONLY

   UCHAR obuff[gsMAXBYTES] ;     // binary image data output buffer
   gString gs ;                  // text formatting
   bool isIVector = false ;      // return value

   srcCount = ZERO ;             // initialize caller's counter

   #if SIMULATED_IMAGE != 0      // DEBUG ONLY
   //* This test relies on a non-standard comment vector ("ENCODER") in *
   //* the source file 'None Of Us.oga'. The sample data is from the    *
   //* Wikipedia article on Base64 encoding.                            *
   gString gssim( ibuff, (ofmi[0].fLen + 1) ) ;
   if ( (gssim.find( "ENCODER" )) == ZERO )
   {
      const char* unencodedSample = 
      // 269 unencoded bytes (excl. nullchar)
      "Man is distinguished, not only by his reason, but by this si"
      "ngular passion from other animals, which is a lust of the mi"
      "nd, that by a perseverance of delight in the continued and i"
      "ndefatigable generation of knowledge, exceeds the short vehe"
      "mence of any carnal pleasure." ;
      #if 0    // REFERENCE
      const char* encodedSample = 
      // 360 encoded bytes (excl. nullchar)
      "TWFuIGlzIGRpc3Rpbmd1aXNoZWQsIG5vdCBvbmx5IGJ5IGhpcyByZWFzb24s"
      "IGJ1dCBieSB0aGlzIHNpbmd1bGFyIHBhc3Npb24gZnJvbSBvdGhlciBhbmlt"
      "YWxzLCB3aGljaCBpcyBhIGx1c3Qgb2YgdGhlIG1pbmQsIHRoYXQgYnkgYSBw"
      "ZXJzZXZlcmFuY2Ugb2YgZGVsaWdodCBpbiB0aGUgY29udGludWVkIGFuZCBp"
      "bmRlZmF0aWdhYmxlIGdlbmVyYXRpb24gb2Yga25vd2xlZGdlLCBleGNlZWRz"
      "IHRoZSBzaG9ydCB2ZWhlbWVuY2Ugb2YgYW55IGNhcm5hbCBwbGVhc3VyZS4=" ;
      #endif   // REFERENCE

      // Total image-vector size: 
      //  22      "METADATA_BLOCK_PICTURE"
      //   1      '='
      //   4      picType
      //   4      mimLen
      //  10      MIME type
      //   4      descLen
      //  21      description
      //  16      4 integer values for image info
      //   4      picEncSize
      // 360      encoded picture data
      // ---
      // 446      total Image-vector size

      Ogg_Image simogi ;
      simogi.totBytes = 446 ;       // total size of Image vector
      simogi.picType = 0x04 ;       // picType code

      gssim = "image/jpeg" ;        // mimType string
      simogi.mimBytes = gssim.utfbytes() - 1 ;
      gssim.copy( simogi.mimType, gsMAXBYTES ) ;

      gssim = "Simulated image data." ; // txtDesc data
      simogi.txtBytes = gssim.utfbytes() - 1 ;
      gssim.copy( simogi.txtDesc, gsMAXBYTES ) ;

      simogi.iWidth  = 0x11000022 ;
      simogi.iHeight = 0x33000044 ;
      simogi.iCDepth = 0x55000066 ;
      simogi.iCIndex = 0x77000088 ;

      UINT32 simIndex = simogi.encodeHeader( (UCHAR*)ibuff ) ;

      //* For this test, remove the Image Vector size *
      //* from the head of the stream.                *
      for ( UINT32 i = ZERO, j = 4 ; j < simIndex ; ++i, ++j )
         ibuff[i] = ibuff[j] ;
      simIndex -= 4 ;

      simogi.picBytes = 269 ;
      simIndex += simogi.picEBytes = 
            simogi.encodeImage( (UCHAR*)&ibuff[simIndex], 
                                (UCHAR*)unencodedSample, simogi.picBytes ) ;
      veLen = ibLen = simIndex ;    // live bytes in source buffer
   }
   #endif   // SIMULATED_IMAGE


   //* Test for an embedded image encoded as a comment vector.*
   gString gstmp( ibuff, (ofmi[0].fLen + 1) ) ;
   if ( (((gstmp.find( ofmi[0].fName )) == ZERO) && // standard name
         ((gstmp.find( L'=')) == ofmi[0].fLen))
       ||
        (((gstmp.find( ofmi[1].fName )) == ZERO) && // non-standard name
         ((gstmp.find( L'=')) == ofmi[1].fLen)) )
   {
      Ogg_Image ogi ;         // temp image buffer
      isIVector = true ;      // image vector verified

      //* Decode the header of the Image Vector         *
      //* ('picBytes' is reset, and 'picExpl' is empty.)*
      UINT32 imgIndex = ogi.decodeHeader( (UCHAR*)ibuff, veLen ) ;
      ogi.picEBytes = veLen - imgIndex ;  // total encoded image bytes
      ibLen -= imgIndex ;     // number of image bytes remaining in buffer

      //* Move the encoded image data to top of buffer *
      // Programmer's Note: This isn't technically necessary, but makes 
      // the following code conceptually cleaner, and clean code is happy code.
      for ( UINT32 i = ZERO ; i < ibLen ; ++i )
         ibuff[i] = ibuff[imgIndex++] ;

      //* Create a node on the ePic linked list and transfer *
      //* the setup data.                                    *
      //* Note that at this time, we abandon the image       *
      //* parameters: iWidth, iHeight, iCDepth and iCIndex.  *
      //* These are "information only" fields, and therefore *
      //* not critical to function.                          *
      EmbeddedImage* eiPtr = this->tData.sf[si].sfTag.addPic() ;
      eiPtr->inUse = true ;                        // signal active node
      eiPtr->mp3img.picEncSize = ogi.picEBytes ;   // size of Base64-encoded image
      eiPtr->mp3img.picType = (UCHAR)ogi.picType ; // picture type
      gstmp = ogi.txtDesc ;                        // description text
      gstmp.copy( eiPtr->mp3img.txtDesc, gsMAXBYTES ) ;
      gstmp = ogi.mimType ;                        // MIME type text
      gstmp.copy( eiPtr->mp3img.mimType, gsMAXBYTES ) ;
      #if 0    // UNDER CONSTRUCTION - TEST FOR URL MIME TYPE
      //* Test for a URL to an external image within the MIME type string *
      gString gsurl ;
      short urlindex = ZERO ;
      if ( (urlindex = gstmp.after( mimeLnk )) > ZERO )
      {
         gsurl = &gstmp.gstr()[urlindex] ;
   
         //* If the external file is a LOCAL file, we can copy its contents.*
         if ( ((gsurl.find( "file://" )) == ZERO) || ((gsurl.find( ".." )) == ZERO) )
         {
         }
         //* Else, external file is a network location      *
         //* (http, https, ftp etc.) Don't try to access it.*
         else
         {
         }
      }
      #endif   // UNDER CONSTRUCTION - TEST FOR URL MIME TYPE

      //* Create a temporary file to hold the image. *
      //* Rename the temporary file so it can be     *
      //* read by an external image viewer.          *
      this->CreateTempname ( gs ) ;
      bool ispng = ((gstmp.find( "png" )) >= ZERO) ;
      gs.append( ispng ? ".png" : ".jpg" ) ;
      gs.copy( eiPtr->mp3img.picPath, gsMAXBYTES ) ;

      ofstream ofs( eiPtr->mp3img.picPath, (ofstream::out | ofstream::trunc) ) ;
      if ( ofs.is_open() )
      {
         //* Number of encoded picture bytes remaining in source stream *
         UINT32 pebUnread = ogi.picEBytes - ibLen,
                //* Bytes to be read from source in each pass *
                //* (init to all remaining bytes)             *
                rCount = pebUnread,
                decodedBytes ;      // number of bytes decoded

         //* Decode the image data which remain in *
         //* the buffer and save to the temp file. *
         ogi.picBytes = decodedBytes = 
                  ogi.decodeImage( obuff, (UCHAR*)ibuff, (ibLen - (ibLen % 4)) ) ;
         ofs.write( (char*)obuff, decodedBytes ) ;
         if ( (ibLen -= (ibLen - (ibLen % 4))) > ZERO )
         {
            //* Move incomplete quartet to top of buffer *
            for ( UINT32 i = ZERO ; i < ibLen ; ++i )
               ibuff[i] = ibuff[decodedBytes + i] ;
         }
         // Programmer's Note: 'ibLen' is now the number of un-decoded 
         // source bytes remaining in input buffer 

         //* Read and decode the remaining image *
         //* data and save to the temp file.     *
         // Programmer's Note: There is an unlikely scenario where all remaining 
         // encoded data have already been read into the buffer:
         //               (pebUnread==ZERO && ibLen>ZERO)
         // The reason this is an unlikely case, is that padding to 4-byte units
         // is mandated by the standard, but we still test for improperly padded
         // source data because that's just who we are. :-) (and so should you be)
         // There may still be an error in the last 1-2 bytes of the file, 
         // but it's better than having the bytes missing.
         if ( (pebUnread > ZERO) || (ibLen > ZERO) )
         {
#if 1    // UNDER CONSTRUCTION - THIS IS UNTESTED SINCE WE DON'T HAVE ANY ACTUAL DATA
            //* Number of source bytes to be read in first pass *
            if ( (pebUnread + ibLen) < gsMAXBYTES ) // if remainder will fit in buffer
               rCount = pebUnread ;
            else  // else fill the buffer (note that gsMAXBYTES is a multiple of 4)
               rCount = gsMAXBYTES - ibLen ;
            if ( rCount > ZERO ) // (see note above about improperly padded data)
            {
               ifs.read( &ibuff[ibLen], rCount ) ;
               pebUnread -= rCount ;
               srcCount += rCount ;       // update caller's accumulator
            }
            //* Decode input data and append it to temp file *
            ogi.picBytes += decodedBytes = 
                     ogi.decodeImage( obuff, (UCHAR*)ibuff, (rCount + ibLen) ) ;
            ofs.write( (char*)obuff, decodedBytes ) ;

            //* Read and decode gsMAXBYTES size chunks *
            while ( (pebUnread / gsMAXBYTES) > ZERO )
            {
               rCount = gsMAXBYTES ;
               ifs.read( ibuff, rCount ) ;
               pebUnread -= rCount ;
               srcCount += rCount ;       // update caller's accumulator
               ogi.picBytes += decodedBytes = 
                     ogi.decodeImage( obuff, (UCHAR*)ibuff, (rCount + ibLen) ) ;
               ofs.write( (char*)obuff, decodedBytes ) ;
            }
            //* Read and decode the remaining source bytes *
            if ( (rCount = pebUnread) > ZERO )
            {
               ifs.read( ibuff, rCount ) ;
               pebUnread -= rCount ;
               srcCount += rCount ;       // update caller's accumulator
               ogi.picBytes += decodedBytes = 
                     ogi.decodeImage( obuff, (UCHAR*)ibuff, (rCount + ibLen) ) ;
               ofs.write( (char*)obuff, decodedBytes ) ;
            }
#endif   // UNDER CONSTRUCTION
         }
         ofs.close() ;                          // close the file
         eiPtr->mp3img.picSize = ogi.picBytes ; // size of temp file
      }  // (ofs.is_open())

      #if DEBUG_EMDATA != 0
      gString gsdbg ;
      if ( dbg.is_open() )
      {
         gsdbg.compose( "\nOGG Embedded Image\n"
                        "------------------\n"
                        "picPath  : %s\n"
                        "picType  : %02hhX\n"
                        "mimBytes : %u\n"
                        "mimType  : %s\n"
                        "txtBytes : %u\n"
                        "txtDesc  : %s\n"
                        "iWidth   : %08X\n"
                        "iHeight  : %08X\n"
                        "iCDepth  : %08X\n"
                        "iCIndex  : %08X\n"
                        "totBytes : %u\n"
                        "picBytes : %u\n"
                        "picEBytes: %u (%d)\n",
                        eiPtr->mp3img.picPath, &eiPtr->mp3img.picType,
                        &ogi.mimBytes, eiPtr->mp3img.mimType, 
                        &ogi.txtBytes, eiPtr->mp3img.txtDesc,
                        &ogi.iWidth, &ogi.iHeight, &ogi.iCDepth, &ogi.iCIndex, 
                        &ogi.totBytes, &ogi.picBytes, &ogi.picEBytes, 
                        &eiPtr->mp3img.picEncSize
                      ) ;
         dbg << gsdbg.ustr() << endl ;
      }
      #endif   // DEBUG_EMDATA
   }
   return isIVector ;

}  //* End oggReadImageVector() *

//*************************
//*  oggEncodePageHeader  *
//*************************
//******************************************************************************
//* Convert the formatted Page 2 header data into a binary stream and write    *
//* it to the output buffer.                                                   *
//*                                                                            *
//* Input  : seg  : segment data                                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void oggEncodePageHeader ( Ogg_Segments& seg, const Ogg_PageHdr& oph )
{
   UCHAR convBuff[16] ;

   //* Write the page identifier: s/b "OggS" *
   for ( short i = ZERO ; i < 4 ; ++i )
      seg.pageBuff[seg.pbIndex++] = oph.tag[i] ;

   //* Write the stream structure revision number and the flag byte *
   seg.pageBuff[seg.pbIndex++] = oph.version ;
   seg.pageBuff[seg.pbIndex++] = oph.ptype ;

   //* Stream the 'granpos' value * (64-bit little-endian integer) *
   oph.intConv64( oph.granpos, convBuff ) ;
   for ( short i = ZERO ; i < 8 ; ++i )
      seg.pageBuff[seg.pbIndex++] = convBuff[i] ;

   //* Stream the 'streamser' value * (32-bit little-endian integer) *
   oph.intConv( oph.streamser, convBuff ) ;
   for ( short i = ZERO ; i < 4 ; ++i )
      seg.pageBuff[seg.pbIndex++] = convBuff[i] ;

   //* Stream the 'pageseq' value * (32-bit little-endian integer) *
   oph.intConv( oph.pageseq, convBuff ) ;
   for ( short i = ZERO ; i < 4 ; ++i )
      seg.pageBuff[seg.pbIndex++] = convBuff[i] ;

   //* Stream the 'cksum' value * (32-bit little-endian integer).          *
   //* NOTE: These 4 bytes will be overwritten by the newly-calculated CRC.*
   oph.intConv( oph.cksum, convBuff ) ;
   for ( short i = ZERO ; i < 4 ; ++i )
      seg.pageBuff[seg.pbIndex++] = convBuff[i] ;

   //* Write the segment count (this value was updated by caller).*
   seg.pageBuff[seg.pbIndex++] = oph.segcount ;

}  //* End oggEncodePageHeader() *

//*************************
//* oggAdjustSegmentTable *
//*************************
//******************************************************************************
//* Adjust the size and contents of the segment table to contain the edited    *
//* comment vectors.                                                           *
//*                                                                            *
//* Input  : seg  : segment data                                               *
//*          dbg  : handle for debugging output stream                         *
//*                 (file is opened by caller if debugging output is required) *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes:                                                                     *
//* ------                                                                     *
//* Segment_00 currently contains:                                             *
//*  a) Packet Header == OGG_PKTHDR_BYTES (7)                                  *
//*  b) Vector Count  == 4 bytes                                               *
//*  c) Vendor Record == 4 + seg.vendor.clen                                   *
//*  d) Some or all of the old comments, and if all, the framing bit           *
//*                                                                            *
//* Segment_01 through Segment_nn: If framing bit not found in Segment_00,     *
//* then comments continue in following segments until framing bit found.      *
//* (or end of page 2).                                                        *
//*                                                                            *
//* Note on Page 2 overflow:                                                   *
//* -- The specification says that each comment may have a length of up to     *
//*    (2 to the 32nd) - 1 bytes. However it also suggests that a comment be   *
//*    no more than "a short paragraph." For this reason, we discard any part  *
//*    of an individual comment beyond four(4) kbytes (gsMAXBYTES).            *
//* -- The specification also allows comments/setup data to extend into the    *
//*    next logical page (Page 3); however this is unlikely to happen because  *
//*    Page 2 can hold 255-squared bytes: (255 * 255 == 65,025 bytes).         *
//*    The number of segments available for comment data is:                   *
//*             max segments - segments used by the setup header.              *
//*    Typically, the Setup Header is between 16 and 24 segments of data, so   *
//*    255 - 24 == 231 segments or 231 * 255 - header ~= 58,000 bytes available*
//*    for comment data. Thus, overflow is very unlikely to occur.             *
//* UNDER CONSTRUCTION - UPDATE THIS COMMENT WHEN PAGE-SPANNING IS IMPLEMENTED.*
//* -- If the combined size of the comments/setup data WOULD extend into       *
//*    Page 3, then we take the easy path and discard all the comments,        *
//*    and insert a single comment that consists of an overflow warning.       *
//*    The setup data will remain intact.                                      *
//* -- Note that when debugging is turned on, an overflow warning will also be *
//*    written to the debug file.                                              *
//* -- In a future release, we may fully implement the specification for data  *
//*    extending into Page 3. This will be strongly desired when we have       *
//*    implemented embedded images. This is especially true because images     *
//*    are encoded as "Base64" which is a dumb-ass method for converting       *
//*    binary data into a stream compatible with UTF-8 encoding (3 bytes of    *
//*    source binary data yields 4 bytes of encoded output) which is           *
//*    _at least_ 25% larger than the original binary data, and statistically  *
//*    about 1.37 times the size of the source.                                *
//*                                                                            *
//*                                                                            *
//*                                                                            *
//*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
//* From the Specification <https://xiph.org/ogg/doc/framing.html>             *
//*                                                                            *
//* Packet segmentation                                                        *
//* -------------------                                                        *
//* Packets are logically divided into multiple segments before encoding into  *
//* a page. Note that the segmentation and fragmentation process is a logical  *
//* one; it's used to compute page header values and the original page data    *
//* need not be disturbed, even when a packet spans page boundaries.           *
//*                                                                            *
//* The raw packet is logically divided into [n] 255 byte segments and a last  *
//* fractional segment of < 255 bytes. A packet size may well consist only of  *
//* the trailing fractional segment, and a fractional segment may be zero      *
//* length. These values, called "lacing values" are then saved and placed into*
//* the header segment table.                                                  *
//*                                                                            *
//* An example should make the basic concept clear:                            *
//*                                                                            *
//* raw packet:                                                                *
//*   ___________________________________________                              *
//*  |______________packet data__________________| 753 bytes                   *
//* lacing values for page header segment table: 255,255,243                   *
//*                                                                            *
//* We simply add the lacing values for the total size; the last lacing value  *
//* for a packet is always the value that is less than 255. Note that this     *
//* encoding both avoids imposing a maximum packet size as well as imposing    *
//* minimum overhead on small packets (as opposed to, eg, simply using two     *
//* bytes at the head of every packet and having a max packet size of 32k.     *
//* Small packets (<255, the typical case) are penalized with twice the        *
//* segmentation overhead). Using the lacing values as suggested, small packets*
//* see the minimum possible byte-aligned overhead (1 byte) and large packets, *
//* over 512 bytes or so, see a fairly constant ~.5% overhead on encoding      *
//* space.                                                                     *
//*                                                                            *
//* Note that a lacing value of 255 implies that a second lacing value follows *
//* in the packet, and a value of < 255 marks the end of the packet after that *
//* many additional bytes. A packet of 255 bytes (or a [EVEN] multiple of      *
//* 255 bytes) is terminated by a lacing value of 0:                           *
//*                                                                            *
//* raw packet:                                                                *
//*   _______________________________                                          *
//*  |________packet data____________|          255 bytes                      *
//* lacing values: 255, 0                                                      *
//*                                                                            *
//* Note also that a 'nil' (zero length) packet is not an error; it consists   *
//* of nothing more than a lacing value of zero in the header.                 *
//*                                                                            *
//* Packets spanning pages                                                     *
//* ----------------------                                                     *
//* Packets are not restricted to beginning and ending within a page, although *
//* individual segments are, by definition, required to do so. Packets are not *
//* restricted to a maximum size, although excessively large packets in the    *
//* data stream are discouraged.                                               *
//*                                                                            *
//* After segmenting a packet, the encoder may decide not to place all the     *
//* resulting segments into the current page; to do so, the encoder places the *
//* lacing values of the segments it wishes to belong to the current page into *
//* the current segment table, then finishes the page. The next page is begun  *
//* with the first value in the segment table belonging to the next packet     *
//* segment, thus continuing the packet (data in the packet body must also     *
//* correspond properly to the lacing values in the spanned pages. The segment *
//* data in the first packet corresponding to the lacing values of the first   *
//* page belong in that page; packet segments listed in the segment table of   *
//* the following page must begin the page body of the subsequent page).       *
//*                                                                            *
//*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
//* [THE ABOVE EXPLANATION IS RIDICULOUSLY OBSCURE AND UN-HELPFUL.]            *
//* Presumably, it means that all values of the segment table in the current   *
//* segment are set to 255 (0xFF). The segment table of the following page     *
//* would then continue with one or more entries describing the data for that  *
//* page. In any case, the segment tables for completely-filled pages will     *
//* have a length of 255, with each entry in the table being 255. The packet   *
//* data would the _exactly_ fill the page. The segment table of the page in   *
//* which the packet ends will have a final value <255.                        *
//*                                                                            *
//* It is unclear, but it appears that a page which is filled with packet data *
//* will have no framing bit. The framing bit will be located at the end of    *
//* the data packet on whatever page the packet data ends.                     *
//*                                                                            *
//* What does a continuation page header look like?                            *
//* -----------------------------------------------                            *
//* OggS vv  pt    gpgpgpgpgpgpgpgp stststst  pspspsps ckkckckck sc            *
//* txt  ver flags     granpos      streamser pageseq  checksum  segment count *
//*      The continuation header s/b the same as the parent page except:       *
//*      The continuation flag is bit0 of the flag byte and s/b set.           *
//*      The value of 'pageseq' byte s/b ?                                     *
//*      The checksum is specific to the data of that page.                    *
//*      The segment count is specific to the data of that page.               *
//* The segment table for the page follows the page header.                    *
//* The remaining packet data follow the segment table                         *
//* The framing bit terminates the page.                                       *
//*                                                                            *
//* There is no packet header on the continuation page because the continuation*
//* page continues the data for the same packet.                               *
//*                                                                            *
//*                                                                            *
//*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  - *
//*                                                                            *
//* The last mechanic to spanning a page boundary is to set the header flag in *
//* the new page to indicate that the first lacing value in the segment table  *
//* continues rather than begins a packet; a header flag of 0x01 is set to     *
//* indicate a continued packet. Although mandatory, it is not actually        *
//* algorithmically necessary; one could inspect the preceding segment table   *
//* to determine if the packet is new or continued. Adding the information to  *
//* the packet_header flag allows a simpler design (with no overhead) that     *
//* needs only inspect the current page header after frame capture. This also  *
//* allows faster error recovery in the event that the packet originates in a  *
//* corrupt preceding page, implying that the previous page's segment table    *
//* cannot be trusted.                                                         *
//*                                                                            *
//* Note that a packet can span an arbitrary number of pages; the above        *
//* spanning process is repeated for each spanned page boundary. Also a 'zero  *
//* termination' on a packet size that is an even multiple of 255 must appear  *
//* even if the lacing value appears in the next page as a zero-length         *
//* continuation of the current packet. The header flag should be set to 0x01  *
//* to indicate that the packet spanned, even though the span is a nil case as *
//* far as data is concerned.                                                  *
//*                                                                            *
//* The encoding looks odd, but is properly optimized for speed and the        *
//* expected case of the majority of packets being between 50 and 200 bytes    *
//* (note that it is designed such that packets of wildly different sizes can  *
//* be handled within the model; placing packet size restrictions on the       *
//* encoder would have only slightly simplified design in page generation and  *
//* increased overall encoder complexity).                                     *
//*                                                                            *
//* The main point behind tracking individual packets (and packet segments) is *
//* to allow more flexible encoding tricks that requiring explicit knowledge   *
//* of packet size. An example is simple bandwidth limiting, implemented by    *
//* simply truncating packets in the nominal case if the packet is arranged so *
//* that the least sensitive portion of the data comes last.                   *
//******************************************************************************

static void oggAdjustSegmentTable ( Ogg_Segments& seg, ofstream& dbg )
{
   //* Count the number of segments occupied by the existing  *
   //* Comment Header, and the total bytes in those segments. *
   int oldTotalBytes = ZERO,     // total segment space occupied by old comment data
       newTotalBytes = ZERO,     // total segment space occupied by new comment data
       lastEntry = ZERO,         // bytes occupied by last segment of comment data
       //* bytes occupied by last segment of the segment table *
       lastTableEntry = seg.segTable[seg.segCount - 1],
       sval ;                    // value from table (promoted to int)

   seg.oldCSegs = ZERO ;         // reset the comment-segment counter
   for ( int s = ZERO ; s < seg.segCount ; ++s )
   {
      ++seg.oldCSegs ;
      sval = (UCHAR)(seg.segTable[s]) ;
      oldTotalBytes += (UCHAR)seg.segTable[s] ;
      if ( sval < OGG_SEGSIZE )
      {
         lastEntry = sval ;
         break ;
      }
   }

   //* Number of non-comment bytes in the comment header segments.*
   //* These will be written regardless of whether any comment    *
   //* vectors are defined.                                       *
   //*  a) Packet Header                                          *
   //*  b) Vendor Record Length (32-bit int)                      *
   //*  c) Vendor Record                                          *
   //*  d) Comment-vector count (32-bit int)                      *
   //*  e) Framing bit (1 byte)                                   *
   int nonComm = OGG_PKTHDR_BYTES + 4 + seg.vendor.clen + 4 + 1 ;

   //* Segments required to hold the edited comment data.*
   newTotalBytes = nonComm + seg.newCBytes ;
   seg.newCSegs = (newTotalBytes / OGG_SEGSIZE) + 1 ;

   // Programmer's Note: 'lastEntry' is the contents of the last table 
   // entry in comment sequence. It is valid for 'lastEntry' to be ZERO 
   // because the end-of-sequence is signalled by a table entry < 255 bytes.
   lastEntry = newTotalBytes % OGG_SEGSIZE ;

   //* Change in overall segment count. (size of setup header does not change) *
   int segAdd = seg.newCSegs - seg.oldCSegs ;

   #if DEBUG_WMDATA != 0   // DEBUGGING ONLY
   if ( dbg.is_open() )
   {
      int new_segCount = seg.segCount + segAdd ;
      gString gs( "Adjust Segment Table:\n"
                  "---------------------\n"
                  "oldCSegs: %d  oldTotalBytes: %d  nonComm: %d\n"
                  "newCSegs: %d  newTotalBytes: %d  lastEntry: %d\n"
                  "segCount + segAdd == %d + %d == %d\n",
                  &seg.oldCSegs, &oldTotalBytes, &nonComm,
                  &seg.newCSegs, &newTotalBytes, &lastEntry,
                  &seg.segCount, &segAdd, &new_segCount ) ;
      dbg << gs.ustr() << endl ;
   }
   #endif   // DEBUG_WMDATA

// UNDER CONSTRUCTION:
// WHEN ADDING IMAGES, WE MAY EXTEND BEYOND THE 255 * 255 == 65025 bytes.
// ALTHOUGH NOT LIKELY, WE NEED TO PREPARE FOR EXTENDING INTO PAGE 3....
// SEE NOTES IN oggAdjustSegmentTable
// IF COMMENT HEADER EXTENDS BEYOND PAGE 2, (OR IF IT PUSHES THE SETUP HEADER 
// BEYOND PAGE 2), THE LAST TABLE ENTRY (255th entry) WILL BE 255 ?
   //* Note that we do not expand the table beyond 255 entries, *
   //* (see note in method header).                             *
   if ( (seg.segCount + segAdd) > OGG_SEGSIZE )
   {
      //* Discard all comments and insert a warning message *
      seg.newCVectors = 1 ;
      gString gs( "Title=Warning! Comment data has exceeded maximum size." ) ;
      seg.newComm[0].intConv( ((gs.utfbytes()) - 1), (UCHAR*)seg.newComm[0].ctxt ) ;
      gs.copy( &seg.newComm[0].ctxt[4], gs.utfbytes() ) ;
      seg.newCBytes = seg.newComm[0].clen = gs.utfbytes() - 1 + 4 ;
      newTotalBytes = nonComm + seg.newCBytes ;
      lastEntry = nonComm + seg.newComm[0].clen ;
      //* Adjust the segment count *
      seg.newCSegs = 1 ;
      segAdd = seg.newCSegs - seg.oldCSegs ;

      #if DEBUG_WMDATA != 0   // DEBUGGING ONLY
      if ( dbg.is_open() )
      {
         gString gs( 
            "ERROR! - Total comments exceed size of page.\n"
            "         All comments discarded.\n"
            "       newCVectors  : %d\n"
            "       newCBytes    : %d\n"
            "       newCSegs     : %d\n"
            "       newTotalBytes: %d\n"
            "       lastEntry    : %d\n",
            &seg.newCVectors, &seg.newCBytes, &seg.newCSegs, &newTotalBytes,
            &lastEntry ) ;
         dbg << gs.ustr() << endl ;
      }
      #endif   // DEBUG_WMDATA
   }

   //* Create a new segment table inserting the final byte counts *
   //* for Comment Header and Setup Header.                       *
   seg.segCount += segAdd ;
   for ( int segIndex = ZERO ; segIndex < seg.segCount ; ++segIndex )
   {
      if ( segIndex == (seg.newCSegs - 1) )
         seg.segTable[segIndex] = lastEntry ;      // last comment header entry < 255
      else if ( segIndex == (seg.segCount - 1)  )
         seg.segTable[segIndex] = lastTableEntry ; // last setup header entry < 255
      else
         seg.segTable[segIndex] = OGG_SEGSIZE ;
   }

   #if DEBUG_WMDATA != 0   // DEBUGGING ONLY
   dbg << endl ;
   dbgDisplaySegmentTable ( dbg, seg ) ;
   #endif   // DEBUG_WMDATA

}  //* End oggAdjustSegmentTable() *

//*************************
//* oggEncodeSegmentTable *
//*************************
//******************************************************************************
//* Convert the formatted Page 2 segment header into a binary stream and write *
//* it to the output buffer. (This is actually a straight copy.)               *
//*                                                                            *
//* Input  : seg  : segment data                                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void oggEncodeSegmentTable ( Ogg_Segments& seg )
{
   for ( short i = ZERO ; i < seg.segCount ; ++i )
      seg.pageBuff[seg.pbIndex++] = seg.segTable[i] ;
   
}  //* End oggEncodeSegmentTable() *

//*************************
//* oggEncodePacketHeader *
//*************************
//******************************************************************************
//* Convert the formatted Page 2 packet header into a binary stream and write  *
//* it to the output buffer.                                                   *
//*                                                                            *
//* Input  : seg  : segment data                                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void oggEncodePacketHeader ( Ogg_Segments& seg )
{
   //* Write the packet type *
   seg.pageBuff[seg.pbIndex++] = seg.opkt.ptype ;

   //* Write the packet identified string *
   for ( short i = ZERO ; i < (OGG_PKTHDR_BYTES - 1) ; ++i )
      seg.pageBuff[seg.pbIndex++] = seg.opkt.pid[i] ;

}  //* End oggEncodePacketHeader() *

//*************************
//* oggEncodeVendorRecord *
//*************************
//******************************************************************************
//* Convert the formatted Page 2 vendor record into a binary stream and write  *
//* it to the output buffer.                                                   *
//*                                                                            *
//* Input  : seg  : segment data                                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void oggEncodeVendorRecord ( Ogg_Segments& seg )
{
   UCHAR convBuff[16] ;

   //* Convert the record length to a 32-bit little-endian *
   //* binary stream and write it to the output buffer.    *
   seg.vendor.intConv( seg.vendor.clen, convBuff ) ;
   for ( short i = ZERO ; i < 4 ; ++i )
      seg.pageBuff[seg.pbIndex++] = convBuff[i] ;

   //* Write the UTF-8 ID string *
   for ( UCHAR i = ZERO ; i < seg.vendor.clen ; ++i )
      seg.pageBuff[seg.pbIndex++] = seg.vendor.ctxt[i] ;

}  //* End oggEncodeVendorRecord() *

//***************************
//* oggEncodeCommentVectors *
//***************************
//******************************************************************************
//* Write the comment vectors as a binary stream to the output buffer.         *
//* Note that the vectors have been pre-converted to binary-stream data.       *
//*   (see oggPrescanMetadata())                                               *
//* Important Note: We ALSO insert the 'framing bit' into the buffer to close  *
//*                 out the comment-segment sequence.                          *
//*                                                                            *
//* Input  : seg   : segment data                                              *
//*          eiPtr : pointer to top of EmbeddedImage linked list               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Notes on Image Vectors:                                                    *
//* -----------------------                                                    *
//* Comment Vectors are passed to us in the seg.newComm[] array.               *
//* This array contains the pre-encoded Comment Vectors, but also zero or more *
//* placholder vectors may be included indicating that an Image Vector should  *
//* be encoded and written to the output buffer. These placeholders are used   *
//* to size the output buffer and to signal the position of the Image Vector   *
//* in the output stream. See oggPrescanMetadata() for more information.       *
//*                                                                            *
//* The placeholder is discarded and the image setup data referenced by the    *
//* 'eiPtr' parameter are encoded, followed by the encoded image data itself.  *
//* The technical details of this encoding are handled by the Ogg_Image class. *
//* What we do here is initialize the Ogg_Image data members and then call     *
//* the encoding methods.                                                      *
//*                                                                            *
//******************************************************************************

static void oggEncodeCommentVectors ( Ogg_Segments& seg, 
                                      const EmbeddedImage* eiPtr )
{
   Ogg_Image ogi ;               // image format conversion
   char ibuff[gsMAXBYTES + 1] ;  // input buffer for binary image data
   gString gs ;                  // text manipulation

   //* Encode the number of comment vectors.          *
   //* (borrow the converter from the vendor record ) *
   //* (because there may not be any actual comments) *
   UCHAR convBuff[16] ;
   seg.vendor.intConv( seg.newCVectors, convBuff ) ;
   for ( short i = ZERO ; i < 4 ; ++i )
      seg.pageBuff[seg.pbIndex++] = convBuff[i] ;


   for ( int v = ZERO ; v < seg.newCVectors ; ++v )
   {
      #if OGG_IMAGES != 0
      gs = &seg.newComm[v].ctxt[4] ;
      if ( (gs.find( ofmi[0].fName )) == ZERO )
      {
         //* Be safe, don't use a pointer you haven't verified *
         if ( eiPtr != NULL && eiPtr->inUse )
         {
            //* Transfer image data to OGG-specified format *
            ogi.reset() ;

            //* Total Image Vector length                        *
            //* (This was calculated during the output prescan.) *
            ogi.totBytes = seg.newComm[v].clen ;

            gs = eiPtr->mp3img.txtDesc ;              // description length and text
            ogi.txtBytes = gs.utfbytes() - 1 ;
            gs.copy( ogi.txtDesc, gsMAXBYTES ) ;
            gs = eiPtr->mp3img.mimType ;              // MIME type length and text
            ogi.mimBytes = gs.utfbytes() - 1 ;
            gs.copy( ogi.mimType, gsMAXBYTES ) ;
            ogi.picType = eiPtr->mp3img.picType ;     // picture type code
            ogi.picBytes = eiPtr->mp3img.picSize ;    // size of binary image
            ogi.picEBytes = eiPtr->mp3img.picEncSize ;// size of encoded image
            // iWidth, iHeight, iCDepth and iCIndex == ZERO

            //* Encode the Image Vector header *
#if 0    // TEMP TEMP - OVERWRITE THE INTEGER DATA WITH TEXT.
UINT32 oldIndex = seg.pbIndex, insertIndex = oldIndex + 4 ;
//UINT32 oldIndex = seg.pbIndex, insertIndex = oldIndex + 4 + 22 + 1 ;
#endif   // TEMP TEMP - OVERWRITE THE INTEGER DATA WITH TEXT.
            seg.pbIndex += ogi.encodeHeader( &seg.pageBuff[seg.pbIndex] ) ;
#if 0    // TEMP TEMP - OVERWRITE THE INTEGER DATA WITH TEXT.
//for ( UINT i = ZERO ; i < 80 ; ++i )
//{
//   gs.compose( "%02hhX ", &seg.pageBuff[oldIndex + i] ) ;
//}
gs = "aaaabbbb=ccccddddeeeeffffgggghhhh" ;
//gs = "aaaabbbbccccddddeeeeffffgggghhhhiiiijjjjkkkkllllmmmmnnnnoooopppp" ;
gs.copy( (char*)&seg.pageBuff[insertIndex], gsMAXBYTES ) ;
#endif   // TEMP TEMP - OVERWRITE THE INTEGER DATA WITH TEXT.

            //* Encode the binary image in "Base64" format.   *
            //* (If source was URL, we may have no image data.*
            if ( eiPtr->mp3img.picSize > ZERO )
            {
               ifstream ifs( eiPtr->mp3img.picPath, ifstream::in ) ;
               if ( ifs.is_open() )
               {
                  int chunks = ogi.picBytes / gsMAXBYTES,
                      rbytes ;
                  while ( chunks-- > ZERO )  // process whole multiples of gsMAXBYTES
                  {
                     ifs.read( ibuff, gsMAXBYTES ) ;
                     rbytes = ifs.gcount() ;
                     seg.pbIndex += ogi.encodeImage( &seg.pageBuff[seg.pbIndex],
                                                     (UCHAR*)ibuff, rbytes ) ;
                  }
                  ifs.read( ibuff, gsMAXBYTES ) ;  // process the remaining data
                  rbytes = ifs.gcount() ;
                  seg.pbIndex += ogi.encodeImage( &seg.pageBuff[seg.pbIndex],
                                                  (UCHAR*)ibuff, rbytes ) ;
                  ifs.close() ;        // close the source file
               }
               else
               {
                  /* This should never happen, but.... */
               }
            }

            eiPtr = eiPtr->next ;      // reference the next image in the list
         }
         else
         {
            /* This should never happen, but.... */
         }
      }
      else
      #endif   // OGG_IMAGES
      {
         for ( UINT32 i = ZERO ; i < seg.newComm[v].clen ; ++i )
            seg.pageBuff[seg.pbIndex++] = seg.newComm[v].ctxt[i] ;
      }
   }
   seg.pageBuff[seg.pbIndex++] = FRAMING_BIT ;

}  //* End oggEncodeCommentVectors() *

//***************************
//*  oggEncodeSetupHeader   *
//***************************
//******************************************************************************
//* Copy the Setup Header from the input stream to the output buffer.          *
//* The Setup Header is all the Page 2 segments that follow the Comment        *
//* Vectors.                                                                   *
//*                                                                            *
//* Input  : ifs  : handle to the open input stream                            *
//*          seg  : segment data                                               *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************
//* Programmer's Note: The loop-control and segment-size values have a size    *
//* mismatch, so direct comparisons will fail. For this reason, we define      *
//* 'ib' to make the comparisons and indices work properly.                    *
//******************************************************************************

static void oggEncodeSetupHeader ( ifstream& ifs, Ogg_Segments& seg )
{
   char inbuff[gsMAXBYTES + 4] ;
   int ib ;

   for ( int s = seg.newCSegs ; s < seg.segCount ; ++s )
   {
      ib = (UCHAR)seg.segTable[s] ;
      ifs.read ( inbuff, ib ) ;
      for ( int i = ZERO ; i < ib ; ++i )
         seg.pageBuff[seg.pbIndex++] = inbuff[i] ;
   }

}  //* End oggEncodeSetupHeader() *

//*************************
//*    oggCalculateCRC    *
//*************************
//******************************************************************************
//* Scan the specified block of data and create a CRC.                         *
//* The data block contains the Page 2 data                                    *
//*                                                                            *
//* Input  : blkPtr : pointer to the input byte array                          *
//*          blkLen : number of bytes in the array                             *
//*                                                                            *
//* Returns: the calculated CRC                                                *
//******************************************************************************
//* Notes:                                                                     *
//* ======                                                                     *
//* CRC error detection is a project in itself. We have implemented CRC as a   *
//* fully generalized algorithm in the CRC_Gen class. This class is actually   *
//* much more flexible that we need here, but we expect to need CRC for other  *
//* projects.                                                                  *
//*                                                                            *
//* The so-called polynomial value i.e. the divisor for CRC arithmetic was     *
//* obtained from theOGG/Vorbis specification:                                 *
//*               https://xiph.org/vorbis/doc/framing.html                     *
//* The magic number is 0x04c11db7 (defined as crcDFLT_POLY in CRC_Gen),       *
//* without which it would not be possible to derive the correct algorithm.    *
//* This same "poly" has a number of other applications including data         *
//* compression and Ethernet packets.                                          *
//*                                                                            *
//* From the specification:   <https://xiph.org/vorbis/doc/framing.html>       *
//*  "32 bit CRC value (direct algorithm, initial val and final XOR = 0,       *
//*   generator polynomial=0x04c11db7). The value is computed over the         *
//*   entire header (with the CRC field in the header set to zero) and         *
//*   then continued over the page. The CRC field is then filled with          *
//*   the computed value."                                                     *
//*           Header   CRC                                                     *
//*            Byte   Value                                                    *
//*             22    0xXX LSB                                                 *
//*             23    0xXX                                                     *
//*             24    0xXX                                                     *
//*             25    0xXX MSB                                                 *
//*                                                                            *
//* The CRC generator setup parameters for OGG/Vorbis CRC are:                 *
//*                                                                            *
//*          regwidth   = 32 ;          // 32-bit register
//*          poly       = 0x04C11DB7 ;  // (from OGG/Vorbis documentation)
//*          reginit    = 0 ;
//*          xorfinal   = 0 ;
//*          reg        = 0 ;           // clear register and setup values
//*          reflectin  = false ;       // no reflection
//*          reflectout = false ;
//******************************************************************************

static UINT32 oggCalculateCRC ( Ogg_Segments& seg )
{
   //* Offset into table for inserting the *
   //* new CRC value into the page header, *
   const int CRC_OFFSET = 22 ;
   //* Setup parameters for instantiating the CRC_Gen class (see above).*
   Crc_Parms crcp = { 32, 0x04C11DB7, 0, 0, 0, false, false } ;
   CRC_Gen crcGen( crcp ) ;      // CRC generator
   UINT32 crcVal = ZERO ;        // new CRC, return value

   //* Reset the CRC target field to zeros *
   seg.vendor.intConv( crcVal, &seg.pageBuff[CRC_OFFSET] ) ;

   #if 0    // FOR DEBUGGING ONLY
   //* For external verification of the CRC value, *
   //* export the contents of pageBuff to a file.  *
   //* Verify using:                               *
   //*    crcplus --file='page.bin' --oggvorbis    *
   ofstream ofs( "./1_TestData/page.bin", ofstream::out | ofstream::trunc ) ;
   if ( ofs.is_open() )
   {
      for ( int i = ZERO ; i < seg.pbIndex ; ++i )
         ofs << seg.pageBuff[i] ;

      ofs.close() ;
   }
   #endif   // FOR DEBUGGING ONLY

   crcVal = crcGen.generateCRC ( seg.pageBuff, seg.pbIndex, true ) ;

   //* Encode the new CRC value and insert it into target field *
   seg.vendor.intConv( crcVal, &seg.pageBuff[CRC_OFFSET] ) ;

   return crcVal ;

}  //* End oggCalculateCRC() *

#if OGG_IMAGES != 0
//*************************
//*  oggImageVectorSize   *
//*************************
//******************************************************************************
//* For the specified image, calculate the number of bytes that will be        *
//* written to the target file. This includes:                                 *
//*  i) one(1) integer value for size of Image Vector [NOT INCLUDED IN TOTAL]  *
//*  1) comment name and equals character ('METADATA_BLOCK_PICTURE=')          *
//*  2) eight(8) integer values for setup data                                 *
//*     a) picture type code            e) image height                        *
//*     b) MIME text length             f) image color depth                   *
//*     c) description text length      g) image color indexing                *
//*     d) image width                  h) bytes of encoded image data         *
//*  3) MIME-type text                                                         *
//*  4) description text                                                       *
//*  5) encoded binary data                                                    *
//*                                                                            *
//*                                                                            *
//* Input  : eiPtr  : pointer to image storage object                          *
//*                                                                            *
//* Returns: total number of bytes needed for the Image Vector                 *
//******************************************************************************
static UINT32 oggImageVectorSize ( const EmbeddedImage* eiPtr )
{
   //* Calculate combined length of:*
   //* a) comment name              *
   //* b) equals character          *
   //* c) description text          *
   //* d) MIME type text            *
   gString gs( "%S=%s%s", ofmi[0].fName, eiPtr->mp3img.txtDesc, eiPtr->mp3img.mimType ) ;
   UINT32 textBytes = gs.utfbytes() - 1,
          ivBytes =   textBytes                  // text bytes
                    + sizeof(UINT32) * 8         // eight(8) integer values
                    + eiPtr->mp3img.picEncSize ; // Base64-encoded image size

   return ivBytes ;

}  //* End oggImageVectorSize() *
#endif   // OGG_IMAGES

//**************************
//*    oggWriteMetadata    *
//**************************
//******************************************************************************
//* Write the formatted page buffer to the target file.                        *
//*                                                                            *
//* Input  : seg  : segment data                                               *
//*                                                                            *
//* Returns: 'true' if write was successful, 'false' if output error           *
//******************************************************************************

static bool oggWriteMetadata ( ofstream& ofs, Ogg_Segments& seg )
{
   bool status = false ;
   
   for ( int i = ZERO ; i < seg.pbIndex ; ++i )
      ofs << seg.pageBuff[i] ;

   status = (ofs.rdstate() == std::ios_base::goodbit) ;
   ofs.flush() ;

   return status ;

}  //* End oggWriteMetadata() *

//  ======================================================
#if DEBUG_WMDATA != 0   // DEBUG ONLY: WriteMetadata_OGG()
//  ======================================================

//**************************
//* dbgDisplaySegmentTable *
//**************************
//******************************************************************************
//* Debug Method:                                                              *
//* -------------                                                              *
//* Write the segment table to the debugging file.                             *
//* Warn if Page 2 data stream extends into Page 3 (unlikely).                 *
//*                                                                            *
//* Input  : dbg     : handle to open output file                              *
//*          seg     : segment data                                            *
//*                    segTable: segment table, 1-256 bytes containing the     *
//*                    size of the defined segments                            *
//*                    segCount: number of elements in the segment table       *
//*                                                                            *
//* Returns: nothing                                                           *
//******************************************************************************

static void dbgDisplaySegmentTable ( ofstream& dbg, const Ogg_Segments& seg )
{
   if ( dbg.is_open() )
   {
      gString gs( "Segment Table\n"
                  "-------- -------- -------- -------- "
                  "-------- -------- -------- -------- " ) ;
      dbg << gs.ustr() << endl ;

      for ( int i = ZERO ; i < seg.segCount ; ++i )
      {
         gs.compose( "%03hhu[%3hhu] ", &i, &seg.segTable[i] ) ;
         dbg << gs.ustr() ;
         if ( !((i + 1) % 8) || (i == (seg.segCount - 1)) )
            dbg << "\n" ;
      }

      //* Test for page overrun i.e. data continues on next page. *
      if ( seg.segTable[seg.segCount - 1] == 255 )
         dbg << "Warning! Page 2 data stream extends into Page 3.\n" ;

      dbg << endl ;
   }

}  //* End dbgDisplaySegmentTable() *
//  ======================================================
#endif   // DEBUG_WMDATA
//  ======================================================

#undef DEBUG_WMDATA     // (definition belongs to this module only)
#undef DEBUG_EMDATA     // (definition belongs to this module only)

//*************************
//*                       *
//*************************
//******************************************************************************
//*                                                                            *
//*                                                                            *
//*                                                                            *
//* Input  :                                                                   *
//*                                                                            *
//* Returns:                                                                   *
//******************************************************************************

